#' Calculate the Surface Temperature (Ts)
#'
#' This function calculates the surface temperature (Ts) from three rasters: albedo, NDVI,
#' and air temperature (Ta). The resulting Ts raster will have the same unit as Ta.
#'
#' @param albedo RasterLayer/SpatRaster of albedo values
#' @param ndvi RasterLayer/SpatRaster of NDVI values
#' @param Ta RasterLayer/SpatRaster of air temperature (Ta) in Kelvin
#' @param output_path File path to save the Ts raster (optional)
#' @param verbose Logical; if TRUE, prints information about dry/wet edges, T_min/T_max, and unit info. Default is TRUE.
#' @return SpatRaster of surface temperature (Ts) in the same unit as Ta
#' @importFrom terra rast same.crs project crs res resample ext origin clamp vect extract writeRaster plot crop
#' @importFrom stats lm coef
#'
#' @examples
#' library(terra)
#'
#' albedo_raster <- rast(system.file("extdata", "albedo_mini.tif", package = "Ts"))
#' Ta_raster     <- rast(system.file("extdata", "Ta_mini.tif", package = "Ts"))
#' ndvi_raster   <- rast(system.file("extdata", "ndvi_mini.tif", package = "Ts"))
#'
#' output_path <- tempfile(fileext = ".tif")
#'
#' calculate_Ts(
#'   albedo = albedo_raster,
#'   Ta     = Ta_raster,
#'   ndvi   = ndvi_raster,
#'   output_path = output_path,
#'   verbose = TRUE
#' )
#'
#' print(output_path)
#'
#' @export
calculate_Ts <- function(albedo, ndvi, Ta, output_path = NULL, verbose = FALSE) {

  if(verbose) {
    cat("Note: Ta is assumed to be in Kelvin. The resulting Ts will be in the same unit as Ta.\n")
  }

  # Ensure SpatRaster
  if (is.character(albedo)) albedo <- terra::rast(albedo)
  if (is.character(ndvi))   ndvi   <- terra::rast(ndvi)
  if (is.character(Ta))     Ta     <- terra::rast(Ta)

  # check rasters contain values
  if (all(is.na(as.vector(terra::values(albedo))))) {
    stop("Albedo raster has no values")
  }
  if (all(is.na(as.vector(terra::values(ndvi))))) {
    stop("NDVI raster has no values")
  }
  if (all(is.na(as.vector(terra::values(Ta))))) {
    stop("Ta raster has no values")
  }

  # Align CRS
  if (!terra::same.crs(Ta, albedo)) Ta <- terra::project(Ta, terra::crs(albedo))
  if (!terra::same.crs(ndvi, albedo)) ndvi <- terra::project(ndvi, terra::crs(albedo))

  # Align extent
  common_extent <- terra::intersect(terra::ext(albedo), terra::ext(ndvi))
  common_extent <- terra::intersect(common_extent, terra::ext(Ta))
  albedo <- terra::crop(albedo, common_extent)
  ndvi   <- terra::crop(ndvi, common_extent)
  Ta     <- terra::crop(Ta, common_extent)

  # Align resolution
  if (!all(terra::res(ndvi) == terra::res(albedo))) ndvi <- terra::resample(ndvi, albedo, method = "bilinear")
  if (!all(terra::res(Ta) == terra::res(albedo))) Ta <- terra::resample(Ta, albedo, method = "bilinear")

  # Extract values
  df <- as.data.frame(c(ndvi, albedo), xy = TRUE, na.rm = TRUE)
  names(df) <- c("x", "y", "ndvi", "albedo")

  # Select driest/wettest pixels (5% or at least 10)
  n_total <- nrow(df)
  if(n_total < 10) stop("Not enough valid pixels in the rasters for calculation")

  # Choose 2% or minimum 5 pixels
  n_pixels <- max(ceiling(0.02 * n_total), 5)

  df_sorted <- df[order(df$ndvi), ]
  bottom <- df_sorted[1:n_pixels, ]
  top    <- df_sorted[(n_total - n_pixels + 1):n_total, ]

  # Fit linear models
  bottom_valid <- bottom[!is.na(bottom$ndvi) & !is.na(bottom$albedo), ]
  top_valid    <- top[!is.na(top$ndvi) & !is.na(top$albedo), ]

  if(nrow(bottom_valid) < 2 || nrow(top_valid) < 2) {
    stop("Not enough non-NA pixels in dry/wet zones to fit the linear model. Consider using larger rasters.")
  }

  dry_fit <- stats::lm(ndvi ~ albedo, data = bottom_valid)
  wet_fit <- stats::lm(ndvi ~ albedo, data = top_valid)

  a_dry <- stats::coef(dry_fit)[2]; b_dry <- stats::coef(dry_fit)[1]
  a_wet <- stats::coef(wet_fit)[2]; b_wet <- stats::coef(wet_fit)[1]

  # Relative Ts
  ndvi_dry <- a_dry * albedo + b_dry
  ndvi_wet <- a_wet * albedo + b_wet
  Ts_relative <- (ndvi_dry - ndvi) / (ndvi_dry - ndvi_wet)
  Ts_relative <- terra::clamp(Ts_relative, 0, 1)

  # Determine T_min and T_max using top/bottom coordinates
  wet_coords <- top[, c("x","y")]
  dry_coords <- bottom[, c("x","y")]

  wet_points <- terra::vect(wet_coords, geom = c("x", "y"), crs = terra::crs(ndvi))
  dry_points <- terra::vect(dry_coords, geom = c("x", "y"), crs = terra::crs(ndvi))

  Ta_wet_values <- terra::extract(Ta, wet_points)[,2]
  Ta_dry_values <- terra::extract(Ta, dry_points)[,2]

  T_min <- mean(Ta_wet_values, na.rm = TRUE)
  T_max <- mean(Ta_dry_values, na.rm = TRUE)

  if(verbose) {
    cat("Dry edge  : NDVI = ", round(a_dry, 3), "* albedo +", round(b_dry, 3), "\n")
    cat("Wet edge  : NDVI = ", round(a_wet, 3), "* albedo +", round(b_wet, 3), "\n")
    cat("T_min (wet zone):", round(T_min, 2), "\n")
    cat("T_max (dry zone):", round(T_max, 2), "\n")
  }

  # Compute Ts
  Ts <- Ts_relative * (T_max - T_min) + T_min

  # Save
  if (!is.null(output_path)) {
    terra::writeRaster(Ts, output_path, overwrite = TRUE)
    terra::plot(Ts, main = "Surface Temperature (Ts)")
  }

  return(Ts)
}
