#' Calculate Phylogenetic Tree Distances
#'
#' This function calculates the pairwise distances between sequences in a phylogenetic tree 
#' and returns a numeric vector of 5th smallest distances for each leaf in the tree.
#'
#' @importFrom stats as.dist sd
#' @importFrom utils write.table
#' @param input_obj A `AAStringSet` object containing sequences.
#' @return A numeric vector containing the 5th smallest distances (`Distances_k`) for each leaf in the phylogenetic tree.
#' @details The function uses the `ape` package to construct a phylogenetic tree and calculate pairwise distances 
#' between sequences. The results are returned as a numeric vector.
#' @examples
#' # Input file
#' input_file <- system.file("extdata", "input.fasta", package = "VIProDesign")
#' seqs <- Biostrings::readAAStringSet(input_file)
#' distances_k <- phyl_tree_distance_k(seqs)
#' @export

phyl_tree_distance_k <- function(input_obj) {
  # Validate input sequences
  if (length(input_obj) == 0) {
    stop("No sequences were found in the input object. Please check the input.")
  }
  seqs <- input_obj
  # Validate that all sequences contain only valid amino acid characters (including gaps)
  valid_amino_acids <- "^[ACDEFGHIKLMNPQRSTVWY-]+$"
  invalid_sequences <- sapply(seqs, function(seq) !grepl(valid_amino_acids, as.character(seq)))

  if (any(invalid_sequences)) {
    stop("The input file contains invalid sequences. Please ensure all sequences are valid amino acid sequences (including gaps).")
  }

  # Check if there is a "-" in at least one of the sequences
  contains_hyphen <- any(sapply(seqs, function(seq) grepl("-", as.character(seq))))
  # Print the result
  if (!contains_hyphen) {
     alignment <- DECIPHER::AlignSeqs(seqs)
  }else
  {alignment <-seqs}

  if (length(alignment) < 2) {
  stop("The alignment must contain at least 2 sequences to calculate a distance matrix.")
  }
  # Check that all sequences in the alignment are the same length
  sequence_lengths <- Biostrings::width(alignment)
  if (length(unique(sequence_lengths)) > 1) {
    stop("Not all sequences in the alignment are the same length.")
  }

  D <-DECIPHER::DistanceMatrix(alignment)
  class(alignment) <- "alignment"
  # Construct a neighbor-joining tree
  tree <- ape::nj(as.dist(D))
  # Check if the tree was constructed successfully
  if (is.null(tree)) {
    stop("Failed to construct the phylogenetic tree. Please check the input sequences.")
  }

  # Extract the number of nodes and distances
  numNodes <- length(tree$edge)
  distances <- tree$edge.length
  numLeaves <- length(tree$tip.label)

  # Extract the lengths of only leaf branches
  leafBranchLengths <- distances[1:numLeaves]

  # Compute average and standard deviation of leaf branch lengths
  averageValue <- mean(leafBranchLengths)
  stdValue <- sd(leafBranchLengths)

  # Compute Distances_k for each leaf
  DD <- as.matrix(D)
  Distances_k <- numeric(numLeaves)

  for (i in 1:numLeaves) {
    temp <- DD[i, ]
    sorted <- sort(temp)
    Distances_k[i] <- sorted[5]  # 5th smallest distance
  }

  # Return Distances_k as a numeric vector
  return(Distances_k)
}



