% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mCUSUM.R
\name{mCUSUM}
\alias{mCUSUM}
\title{Modified Cumulative Sum}
\usage{
mCUSUM(data, column, k = 1, h = 2, move_t)
}
\arguments{
\item{data}{A data frame containing the warning indicator columns, arranged in time-based order.}

\item{column}{A column name or column number, used to specify the warning indicator.}

\item{k}{The standard deviation coefficient \eqn{k}.}

\item{h}{The threshold coefficient \eqn{h}.}

\item{move_t}{The moving period \eqn{t_{move}}.}
}
\value{
A data frame containing C1', C2' and C3' warning results. The value of the warning column is 1 for warning and 0 for no warning.
}
\description{
Modified CUSUM method for outbreak detection in infectious disease surveillance data.
Implements three variants (C1', C2', C3') with dynamic thresholds for time series analysis.
}
\details{
Let \eqn{\mathbf{X} = (X_1,\ldots,X_T)^\top} be an observed time series of disease case counts,
where \eqn{X_t} represents the aggregated counts at time \eqn{t} (e.g., daily, weekly, or monthly observations).
We assume \eqn{X_t \sim N(\mu, \sigma^2)} for the underlying distribution.

The modified CUSUM models accumulate excess cases beyond control limits:

\deqn{C1'_0 = C2'_0 = 0}
\deqn{C1'_t = \max\left(0, X_t - (\hat{\mu}_t + k\hat{\sigma}_t) + C1'_{t-1}\right)}
\deqn{C2'_t = \max\left(0, X_t - (\hat{\mu}_t + k\hat{\sigma}_t) + C2'_{t-1}\right)}
\deqn{C3'_t = C2'_t + C2'_{t-1} + C2'_{t-2}}
\deqn{H_t = h\hat{\sigma}_t}

where:
\itemize{
  \item \eqn{k}: Standard deviation coefficient (typical range 0.5–1.5), adjusts sensitivity to deviations
  \item \eqn{h}: Threshold coefficient (typical range 2–5), controls alarm stringency
  \item \eqn{H}: Threshold
  }

Model specifications:
\itemize{
  \item \strong{C1'}: Baseline \eqn{\hat{\mu}_t, \hat{\sigma}_t} estimated from \eqn{(X_{t-t_{move}},...,X_{t-1})}
  \item \strong{C2'}: Baseline \eqn{\hat{\mu}_t, \hat{\sigma}_t} estimated from \eqn{(X_{t-2-t_{move}},...,X_{t-3})} to avoid recent outbreaks
  \item \strong{C3'}: 3-day cumulative sum of C2' values
  \item Alarms trigger when \eqn{Cx'_t > H_t} for each model (x = 1,2,3)
}
}
\examples{
## simulate reported cases
set.seed(123)
cases <- c(round(rnorm(10, 10, 1)), seq(12,21,3), seq(15,5,-5))
dates <- seq(as.Date("2025-01-01"), by = "7 days", length.out = length(cases))
data_frame <- data.frame(date = dates, case = cases)

## modeling
output <- mCUSUM(data_frame, 'case', k = 1, h = 2.5, move_t = 4)
output

## visualize alerts
### C1'
plot(output$date, output$case, type = "l")
points(output$date[output$C1_prime_warning == 1],
       output$case[output$C1_prime_warning == 1], col = "red")

### C2'
plot(output$date, output$case, type = "l")
points(output$date[output$C2_prime_warning == 1],
       output$case[output$C2_prime_warning == 1], col = "red")

### C3'
plot(output$date, output$case, type = "l")
points(output$date[output$C3_prime_warning == 1],
       output$case[output$C3_prime_warning == 1], col = "red")

}
\references{
Wang X, Zeng D, Seale H, et al. Comparing early outbreak detection algorithms based on their optimized parameter values. J Biomed Inform, 2010,43(1):97-103.
}
