% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/telemetry_ingest.R
\name{bid_suggest_analytics}
\alias{bid_suggest_analytics}
\title{Suggest alternative analytics solutions for static dashboards}
\usage{
bid_suggest_analytics(
  dashboard_type = c("static", "ojs", "python"),
  privacy_preference = c("gdpr_compliant", "privacy_focused", "standard"),
  budget = c("flexible", "free", "low"),
  self_hosted = FALSE
)
}
\arguments{
\item{dashboard_type}{Character string specifying the type of dashboard:
\describe{
\item{static}{Static HTML Quarto dashboard (default)}
\item{ojs}{Quarto dashboard using Observable JS}
\item{python}{Static dashboard with Python/Jupyter}
}}

\item{privacy_preference}{Character string indicating privacy requirements:
\describe{
\item{gdpr_compliant}{Prioritize GDPR-compliant solutions (default)}
\item{privacy_focused}{Emphasize user privacy and no tracking}
\item{standard}{Standard analytics with typical tracking}
}}

\item{budget}{Character string indicating budget constraints:
\describe{
\item{free}{Only free/open-source solutions}
\item{low}{Low-cost solutions (< $10/month)}
\item{flexible}{Any cost tier (default)}
}}

\item{self_hosted}{Logical indicating whether self-hosted solutions are
preferred (default: FALSE)}
}
\value{
A data frame with recommended analytics solutions, including:
\item{solution}{Name of the analytics platform}
\item{type}{Type of solution (privacy-focused, traditional, open-source)}
\item{cost}{Cost tier (free, paid, freemium)}
\item{self_hosted}{Whether self-hosting is available}
\item{gdpr_compliant}{Whether the solution is GDPR compliant}
\item{integration_method}{How to integrate (script tag, API, etc.)}
\item{key_features}{Main features for UX analysis}
\item{bidux_compatibility}{How well it works with BID framework}
\item{docs_url}{Link to integration documentation}
}
\description{
Provides recommendations for analytics and telemetry solutions suitable for
static Quarto dashboards, where Shiny-based telemetry (shiny.telemetry or
OpenTelemetry) is not available. This function helps you choose the right
analytics tool based on your needs and constraints.

\strong{Important}: shiny.telemetry and Shiny OpenTelemetry only work with
\code{server: shiny} in Quarto YAML. For static Quarto dashboards (including
OJS-based dashboards), you need alternative web analytics solutions.
}
\section{Integration Patterns}{


\strong{For Static Quarto Dashboards:}
\enumerate{
\item \strong{Event Tracking} - Track user interactions with custom events:
\itemize{
\item Button clicks, filter changes, tab switches
\item Use JavaScript event listeners in Quarto
\item Send events to analytics platform via API
}
\item \strong{Session Analysis} - Monitor user sessions:
\itemize{
\item Page views, time on page, bounce rate
\item User flow through dashboard sections
\item Identify drop-off points
}
\item \strong{Custom Dimensions} - Track dashboard-specific metrics:
\itemize{
\item Selected filters, date ranges, visualization types
\item User cohorts, roles, or departments
\item Dashboard version or configuration
}
}

\strong{Example Integration (Plausible Analytics):}

Add to your Quarto dashboard header:

\if{html}{\out{<div class="sourceCode html">}}\preformatted{<script defer data-domain="yourdomain.com"
  src="https://plausible.io/js/script.tagged-events.js"></script>
}\if{html}{\out{</div>}}

Track custom events in your dashboard JavaScript:

\if{html}{\out{<div class="sourceCode javascript">}}\preformatted{// Track filter change
document.getElementById('regionFilter').addEventListener('change', function(e) \{
  plausible('Filter Changed', \{props: \{filter: 'region', value: e.target.value\}\});
\});

// Track visualization interaction
plotElement.on('plotly_click', function(data) \{
  plausible('Chart Interaction', \{props: \{chart: 'sales_plot', action: 'click'\}\});
\});
}\if{html}{\out{</div>}}

\strong{Analyzing Results with BID Framework:}

While these analytics tools won't automatically integrate with \code{bid_ingest_telemetry()},
you can still apply BID framework principles:
\enumerate{
\item \strong{Notice} - Export analytics data, identify friction points manually
\item \strong{Interpret} - Use \code{bid_interpret()} with insights from analytics
\item \strong{Anticipate} - Apply \code{bid_anticipate()} to plan improvements
\item \strong{Structure} - Design improvements with \code{bid_structure()}
\item \strong{Validate} - Measure impact with before/after analytics comparison
}
}

\examples{
# Get recommendations for static Quarto dashboard with GDPR compliance
suggestions <- bid_suggest_analytics(
  dashboard_type = "static",
  privacy_preference = "gdpr_compliant"
)
print(suggestions)

# Find free, privacy-focused solutions for OJS dashboard
privacy_options <- bid_suggest_analytics(
  dashboard_type = "ojs",
  privacy_preference = "privacy_focused",
  budget = "free"
)

# Get self-hosted options
self_hosted <- bid_suggest_analytics(
  dashboard_type = "static",
  self_hosted = TRUE
)

# View top recommendation
top_choice <- suggestions[1, ]
cat(sprintf("Recommended: \%s\n", top_choice$solution))
cat(sprintf("Integration: \%s\n", top_choice$integration_method))
cat(sprintf("Docs: \%s\n", top_choice$docs_url))

}
