% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Inf.D3.R
\name{Inf.D3}
\alias{Inf.D3}
\title{Find the optimal testing configuration for informative 
three-stage hierarchical testing}
\usage{
Inf.D3(p, Se, Sp, group.sz, obj.fn, weights = NULL, alpha = 2)
}
\arguments{
\item{p}{the probability of disease, which can be specified as an overall 
probability of disease, from which a heterogeneous vector of individual 
probabilities will be generated, or a heterogeneous vector of individual 
probabilities specified by the user.}

\item{Se}{the sensitivity of the diagnostic test.}

\item{Sp}{the specificity of the diagnostic test.}

\item{group.sz}{a single group size over which to find the OTC
out of all possible testing configurations, or a range of group sizes
over which to find the OTC.}

\item{obj.fn}{a list of objective functions which are minimized to find the
OTC. The expected number of tests per individual, "\kbd{ET}", will always 
be calculated. Additional options include "\kbd{MAR}" 
(the expected number of tests divided by the expected number of correct 
classifications, described in Malinovsky et al. (2016)), and "\kbd{GR}" 
(a linear combination of the expected number of tests, the number of 
misclassified negatives, and the number of misclassified positives, 
described in Graff & Roeloffs (1972)). See Hitt et al. (2018) at
\url{http://chrisbilder.com/grouptesting/} for additional details.}

\item{weights}{a matrix of up to six sets of weights for the GR function. 
Each set of weights is specified by a row of the matrix.}

\item{alpha}{a scale parameter for the beta distribution that specifies 
the degree of heterogeneity for the generated probability vector. If a 
heterogeneous vector of individual probabilities is specified by the user, 
\kbd{alpha} can be specified as \kbd{NA} or will be ignored.}
}
\value{
A list containing:
\item{prob}{the probability of disease, as specified by the user.}
\item{alpha}{the level of heterogeneity used to generate the vector of individual
probabilities.}
\item{Se}{the sensitivity of the diagnostic test.}
\item{Sp}{the specificity of the diagnostic test.}
\item{opt.ET, opt.MAR, opt.GR}{a list for each objective function specified 
by the user, containing:
\describe{
\item{OTC}{a list specifying elements of the optimal testing configuration, 
which include:
\describe{
\item{Stage1}{pool size for the first stage of testing, i.e. the initial
group size.}
\item{Stage2}{pool sizes for the second stage of testing.}}}
\item{p.vec}{the sorted vector of individual probabilities.}
\item{ET}{the expected testing expenditure for the OTC.}
\item{value}{the value of the objective function per individual.}
\item{PSe}{the overall pooling sensitivity for the algorithm.
Further details are given under 'Details'.}
\item{PSp}{the overall pooling specificity for the algorithm.
Further details are given under 'Details'.}
\item{PPPV}{the overall pooling positive predictive value for the algorithm.
Further details are given under 'Details'.}
\item{PNPV}{the overall pooling negative predictive value for the algorithm.
Further details are given under 'Details'.}}}
}
\description{
Find the optimal testing configuration (OTC) for 
informative three-stage hierarchical testing and calculate the 
associated operating charcteristics.
}
\details{
This function finds the OTC and computes the associated 
operating characteristics for informative three-stage hierarchical testing. 
This function finds the optimal testing configuration by considering all 
possible testing configurations. Operating characteristics calculated are
expected number of tests, pooling sensitivity, pooling specificity, pooling
positive predictive value, and pooling negative predictive value for the algorithm.
See Hitt et al. (2018) or Black et al. (2015) at 
\url{http://chrisbilder.com/grouptesting/} for additional details on the 
implementation of informative three-stage hierarchical testing.

The value(s) specified by \kbd{group.sz} represent the initial (stage 1) 
group size. If a single value is provided for \kbd{group.sz}, the OTC 
will be found over all possible testing configurations for that initial 
group size. If a range of group sizes is specified, the OTC will be found 
over all group sizes.

The displayed pooling sensitivity, pooling specificity, pooling positive 
predictive value, and pooling negative predictive value are weighted 
averages of the corresponding individual accuracy measures for all 
individuals within the initial group for a hierarchical algorithm, or 
within the entire array for an array-based algorithm.
Expressions for these averages are provided in the Supplementary 
Material for Hitt et al. (2018). These expressions are based on accuracy 
definitions given by Altman and Bland (1994a, 1994b).
}
\examples{
# Find the OTC for informative three-stage hierarchical 
#   testing over a range of group sizes.
# A vector of individual probabilities is generated using
#   the expected value of order statistics from a beta 
#   distribution with p = 0.05 and a heterogeneity level 
#   of alpha = 0.5. Depending on the specified probability, 
#   alpha level, and overall group size, simulation may 
#   be necessary in order to generate the vector of individual
#   probabilities. This is done using p.vec.func() and 
#   requires the user to set a seed in order to reproduce 
#   results.
# This example takes approximately 20 seconds to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
\dontrun{
set.seed(8318)
Inf.D3(p=0.05, Se=0.99, Sp=0.99, group.sz=3:30, 
obj.fn=c("ET", "MAR"), alpha=0.5)}

# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
set.seed(8318)
Inf.D3(p=0.05, Se=0.99, Sp=0.99, group.sz=10:15, 
obj.fn=c("ET", "MAR"), alpha=0.5)

# Find the OTC out of all possible testing configurations
#   for a specified group size and vector of individual 
#   probabilities.
# This example takes less than 1 second to run.
# Estimated running time was calculated using a 
#   computer with 16 GB of RAM and one core of an 
#   Intel i7-6500U processor.
set.seed(1216)
p.vec <- p.vec.func(p=0.10, alpha=2, grp.sz=12)
Inf.D3(p=p.vec, Se=0.99, Sp=0.99, group.sz=12,
obj.fn=c("ET", "MAR", "GR"), weights=matrix(data=c(1,1), 
nrow=1, ncol=2, byrow=TRUE), alpha=NA)
}
\references{
\insertRef{Altman1994a}{binGroup}

\insertRef{Altman1994b}{binGroup}

\insertRef{Black2015}{binGroup}

\insertRef{Graff1972}{binGroup}

\insertRef{Hitt2018}{binGroup}

\insertRef{Malinovsky2016}{binGroup}
}
\seealso{
\code{\link{NI.D3}} for non-informative three-stage hierarchical testing
and \code{\link{OTC}} for finding the optimal testing configuration for a
number of standard group testing algorithms.

\url{http://chrisbilder.com/grouptesting/}

Other OTC functions: 
\code{\link{Inf.Array}()},
\code{\link{Inf.Dorf}()},
\code{\link{NI.A2M}()},
\code{\link{NI.Array}()},
\code{\link{NI.D3}()},
\code{\link{NI.Dorf}()},
\code{\link{OTC}()}
}
\author{
Brianna D. Hitt
}
\concept{OTC functions}
