% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/standvar.r
\name{standvar}
\alias{standvar}
\title{Function to compute stand-level variables for a given
imputed-tree-list.}
\usage{
standvar(
  data,
  plot.id,
  plot.area,
  time = NA,
  d,
  y,
  h = NA,
  factvar = NA,
  metric = TRUE,
  eng = TRUE,
  ...
)
}
\arguments{
\item{data}{data frame having the tree list of a sample plot.}

\item{plot.id}{a string having the plot code-number or unique
identificator.}

\item{plot.area}{column name having the plot area in m\eqn{^{2}}{^2}.}

\item{time}{a number of year of measurement, if not provided the
current year is assigned by default.}

\item{d}{a text of the column-name having the diameter at
breast-heigth. By default is assumed to be in cm. See option
\code{metrics} to change it to the imperial system.}

\item{y}{a string-vector with the name(s) of the tree-level
variable(s) to which aggregated stand variables are needed
to be computed. For example, volume is such a variable. By
default is set to \code{NA}, thus only stand basal area will be
computed. For instance, if is a vector, \code{y=c("vgross","vnet")},
those variables must be present in the tree list as column names.
Notice that the resulting stand variable accronimg for each
variable will be "vgross.ha" and "vnet.ha", by default.}

\item{h}{a text of the column-name having the total height
of the tree. By default is set to NA. If provided this variable
is assumed to be measured in meters. See option
\code{metrics} to use the imperial system.}

\item{factvar}{a string having de name of the variable
used as factor. Each level of the 'factvar` is a category.}

\item{metric}{is a logic value, the default is to \code{TRUE}, thus
the diameter \code{d} has to be in cm, and the resulting tree-level
basal area will be in m\eqn{^{2}}{^2}. If \code{metric} is \code{FALSE},
the diameter \code{d} has to be in inches, and the computed tree
basal area will be in ft\eqn{^{2}}{^2}.}

\item{eng}{logical; if \code{TRUE} (by default), the language of some
of the output-column names will be English; if "FALSE" will
be Spanish. For instance, the levels of the
factor-variable (\code{factvar}) is named "level"; otherwise will be
"nivel".}

\item{...}{aditional options for basic stats functions.}
}
\value{
Returns a data frame with the the stand variables
per plot. If \code{factvar} is provided, the stand variables
will be a also computed for each level of the \code{factvar}.
Dominant diameter and dominant height are computed using
the function \code{domvar()}.
}
\description{
Computes stand-level variables for a given sample plot. The
variables are the following: density, basal area,
quadratic diameter diameter, average height, top height,
and stand volume.
}
\details{
For a given imputed tree list of a sample plot, several
stand-level variales are computed. Note that the imputed-tree
list must have all the tree-level variables needed to obtain
the stand-level ones, such as, height, and volume. If there
remeasurement for a plot, the computation is by plot and
measurement year.
}
\examples{

df<-biometrics::eucaplot2
#see the metadata by typing ?eucaplot2
head(df)
datana::descstat(df[,c("dap","atot")])
## Preparing the treelist, in the required format
df$parce<-1;df$sup.plot<-500
## Estimating tree-volume using an artifical factor form
df$vol<-gtree(x=df$dap)*df$atot*0.35
## Using the function
standvar(data=df,plot.id="parce",plot.area="sup.plot",
         d="dap",h="atot",y="vol")
# Do the same as before, but adding the computation by a factor
standvar(data=df,plot.id="parce",plot.area="sup.plot",
         d="dap",h="atot",y="vol",factvar = "clase.copa")
## More than one aggregated variable. For instance, adding biomass
##  and tree carbon, aside of volume. A naive estimation
##  of tree-biomass and carbon, could be 
df$biom<-df$v*420 #(kg/m3)
df$carb<-df$biom*0.5 #50\% of biomass is carbon
df
standvar(data=df,plot.id="parce",plot.area="sup.plot",
         d="dap",h="atot",y=c("vol","biom","carb"))
 #what if the sample plot has a remeasurement
df$measu.yr<-2025;df$measu.yr[1:5]<-2020
df
#' ## Using the function per measurement year
standvar(data=df,plot.id="parce",plot.area="sup.plot",
         d="dap",h="atot",y=c("vol","biom","carb"),time="measu.yr")
# Do the same as before, but adding the computation by a factor
standvar(data=df,plot.id="parce",plot.area="sup.plot",
         d="dap",h="atot",y=c("vol","biom","carb"),time="measu.yr",
         factvar = "clase.copa")
# More than one plot
df<-biometrics::radiatapl2
table(df$parce)
## naive imputation of tree-height
df[is.na(df$atot),"atot"]<-df[is.na(df$atot),"dap"]*0.8
## Estimating tree-volume using an artifical factor form
df$vtot<-gtree(x=df$dap)*df$atot*0.35
datana::descstat(df[,c("dap","atot","vtot")])
df$sup.plot<-150
standvar(data=df,plot.id="parce",plot.area="sup.plot",
         d="dap",h="atot",y="vtot")
standvar(data=df,plot.id="parce",plot.area="sup.plot",
             d="dap",h="atot",y="vtot",factvar = "sanidad")
}
\references{
\itemize{
\item Salas-Eljatib C. 2025. Biometría y Modelación Forestal.
Borrador de libro, en revisión. 352 p.
}
}
\author{
Christian Salas-Eljatib.
}
