% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BIOMOD_EnsembleModeling.R
\name{BIOMOD_EnsembleModeling}
\alias{BIOMOD_EnsembleModeling}
\title{Create and evaluate an ensemble set of models and predictions}
\usage{
BIOMOD_EnsembleModeling(
  bm.mod,
  models.chosen = "all",
  em.by = "PA+run",
  em.algo,
  metric.select = "all",
  metric.select.thresh = NULL,
  metric.select.table = NULL,
  metric.select.dataset = NULL,
  metric.eval = c("KAPPA", "TSS", "AUCroc"),
  var.import = 0,
  EMci.alpha = 0.05,
  EMwmean.decay = "proportional",
  nb.cpu = 1,
  seed.val = NULL,
  do.progress = TRUE
)
}
\arguments{
\item{bm.mod}{a \code{\link{BIOMOD.models.out}} object returned by the 
\code{\link{BIOMOD_Modeling}} function}

\item{models.chosen}{a \code{vector} containing model names to be kept, must be either 
\code{all} or a sub-selection of model names that can be obtained with the 
\code{\link{get_built_models}} function applied to \code{bm.mod}}

\item{em.by}{a \code{character} corresponding to the way kept models will be combined to build 
the ensemble models, must be among \code{all}, \code{algo}, \code{PA}, \code{PA+algo}, 
\code{PA+run}}

\item{em.algo}{a \code{vector} corresponding to the ensemble models that will be computed, 
must be among \code{EMmean}, \code{EMmedian}, \code{EMcv}, \code{EMci}, 
\code{EMca}, \code{EMwmean}, \code{EMmode}, \code{EMfreq}}

\item{metric.select}{a \code{vector} containing evaluation metric names to be used to select 
single models based on their evaluation scores, must be among \code{user.defined} or 
\code{AUCroc}, \code{AUCprg}, \code{TSS}, \code{KAPPA}, \code{ACCURACY}, \code{BIAS}, \code{POD}, 
\code{FAR}, \code{POFD}, \code{SR}, \code{CSI}, \code{ETS}, \code{OR}, \code{ORSS}, 
\code{BOYCE}, \code{MPA} (\emph{binary data}), 
\code{RMSE}, \code{MAE}, \code{MSE}, \code{Rsquared}, \code{Rsquared_aj}, \code{Max_error} 
(\emph{abundance / count / relative data}), 
\code{Accuracy}, \code{Recall}, \code{Precision}, \code{F1} (\emph{ordinal data})}

\item{metric.select.thresh}{(\emph{optional, default} \code{NULL}) \cr 
A \code{vector} of \code{numeric} values corresponding to the minimum scores (one for each 
\code{metric.select}) below which single models will be excluded from the ensemble model 
building}

\item{metric.select.table}{(\emph{optional, default} \code{NULL}) \cr
If \code{metric.select = 'user.defined'}, a \code{data.frame} containing evaluation scores 
calculated for each single models and that will be compared to \code{metric.select.thresh} 
values below which single models will be excluded from the ensemble model, with 
\code{metric.select} rownames, and \code{models.chosen} colnames}

\item{metric.select.dataset}{(\emph{optional, default} \code{validation} if possible) \cr
A \code{character} defining which dataset should be used to filter and/or weight the ensemble 
models, must be among \code{calibration}, \code{validation}, \code{evaluation}}

\item{metric.eval}{a \code{vector} containing evaluation metric names to be used, must 
be among \code{AUCroc}, \code{AUCprg}, \code{TSS}, \code{KAPPA}, \code{ACCURACY}, \code{BIAS}, \code{POD}, 
\code{FAR}, \code{POFD}, \code{SR}, \code{CSI}, \code{ETS}, \code{OR}, \code{ORSS}, 
\code{BOYCE}, \code{MPA} (\emph{binary data}), 
\code{RMSE}, \code{MAE}, \code{MSE}, \code{Rsquared}, \code{Rsquared_aj}, \code{Max_error} 
(\emph{abundance / count / relative data}), 
\code{Accuracy}, \code{Recall}, \code{Precision}, \code{F1} (\emph{ordinal data})}

\item{var.import}{(\emph{optional, default} \code{NULL}) \cr 
An \code{integer} corresponding to the number of permutations to be done for each variable to 
estimate variable importance}

\item{EMci.alpha}{(\emph{optional, default} \code{0.05}) \cr 
A \code{numeric} value corresponding to the significance level to estimate confidence interval}

\item{EMwmean.decay}{(\emph{optional, default} \code{proportional}) \cr 
If \code{em.algo = 'EMWmean'}, a \code{numeric} value defining the relative importance of 
weights. A high value will strongly discriminate \emph{good} models from the \code{bad} ones 
(see Details). It is also possible to set it to \code{proportional} and weights will be 
proportional to the single models evaluation scores, or to provide a \code{function}.}

\item{nb.cpu}{(\emph{optional, default} \code{1}) \cr 
An \code{integer} value corresponding to the number of computing resources to be used to 
parallelize the single models predictions and the ensemble models computation}

\item{seed.val}{(\emph{optional, default} \code{NULL}) \cr 
An \code{integer} value corresponding to the new seed value to be set}

\item{do.progress}{(\emph{optional, default} \code{TRUE}) \cr 
A \code{logical} value defining whether the progress bar is to be rendered or not}
}
\value{
A \code{\link{BIOMOD.ensemble.models.out}} object containing models outputs, or links to saved 
outputs. \cr Models outputs are stored out of \R (for memory storage reasons) in 2 different 
folders created in the current working directory :
\enumerate{
  \item a \emph{models} folder, named after the \code{resp.name} argument of 
  \code{\link{BIOMOD_FormatingData}}, and containing all ensemble models
  \item a \emph{hidden} folder, named \code{.BIOMOD_DATA}, and containing outputs related 
  files (original dataset, calibration lines, pseudo-absences selected, predictions, 
  variables importance, evaluation values...), that can be retrieved with 
  \href{https://biomodhub.github.io/biomod2/reference/getters.out.html}{\code{get_[...]}} 
  or \code{\link{load}} functions, and used by other \pkg{biomod2} functions, like 
  \code{\link{BIOMOD_EnsembleForecasting}}
}
}
\description{
This function allows to combine a range of models built with the 
\code{\link{BIOMOD_Modeling}} function in one (or several) ensemble model. Modeling 
uncertainty can be assessed as well as variables importance, ensemble predictions can be 
evaluated against original data, and created ensemble models can be projected over new 
conditions (see Details).
}
\details{
\bold{Concerning models sub-selection (\code{models.chosen}) :} \cr
Applying \code{\link{get_built_models}} function to the \code{bm.mod} object gives the names 
of the single models created with the \code{\link{BIOMOD_Modeling}} function. The 
\code{models.chosen} argument can take either a sub-selection of these single model names, or 
the \code{all} default value, to decide which single models will be used for the ensemble 
model building. \cr \cr

\bold{Concerning models assembly rules (\code{em.by}) :} \cr
Single models built with the \code{\link{BIOMOD_Modeling}} function can be combined in 5 
different ways to obtain ensemble models :
\describe{
  \item{PA+run}{each combination of pseudo-absence and repetition datasets is done, 
  \emph{merging} algorithms together}
  \item{PA+algo}{each combination of pseudo-absence and algorithm datasets is done, 
  \emph{merging} repetitions together}
  \item{PA}{pseudo-absence datasets are considered individually, \emph{merging} algorithms 
  and repetitions together}
  \item{algo}{algorithm datasets are considered individually, \emph{merging} pseudo-absence 
  and repetitions together}
  \item{all}{all single models are combined into one}
}
Hence, depending on the chosen method, the number of ensemble models built will vary. \cr
\emph{If no evaluation data was given to the \code{\link{BIOMOD_FormatingData}} function, 
some ensemble model evaluations may be biased due to difference in data used for single 
model evaluations.} \cr
\bold{Be aware that all of these combinations are allowed, but some may not make sense 
depending mainly on how pseudo-absence datasets have been built and whether all of them 
have been used for all single models or not} (see \code{PA.nb.absences} and \code{models.pa} 
parameters in \code{\link{BIOMOD_FormatingData}} and \code{\link{BIOMOD_Modeling}} functions 
respectively). \cr \cr \cr

\bold{Concerning evaluation metrics :}
\describe{
  \item{metric.select}{metric(s) must be chosen among the ones used within the 
  \code{\link{BIOMOD_Modeling}} function to build the \code{bm.mod} object, unless 
  \code{metric.select = 'user.defined'} and therefore values will be provided through the 
  \code{metric.select.table} parameter. \cr 
  Each selected metric will be used at different steps of the ensemble modeling function to :
  \enumerate{
    \item remove \emph{low quality} single models having a score lower than
    \code{metric.select.thresh}
    \item perform the binary transformation if \code{em.algo = 'EMca'}
    \item weight models if \code{em.algo = 'EMwmean'}
  }
  \emph{Note that metrics are not combined together, and one ensemble model is built for each 
  metric provided.}
  }
  \item{metric.select.table}{if \code{metric.select = 'user.defined'}, this parameter allows 
  to use evaluation metrics other than those calculated within \pkg{biomod2}. It must be a 
  \code{data.frame} containing as many columns as \code{models.chosen} with matching names, 
  and as many rows as evaluation metrics to be used. The number of rows must match the length 
  of \code{metric.select.thresh}, and values will be compared to those defined in 
  \code{metric.select.thresh} to remove \emph{low quality} single models from the ensemble 
  model building.
  }
  \item{metric.select.dataset}{by default, \emph{validation} datasets will be used, unless no 
  validation is available (no cross-validation) in which case \emph{calibration} datasets 
  will be used \cr \cr \cr}
}

\bold{Concerning ensemble algorithms :} \cr
6 modeling techniques are currently available :
\describe{
  \item{EMmedian}{median of probabilities over the selected models \cr 
  
  Less sensitive to outliers than the mean}
  
  \item{EMmean}{mean of probabilities over the selected models}
  
  \item{EMwmean}{weighted mean of probabilities over the selected models \cr
  
  Probabilities are weighted according to their model evaluation scores obtained when 
  building the \code{bm.out} object with the \code{BIOMOD_Modeling} function (\emph{better a 
  model is, more importance it has in the ensemble}) and summed.
  
  The \code{EMwmean.decay} is the ratio between a weight and the next or previous one. \cr
  The formula is : \code{W = W(-1) * EMwmean.decay}. \cr 
  \emph{For example, with the value of \code{1.6} and \code{4} weights wanted, the relative 
  importance of the weights will be \code{1 / 1.6 / 2.56 (=1.6*1.6) / 4.096 (=2.56*1.6)} from 
  the weakest to the strongest, and gives \code{0.11 / 0.17 / 0.275 / 0.445} considering that 
  the sum of the weights is equal to one. The lower the \code{EMwmean.decay}, the smoother 
  the differences between the weights enhancing a weak discrimination between models.}
  
  If \code{EMwmean.decay = 'proportional'}, the weights are assigned to each model 
  proportionally to their evaluation scores. The discrimination is fairer than using the 
  \emph{decay} method where close scores can have strongly diverging weights, while the 
  proportional method would assign them similar weights.

  It is also possible to define the \code{EMwmean.decay} parameter as a function that will be 
  applied to single models scores and transform them into weights. \cr 
  \emph{For example, if \code{EMwmean.decay = function(x) {x^2}}, the squared of evaluation 
  score of each model will be used to weight the models predictions.}
  }
    
  \item{EMca}{committee averaging over the selected models \cr
  
  Probabilities are first transformed into binary data according to the threshold defined 
  when building the \code{bm.out} object with the \code{BIOMOD_Modeling} function 
  (maximizing the evaluation metric score over the \emph{calibration} dataset). The committee 
  averaging score is obtained by taking the average of these binary predictions. \cr
  It is built on the analogy of a simple vote :
  \itemize{
    \item each single model votes for the species being either present (\code{1}) or absent 
    (\code{0})
    \item the sum of \code{1} is then divided by the number of single models \emph{voting}
  }
  The interesting feature of this measure is that it gives both a prediction and a measure of 
  uncertainty. When the prediction is close to \code{0} or \code{1}, it means that all models 
  agree to predict \code{0} or \code{1} respectively. When the prediction is around 
  \code{0.5}, it means that half the models predict \code{1} and the other half \code{0}. \cr
  \emph{Note that this is for binary data only.}
  }
  
  \item{EMci}{confidence interval around the mean of probabilities of the selected models \cr 
  
  It creates 2 \emph{ensemble} models :
  \itemize{
    \item \emph{LOWER} : there is less than \code{100 * EMci.alpha / 2} \% of chance to get 
    probabilities lower than the given ones
    \item \emph{UPPER} : there is less than \code{100 * EMci.alpha / 2} \% of chance to get 
    probabilities upper than the given ones
  }
  These intervals are calculated with the following function :
  \deqn{I_c = [ \bar{x} -  \frac{t_\alpha sd }{ \sqrt{n} }; 
  \bar{x} +  \frac{t_\alpha sd }{ \sqrt{n} }]}
  }
    
  \item{EMcv}{coefficient of variation (\code{sd / mean}) of probabilities over the selected 
  models \cr 
  
  This is the only \emph{ensemble} model that might not be over the same scale than the 
  others, as CV is a measure of uncertainty rather a measure of probability of occurrence. 
  It will be evaluated like all other ensemble models although its interpretation will be 
  obviously different. If the CV gets a high evaluation score, it means that the uncertainty 
  is high where the species is observed (which might not be a good feature of the model). 
  \emph{The lower is the score, the better are the models.} 
  }
  
  \item{EMmode}{mode of the predictions over the selected models \cr
  
  For multiclass and ordinal data, EMmode will return the most frequent class found for each point.
  This is the only \emph{ensemble} model that will return categorical data and not numeric values.
  }  
  
  \item{EMfreq}{mode frequency of the predictions over the selected models \cr
  
  For multiclass and ordinal data, EMfreq will return the frequency of the mode found for each point.
  This is a way of assessing the uncertainty between models: the higher the frequency, the lower the uncertainty.
  }
  
}
}
\examples{

library(terra)
# Load species occurrences (6 species available)
data(DataSpecies)
head(DataSpecies)

# Select the name of the studied species
myRespName <- 'GuloGulo'

# Get corresponding presence/absence data
myResp <- as.numeric(DataSpecies[, myRespName])

# Get corresponding XY coordinates
myRespXY <- DataSpecies[, c('X_WGS84', 'Y_WGS84')]

# Load environmental variables extracted from BIOCLIM (bio_3, bio_4, bio_7, bio_11 & bio_12)
data(bioclim_current)
myExpl <- terra::rast(bioclim_current)

\dontshow{
myExtent <- terra::ext(0,30,45,70)
myExpl <- terra::crop(myExpl, myExtent)
}

## ----------------------------------------------------------------------- #
file.out <- paste0(myRespName, "/", myRespName, ".AllModels.models.out")
if (file.exists(file.out)) {
  myBiomodModelOut <- get(load(file.out))
} else {

  # Format Data with true absences
  myBiomodData <- BIOMOD_FormatingData(resp.name = myRespName,
                                       resp.var = myResp,
                                       resp.xy = myRespXY,
                                       expl.var = myExpl)

  # Model single models
  myBiomodModelOut <- BIOMOD_Modeling(bm.format = myBiomodData,
                                      modeling.id = 'AllModels',
                                      models = c('RF', 'GLM'),
                                      CV.strategy = 'random',
                                      CV.nb.rep = 2,
                                      CV.perc = 0.8,
                                      OPT.strategy = 'bigboss',
                                      metric.eval = c('TSS', 'AUCroc'),
                                      var.import = 3,
                                      seed.val = 42)
}

## ----------------------------------------------------------------------- #
# Model ensemble models
myBiomodEM <- BIOMOD_EnsembleModeling(bm.mod = myBiomodModelOut,
                                      models.chosen = 'all',
                                      em.by = 'all',
                                      em.algo = c('EMmean', 'EMca'),
                                      metric.select = c('TSS'),
                                      metric.select.thresh = c(0.7),
                                      metric.eval = c('TSS', 'AUCroc'),
                                      var.import = 3,
                                      seed.val = 42)
myBiomodEM

# Get evaluation scores & variables importance
get_evaluations(myBiomodEM)
get_variables_importance(myBiomodEM)

# Represent evaluation scores
bm_PlotEvalMean(bm.out = myBiomodEM, dataset = 'calibration')
bm_PlotEvalBoxplot(bm.out = myBiomodEM, group.by = c('algo', 'algo'))

# # Represent variables importance
# bm_PlotVarImpBoxplot(bm.out = myBiomodEM, group.by = c('expl.var', 'algo', 'algo'))
# bm_PlotVarImpBoxplot(bm.out = myBiomodEM, group.by = c('expl.var', 'algo', 'merged.by.PA'))
# bm_PlotVarImpBoxplot(bm.out = myBiomodEM, group.by = c('algo', 'expl.var', 'merged.by.PA'))

# # Represent response curves
# bm_PlotResponseCurves(bm.out = myBiomodEM, 
#                       models.chosen = get_built_models(myBiomodEM),
#                       fixed.var = 'median')
# bm_PlotResponseCurves(bm.out = myBiomodEM, 
#                       models.chosen = get_built_models(myBiomodEM),
#                       fixed.var = 'min')
# bm_PlotResponseCurves(bm.out = myBiomodEM, 
#                       models.chosen = get_built_models(myBiomodEM, algo = 'EMmean'),
#                       fixed.var = 'median',
#                       do.bivariate = TRUE)


}
\seealso{
\code{\link{BIOMOD_FormatingData}}, \code{\link{bm_ModelingOptions}}, 
\code{\link{bm_CrossValidation}}, \code{\link{bm_VariablesImportance}}, 
\code{\link{BIOMOD_Modeling}}, \code{\link{BIOMOD_EnsembleForecasting}},
\code{\link{bm_PlotEvalMean}}, \code{\link{bm_PlotEvalBoxplot}}, 
\code{\link{bm_PlotVarImpBoxplot}}, \code{\link{bm_PlotResponseCurves}}

Other Main functions: 
\code{\link{BIOMOD_EnsembleForecasting}()},
\code{\link{BIOMOD_FormatingData}()},
\code{\link{BIOMOD_LoadModels}()},
\code{\link{BIOMOD_Modeling}()},
\code{\link{BIOMOD_Projection}()},
\code{\link{BIOMOD_RangeSize}()}
}
\author{
Wilfried Thuiller, Damien Georges, Robin Engler
}
\concept{Main functions}
\keyword{ensemble}
\keyword{models}
\keyword{weights}
