% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dbn.R
\name{run_dbn}
\alias{run_dbn}
\title{Fit a Two-Slice Dynamic Bayesian Network (DBN) for I, C, and Regime}
\usage{
run_dbn(DT)
}
\arguments{
\item{DT}{A \code{data.frame} or \code{data.table} containing at least:
\itemize{
\item \code{I_cat}, \code{C_cat}: discretized (e.g., tercile) versions
of \code{I} and \code{C}.
\item \code{Regime}: categorical regime indicator.
}
The function internally renames these to \code{Ic}, \code{Cc}, and
\code{R}, constructs their lag-1 counterparts, and drops rows with
missing lags.}
}
\value{
A list with components:
\itemize{
\item \code{dag}: the learned Bayesian network structure
(\code{bnlearn} \code{"bn"} object).
\item \code{fit}: the fitted DBN (\code{"bn.fit"} object).
\item \code{data}: the processed data frame (\code{Ic}, \code{Cc},
\code{R}, and their lag-1 versions) used to learn/fit the DBN.
}
}
\description{
Constructs and estimates a simple two-slice Dynamic Bayesian Network
(DBN) over discretized versions of \code{I}, \code{C}, and \code{Regime}
using \pkg{bnlearn}. The network includes current and lag-1 nodes for
each variable, with structural constraints enforcing the DBN topology.
}
\details{
The DBN is defined on the nodes
\code{Ic}, \code{Cc}, \code{R}, \code{Ic_l1}, \code{Cc_l1}, \code{R_l1}.
A blacklist is used to forbid arrows from current to lagged nodes, while
a whitelist ensures arrows from lagged to current nodes:
\itemize{
\item Blacklist: \code{Ic → Ic_l1}, \code{Cc → Cc_l1}, \code{R → R_l1}.
  \item Whitelist: \code{Ic_l1 → Ic}, \code{Cc_l1 → Cc}, \code{R_l1 → R}.
}

The structure is learned via hill-climbing (\code{bnlearn::hc()}) with
BDe score (\code{score = "bde"}) and imaginary sample size \code{iss = 10}.
Parameters are then estimated via \code{bnlearn::bn.fit()} using Bayesian
estimation with the same \code{iss}.

If \pkg{Rgraphviz} is available, a graph of the learned DAG is produced
and saved as \code{"dbn_graph.png"} in the directory specified by a
global object \code{dir_figs} (character scalar). The preprocessed data
used to fit the DBN are written to \code{"dbn_data.csv"} in \code{dir_csv},
and the fitted objects are saved as \code{"dbn_fit.rds"} in \code{dir_out}.

The function assumes that \code{dir_csv}, \code{dir_out}, and (optionally)
\code{dir_figs} exist as global character scalars specifying output
directories.
}
\examples{
\donttest{
library(data.table)

# 1. Create dummy data (Fixed: wrapped in factor() for bnlearn)
DT <- data.table(
  I_cat  = factor(sample(c("Low", "Medium", "High"), 100, replace = TRUE)),
  C_cat  = factor(sample(c("Low", "Medium", "High"), 100, replace = TRUE)),
  Regime = factor(sample(c("Growth", "Crisis"), 100, replace = TRUE))
)

# 2. Define global paths using tempdir()
tmp_dir <- tempdir()
dir_csv  <- file.path(tmp_dir, "csv")
dir_out  <- file.path(tmp_dir, "dbn")
dir_figs <- file.path(tmp_dir, "figs")

dir.create(dir_csv,  showWarnings = FALSE, recursive = TRUE)
dir.create(dir_out,  showWarnings = FALSE, recursive = TRUE)
dir.create(dir_figs, showWarnings = FALSE, recursive = TRUE)

# 3. Run the function
dbn_res <- run_dbn(DT)

# Inspect the result
print(dbn_res$dag)
}

}
