#' Generate efficient 3-Level Fractional Factorial Designs Using Beam Search
#'
#' This function constructs efficient 3-level unblocked and blocked fractional factorial designs
#' using an iterative beam-search generator selection algorithm.
#'
#' It evaluates candidate generators using moment-based proxy criteria
#' (\eqn{A_3}, \eqn{A_4}, \eqn{A_5}, \eqn{A_6}) and returns the best-ranked
#' designs along with:
#' \itemize{
#'   \item generator coefficient vectors,
#'   \item canonical ternary design keys,
#'   \item full design matrices,
#'   \item word-length patterns,
#'   \item alias structure (main + 2-factor),
#'   \item automatically determined block structures.
#' }
#'
#' This is the 3-level analogue of \code{dol2()}, supporting generation of
#' 3-level fractional factorial design generation.
#'
#'
#' @param n Integer. Total number of factors (base + generated).
#' @param k Integer. Number of dependent generator columns to add i.e. size of fraction.
#'   The number of base factors is \code{r = n - k}.
#' @param max_results Integer. Maximum number of final best-ranked designs
#'   returned. Default: \code{20}.
#' @param top_k_block Integer. Number of top block generators to consider when
#'   automatically selecting block structures. Default: \code{3}.
#' @param beam_width Integer. Maximum beam width used during the beam search.
#'   Default: \code{3000}.
#' @param verbose Logical. If \code{TRUE}, prints ranked designs, WL patterns,
#'   alias structures, and blocked designs. Default: \code{TRUE}.
#' @param time_limit Numeric. Maximum elapsed time (seconds) allowed for the
#'   beam-search expansion. Default: \code{600}.
#'
#'
#' @return A list (invisible) of the best-ranked 3-level fractional factorial
#' designs.
#'
#' Each list element contains:
#' \itemize{
#'   \item \code{generators}: list of generator coefficient vectors,
#'   \item \code{generators_str}: Defining contrast expressions,
#'   \item \code{design}: final design matrix with renamed factor levels,
#'   \item \code{A}: word-length pattern \eqn{(A_3, A_4, A_5, A_6)}.
#' }
#'
#' When \code{verbose = TRUE}, the function additionally prints:
#' \itemize{
#'   \item canonical generator expressions,
#'   \item design matrix,
#'   \item alias structure (main, 2-factor interactions),
#'   \item automatically generated blocked design with confounding summary.
#' }
#'
#'
#' @details
#' Internally, \code{dol3()} performs:
#' \itemize{
#'   \item ternary (0/1/2) grid generation for base factors,
#'   \item dynamic enumeration of canonical generator coefficient vectors,
#'   \item computation of moment based K statistics and \eqn{A_3-A_6},
#'   \item canonical design key generation to avoid duplication,
#'   \item beam-search pruning with user-specified width,
#'   \item automatic selection of efficient 3-level block generators.
#' }
#'
#' The generated designs are well suited for industrial, agricultural, and
#' scientific investigations that demand high-resolution three-level fractional
#' factorial structures with optional blocking. These designs efficiently
#' accommodate multi-level factors, enable precise estimation of main effects
#' and critical interaction terms, and offer flexibility for managing heterogeneity
#' through block formation. As a result, they provide a robust and resource-efficient
#' framework for complex experimental systems conducted in field or laboratory settings.
#'
#'
#' @examples
#' \donttest{
#' # Generate a 3-level fractional factorial design:
#' res3 <- dol3(n = 5, k = 2, max_results = 3, verbose = TRUE)
#'
#' # View the best-ranked design:
#' res3[[1]]$design
#' res3[[1]]$generators_str
#' res3[[1]]$A   # Word length pattern A3-A6
#' }
#'
#' @export
#' @references
#' Xu, H. (2005).
#' A catalogue of three-level regular fractional factorial designs.
#' *Metrika*, 62, 259-281.

dol3 <- function(n, k, max_results = 20, top_k_block = 3,
                 beam_width = 3000, verbose = TRUE, time_limit = 600) {

modn <- function(x, n) { ((x %% n) + n) %% n }

  make_factor_names <- function(m) {
    letters_vec <- LETTERS
    if (m <= length(letters_vec)) return(letters_vec[1:m])
    res <- c(); i <- 1
    while (length(res) < m) {
      if (i == 1) res <- c(res, letters_vec) else for (a in letters_vec) res <- c(res, paste0(letters_vec[i %% length(letters_vec)], a))
      i <- i + 1
    }
    res[1:m]
  }

  gen_3level_from_base <- function(r) {
    if (r <= 0) stop("r must be >= 1")
    grid <- expand.grid(rep(list(0:2), r))
    names(grid) <- make_factor_names(r)
    grid
  }

  add_dependent_col_from_coef3 <- function(base_design, coef, newname) {
    if (length(coef) != ncol(base_design)) stop("coef length mismatch")
    mat <- as.matrix(base_design)
    vals <- mat %*% matrix(as.integer(coef), ncol = 1)
    vals <- as.integer(modn(vals, 3))
    base_design[[newname]] <- vals
    base_design
  }

  enumerate_coef3_dynamic <- function(p) {
    # canonical representatives under multiplication by 2 (mod 3)
    ns <- 3^p
    res <- list()
    for (i in 0:(ns-1)) {
      vec <- integer(p); tmp <- i
      for (j in seq_len(p)) { vec[j] <- tmp %% 3L; tmp <- tmp %/% 3L }
      if (all(vec == 0L)) next
      vec2 <- as.integer((2L * vec) %% 3L)
      if (paste(vec, collapse = ",") <= paste(vec2, collapse = ",")) res[[length(res)+1]] <- as.integer(vec)
    }
    res
  }

  col_code_ternary <- function(vec) {
    v <- as.integer(vec)
    if (!all(v %in% 0:2)) stop("ternary vector must be 0/1/2")
    s <- 0L
    for (i in seq_along(v)) s <- s + v[i] * (3^(i-1))
    s
  }
  design_canonical_key_ternary <- function(design_df) {
    M <- as.matrix(design_df)
    cols <- apply(M, 2, col_code_ternary)
    paste(sort(cols), collapse = "_")
  }

  pairwise_delta_counts <- function(design) {
    N <- nrow(design); mat <- as.matrix(design)
    res <- matrix(0, nrow = N, ncol = N)
    for (i in 1:N) {
      eqs <- t(t(mat) == mat[i, ])
      res[i, ] <- rowSums(eqs)
    }
    res
  }
  compute_Kt <- function(design, t = 3) {
    N <- nrow(design)
    deltas <- pairwise_delta_counts(design)
    sum(deltas^t) / (N^2)
  }

  compute_A3_A6_from_K_3level <- function(design) {
    # returns A3..A6 computed from moment-like K3..K6 values
    ncols <- ncol(design)
    K3 <- compute_Kt(design, 3)
    K4 <- compute_Kt(design, 4)
    K5 <- compute_Kt(design, 5)
    K6 <- compute_Kt(design, 6)
    A3 <- (27*K3 - ncols*(2 + 6*ncols + ncols^2)) / 12
    A4 <- (27*K4 - 18*(3 + 2*ncols)*K3 + ncols*(6 + 8*ncols + 6*ncols^2 + ncols^3)) / 16
    A5 <- (81*K5 - 135*(2 + ncols)*K4 + 45*(15 + 4*ncols + 2*ncols^2)*K3 -
             ncols*(60 - 110*ncols - 25*ncols^2 - 10*ncols^3 - 2*ncols^4)) / 80
    A6 <- (729*K6 - (3645 + 1458*ncols)*K5 + 1215*(11 + 3*ncols + ncols^2)*K4 -
             135*(165 + 80*ncols + 6*ncols^2 + 4*ncols^3)*K3 +
             ncols*(2148 + 3010*ncols + 1485*ncols^2 + 175*ncols^3 + 30*ncols^4 + 10*ncols^5)) / 1440
    list(A3 = A3, A4 = A4, A5 = A5, A6 = A6)
  }

  coef3_to_string <- function(coef) {
    base_names <- make_factor_names(length(coef))
    terms <- c()
    for (j in seq_along(coef)) {
      v <- as.integer(coef[j])
      if (v == 0) next
      terms <- c(terms, paste0(v, "*", base_names[j]))
    }
    if (length(terms) == 0) return("0")
    paste(terms, collapse = " + ")
  }
  #' @importFrom utils combn
  list_alias_structure_3level <- function(design_df, effects = c("main","2factor")) {
    M <- as.matrix(design_df); p <- ncol(M); fac_names <- colnames(design_df)
    effect_list <- list(); effect_labels <- c()
    if ("main" %in% effects) for (i in 1:p) { effect_list[[length(effect_list)+1]] <- M[,i]; effect_labels <- c(effect_labels, fac_names[i]) }
    if ("2factor" %in% effects && p >= 2) {
      combs <- combn(p, 2, simplify = FALSE)
      for (pair in combs) {
        eff <- (M[, pair[1]] + M[, pair[2]]) %% 3
        effect_list[[length(effect_list)+1]] <- eff
        effect_labels <- c(effect_labels, paste(fac_names[pair], collapse=":"))
      }
    }
    if (length(effect_list) == 0) return(list())
    E <- do.call(cbind, effect_list)
    nc <- ncol(E); alias_groups <- list(); used <- rep(FALSE, nc)
    for (i in seq_len(nc)) {
      if (used[i]) next
      group <- i
      for (j in (i+1):nc) {
        if (j <= nc && all(E[,i] == E[,j])) { group <- c(group, j); used[j] <- TRUE }
      }
      if (length(group) > 1) alias_groups[[length(alias_groups)+1]] <- effect_labels[group]
    }
    alias_groups
  }

  rename_generated_columns <- function(design_df, r) {
    p <- ncol(design_df)
    new_names <- make_factor_names(p)
    if (!is.null(colnames(design_df))) {
      curr <- colnames(design_df)
      gidx <- grepl("^G\\d+$", curr)
      if (any(gidx)) {
        repl <- make_factor_names(p); curr[gidx] <- repl[gidx]; colnames(design_df) <- curr; return(design_df)
      }
    }
    colnames(design_df) <- new_names
    design_df
  }

  # ---------- blocking helpers ----------
  assign_blocks_3level <- function(design_df, lambda) {
    M <- as.matrix(design_df)
    if (ncol(M) != length(lambda)) stop("lambda length must equal number of columns in design")
    block_id <- as.integer((M %*% matrix(as.integer(lambda), ncol=1)) %% 3)
    block_label <- factor(block_id + 1, levels = 1:3, labels = paste0("block", 1:3))
    res <- design_df; res$.block <- block_label
    blocks <- split(res[ , setdiff(names(res), ".block"), drop=FALSE], res$.block)
    list(full = res, blocks = blocks, block_id = block_label)
  }
  #' @importFrom utils combn
  assess_confounding_3level <- function(design_df, lambda, report = TRUE) {
    out_assign <- assign_blocks_3level(design_df, lambda)
    full <- out_assign$full
    m <- ncol(design_df)
    main_confounded <- character(0); two_confounded <- character(0)
    for (j in seq_len(m)) {
      tab <- table(full$.block, full[[j]])
      balanced <- all(apply(tab, 2, function(col) length(unique(as.numeric(col))) == 1))
      if (!balanced) main_confounded <- c(main_confounded, colnames(full)[j])
    }
    if (m >= 2) {
      cols <- combn(seq_len(m), 2, simplify = FALSE)
      for (pp in cols) {
        i <- pp[1]; j <- pp[2]
        cells <- paste0(full[[i]], "_", full[[j]])
        tab <- table(full$.block, cells)
        balanced <- all(apply(tab, 2, function(col) length(unique(as.numeric(col))) == 1))
        if (!balanced) two_confounded <- c(two_confounded, paste0(colnames(full)[i], ":", colnames(full)[j]))
      }
    }
    res <- list(lambda = lambda, n_main_conf = length(main_confounded), n_two_conf = length(two_confounded), main_confounded = main_confounded, two_confounded = two_confounded)
    if (report) cat("lambda:", paste(lambda, collapse=","), "-> main_conf:", res$n_main_conf, " two_conf:", res$n_two_conf, "\n")
    res
  }
  enumerate_coef3_dynamic_local <- function(p) {
    enumerate_coef3_dynamic(p)
  }

  auto_choose_block_generator_3level <- function(design_df, top_k = 3, verbose = TRUE) {
    p <- ncol(design_df); cands <- enumerate_coef3_dynamic_local(p)
    if (length(cands) == 0) stop("no block coefficients")
    assessed <- lapply(cands, function(co) assess_confounding_3level(design_df, co, report = FALSE))
    summary_df <- do.call(rbind, lapply(assessed, function(x) data.frame(lambda = paste(x$lambda, collapse=","), n_main_conf = x$n_main_conf, n_two_conf = x$n_two_conf, stringsAsFactors = FALSE)))
    ord <- order(summary_df$n_main_conf, summary_df$n_two_conf)
    summary_df <- summary_df[ord, , drop = FALSE]
    top <- summary_df[1:min(top_k, nrow(summary_df)), , drop = FALSE]
    if (verbose) { cat("Top block candidates (3-level):\n"); print(top, row.names = FALSE) }
    top_assessed <- assessed[ord[1:min(top_k, length(ord))]]
    list(summary = summary_df, top = top, top_assessed = top_assessed)
  }

  make_blocked_design_3level <- function(design_df, lambda = NULL, auto = TRUE, top_k = 3, verbose = TRUE) {

    if (!is.null(lambda) && length(lambda) != ncol(design_df))
      stop("lambda length mismatch")

    if (auto) {
      cand <- auto_choose_block_generator_3level(design_df, top_k = top_k, verbose = verbose)
      chosen <- cand$top_assessed[[1]]$lambda
      chosen_report <- cand$top_assessed[[1]]
    } else {
      chosen <- lambda
      chosen_report <- assess_confounding_3level(design_df, chosen, report = verbose)
    }

    assigned <- assign_blocks_3level(design_df, chosen)$blocks

    for (bname in names(assigned)) {
      cat("\n=== ", bname, " (", nrow(assigned[[bname]]), "runs) ===\n", sep = "")
      print(assigned[[bname]])
    }

    report <- list(
      main_confounded = chosen_report$main_confounded,
      two_confounded  = chosen_report$two_confounded
    )


    scores <- list(
      n_main_conf = length(report$main_confounded),
      n_two_conf  = length(report$two_confounded),
      confounding_score = length(report$main_confounded) +
        length(report$two_confounded)
    )


    return(list(
      chosen_design = design_df,
      chosen_report = report,
      scores        = scores,   # NEVER NULL now
      lambda        = chosen
    ))
  }

  finalize_MA3_output <- function(finals, r, max_results = 20) {
    if (length(finals) == 0) return(list())
    scoremat <- do.call(rbind, lapply(finals, function(x) if (is.null(x$score)) rep(Inf,4) else x$score))
    ord <- order(scoremat[,1], scoremat[,2], scoremat[,3], scoremat[,4])
    finals <- finals[ord[1:min(max_results, length(ord))]]
    output <- lapply(finals, function(x) {
      gens_str <- sapply(x$gens, function(co) coef3_to_string(as.integer(co)))
      design <- x$design
      colnames(design) <- make_factor_names(ncol(design))
      list(generators = x$gens, generators_str = gens_str, design = design, A = x$score)
    })
    output
  }

  if (!is.numeric(n) || !is.numeric(k)) stop("n and k must be numeric")
  r <- as.integer(n - k)
  if (r <= 0) stop("n - k must be >= 1")
  base <- gen_3level_from_base(r)

  start_time <- Sys.time()
  beam <- list(list(gens = list(), design = base, key = design_canonical_key_ternary(base), score = NULL))

  depth <- 0
  while (depth < k) {
    depth <- depth + 1
    elapsed <- as.numeric(difftime(Sys.time(), start_time, units = "secs"))
    if (elapsed > time_limit) { warning("Time limit exceeded, returning best so far."); break }
    if (verbose) cat("3-level search depth", depth, "| beam size", length(beam), "| elapsed:", round(elapsed,1), "s\n")

    candidates_next <- list()
    for (node in beam) {
      p_now <- ncol(node$design)
      coefs <- enumerate_coef3_dynamic(p_now)
      if (length(coefs) == 0) next
      # order by canonical column code
      codes <- sapply(coefs, function(co) {
        tmp <- add_dependent_col_from_coef3(node$design, co, "TMP"); col_code_ternary(tmp$TMP)
      })
      ordc <- order(codes); coefs <- coefs[ordc]
      for (co in coefs) {
        new_design <- add_dependent_col_from_coef3(node$design, co, paste0("G", depth))
        key <- design_canonical_key_ternary(new_design)
        scr_list <- compute_A3_A6_from_K_3level(new_design)
        score_vec <- c(scr_list$A3, scr_list$A4, scr_list$A5, scr_list$A6)
        candidates_next[[length(candidates_next)+1]] <- list(gens = c(node$gens, list(co)), design = new_design, key = key, score = score_vec)
      }
    }

    if (length(candidates_next) == 0) break
    scoremat <- do.call(rbind, lapply(candidates_next, function(x) x$score))
    ord <- order(scoremat[,1], scoremat[,2], scoremat[,3], scoremat[,4])
    beam <- list(); seen <- new.env(parent = emptyenv())
    for (idx in ord) {
      key <- candidates_next[[idx]]$key
      if (exists(key, envir = seen, inherits = FALSE)) next
      assign(key, TRUE, envir = seen)
      beam[[length(beam)+1]] <- candidates_next[[idx]]
      if (length(beam) >= beam_width) break
    }
  }

  finals <- list(); seen_final <- new.env(parent = emptyenv())
  for (node in beam) {
    if (exists(node$key, envir = seen_final, inherits = FALSE)) next
    assign(node$key, TRUE, envir = seen_final)
    finals[[length(finals)+1]] <- node
  }
  if (length(finals) == 0) return(list())
  # order and trim to max_results
  scoremat <- do.call(rbind, lapply(finals, function(x) x$score))
  ordf <- order(scoremat[,1], scoremat[,2], scoremat[,3], scoremat[,4])
  finals <- finals[ordf[1:min(length(ordf), max_results)]]

  # prepare output with readable labels and attach exact A-values (A3..A6)
  output <- lapply(finals, function(x) {
    Avals <- compute_A3_A6_from_K_3level(x$design)
    gens_str <- sapply(x$gens, function(co) coef3_to_string(as.integer(co)))
    design <- x$design
    colnames(design) <- c(make_factor_names(r), paste0("G", seq_len(ncol(design)-r)))
    list(generators = x$gens, generators_str = gens_str, design = design, A = c(Avals$A3, Avals$A4, Avals$A5, Avals$A6))
  })

  # rename generated columns to nice letters
  for (i in seq_along(output)) output[[i]]$design <- rename_generated_columns(output[[i]]$design, r)

  if (verbose) {

    print_final_output_3level <- function(res3, top_index = 1, top_k = 5) {

      res3[[top_index]]$design <- rename_generated_columns(res3[[top_index]]$design)
      design3 <- res3[[top_index]]$design


      cat("Generators:", paste(res3[[top_index]]$generators_str, collapse=", "), "\n\n")

      if (!is.null(res3[[top_index]]$A)) {
        Avals <- res3[[top_index]]$A
        names(Avals) <- paste0("A", 3:6)
        cat("Word length pattern (A3-A6):\n")
        cat(paste(names(Avals), Avals, sep=" = ", collapse=", "), "\n\n")
      }

      cat("Fractional Factorial Design:\n")
      print(design3, row.names = FALSE)
      cat("\n")

      aliases <- list_alias_structure_3level(design3)
      cat("Alias Structure:\n")
      if (length(aliases) == 0) {
        cat("  None\n")
      } else {
        for (a in aliases) cat("  ", paste(a, collapse=" = "), "\n")
      }
      cat("\n")


      cat("Blocked Fractional Factorial Design:\n\n")
      res_blocked3 <- make_blocked_design_3level(design3, auto = TRUE, top_k = top_k, verbose = FALSE)

      blkdf <- res_blocked3$blocked_design
      for (b in sort(unique(blkdf$Block))) {
        sub <- blkdf[blkdf$Block == b, ]
        sub <- sub[, !(names(sub) %in% "Block"), drop = FALSE]
        cat("=== block", b, "(", nrow(sub), "runs) ===\n")
        print(sub)
        cat("\n")
      }


      cat("==== SCORES CHECK ====\n")
      print(res_blocked3$scores)
      cat("\n")

      cr <- res_blocked3$chosen_report
      cat("Confounding Summary:\n")
      cat("Mask:", paste(cr$mask, collapse=" "), "\n")
      cat("Main Confounded:",
          if (length(cr$main_confounded)==0) "None"
          else paste(cr$main_confounded, collapse=", "), "\n")
      cat("Two-Factor Confounded:",
          if (length(cr$two_confounded)==0) "None"
          else paste(cr$two_confounded, collapse=", "), "\n")

      invisible(NULL)
    }

    for (i in seq_len(min(length(output), max_results))) {
      cat("\n\n==============================\n")
      cat("      DESIGN RANK", i, "\n")
      cat("==============================\n\n")

      print_final_output_3level(output, top_index = i, top_k = 5)
    }
  }

  invisible(output)
}
