% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/sim_autoplot.R
\name{autoplot.sim}
\alias{autoplot.sim}
\title{Create a complete ggplot for a spatial interaction model}
\usage{
\method{autoplot}{sim}(
  object,
  flows = c("full", "destination", "attractiveness"),
  with_names = FALSE,
  with_positions = FALSE,
  show_destination = FALSE,
  show_attractiveness = FALSE,
  show_production = FALSE,
  cut_off = 100 * .Machine$double.eps^0.5,
  adjust_limits = FALSE,
  with_labels = FALSE,
  ...
)
}
\arguments{
\item{object}{a spatial interaction model object}

\item{flows}{\code{"full"} (default),  \code{"destination"} or \code{"attractiveness"}, see
details.}

\item{with_names}{specifies whether the graphical representation includes
location names (\code{FALSE} by default)}

\item{with_positions}{specifies whether the graphical representation is based
on location positions (\code{FALSE} by default)}

\item{show_destination}{specifies whether the position based \code{"full"} flow
figure includes a representation of the destination flows (\code{FALSE} by
default)}

\item{show_attractiveness}{specifies whether the position based \code{"full"} flow
figure includes a representation of the attractivenesses  (\code{FALSE} by
default)}

\item{show_production}{specifies whether the position based \code{"full"} flow
figure includes a representation of the productions (\code{FALSE} by default)}

\item{cut_off}{cut off limit for inclusion of a graphical primitive when
\code{with_positions = TRUE}. In the full flow matrix representation, segments
are removed when their flow is smaller than the cut off. In the
attractiveness or destination representation, disks are removed when the
corresponding value is below the cut off.}

\item{adjust_limits}{if \code{FALSE} (default value), the limits of the position
based graph are not adjusted after removing graphical primitives. This
eases comparison between graphical representations with different cut off
value. If \code{TRUE}, limits are adjusted to the data using the standard
ggplot2 behaviour.}

\item{with_labels}{if \code{FALSE} (default value) names are displayed using plain
texts. If \code{TRUE}, names are shown using labels.}

\item{...}{additional parameters, see details}
}
\value{
a ggplot object
}
\description{
This function represents graphical the flows of a spatial interaction model,
in different direct or aggregated forms.
}
\details{
The graphical representation depends on the values of \code{flows} and
\code{with_positions}. It is based on the data frame representation produced by
\code{\link[=fortify.sim]{fortify.sim()}}.

If \code{with_position} is \code{FALSE} (default value), the graphical representations
are "abstract". Depending on \code{flows} we have the following representations:
\itemize{
\item \code{"full"}: this is the default case for which the full flow matrix is
represented. It is extracted from the spatial interaction model with
\code{\link[=flows]{flows()}} and displayed using a matrix representation with origin locations
in rows and destination locations in columns. The colour of a cell
corresponds to the intensity of a flow between the corresponding locations.
To mimic the standard top to bottom reading order of a flow matrix, the top
row of the graphical representation corresponds to the first origin location.
\item \code{"destination"}: the function computes the
incoming flows for destination locations (using \code{\link[=destination_flow]{destination_flow()}}) and
represents them with a bar plot (each bar is proportional to the incoming
flow);
\item \code{"attractiveness"}: the function uses a bar plot to represent the
attractivenesses of the destination locations (as given by
\code{\link[=attractiveness]{attractiveness()}}). This is interesting for dynamic models where those
values are updated during the iterations (see \code{\link[=blvim]{blvim()}} for details). When
the calculation has converged (see \code{\link[=sim_converged]{sim_converged()}}), both \code{"destination"}
and \code{"attractiveness"} graphics should be almost identical.
}

When the \code{with_names} parameter is \code{TRUE}, the location names
(\code{\link[=location_names]{location_names()}}) are used to label the axis of the graphical
representation. If names are not specified, they are replaced by indexes.

When the  \code{with_positions} parameter is \code{TRUE}, the location positions
(\code{\link[=location_positions]{location_positions()}}) are used to produce more "geographically informed"
representations. Notice that if no positions are known for the locations, the
use of \code{with_positions = TRUE} is an error. Depending on \code{flows} we have the
following representations:
\itemize{
\item \code{"full"}: this is the default case for which the full flow matrix is
represented. Positions for both origin and destination locations are needed.
The representation uses arrows from origin location positions to destination
location positions. The thickness of the lines (\code{linewidth} aesthetics) is
proportional to the flows. Only segments that carry a flow above the
\code{cut_off} value are included. When the spatial interaction model is not
bipartite (see \code{\link[=sim_is_bipartite]{sim_is_bipartite()}}), zero length segments corresponding to
self exchange are removed. Additional parameters in \code{...} are submitted to
\code{\link[ggplot2:geom_segment]{ggplot2::geom_segment()}}. This can be used to override defaults parameters
used for the arrow shapes, for instance. Those parameters must be named. In
addition to the individual flows, the representation can include location
based information. If \code{show_production} is \code{TRUE}, the production constraints
(obtained by \code{\link[=production]{production()}}) are shown as disks centred on the origin
locations. If \code{show_destination} is \code{TRUE}, incoming flows (obtained by
\code{\link[=destination_flow]{destination_flow()}}) are shown as disks centred on the destination
locations. If \code{show_attractiveness} is \code{TRUE}, attractivenesses (obtained by
\code{\link[=attractiveness]{attractiveness()}}) are shown as disks centred on the destination locations.
\code{show_destination} and \code{show_attractiveness} are not compatible (only one can
be \code{TRUE}). \code{show_production} is compatible with \code{show_destination} or
\code{show_attractiveness} for bipartite models only (see \code{\link[=sim_is_bipartite]{sim_is_bipartite()}}).
When disks are used, segments are drawn without arrows, while the default
drawing uses arrows. This can be modified with the additional parameters.
\item \code{"destination"}: the function draws a disk at each destination location
using for the \code{size} aesthetics the incoming flow at this destination
location (using \code{\link[=destination_flow]{destination_flow()}}). Only destinations with an incoming
flow above  the \code{cut_off} value are included.
\item \code{"attractiveness"}: the function draws a disk at each destination location
using for the \code{size} aesthetics the attractiveness of the destination. When
the calculation has converged (see \code{\link[=sim_converged]{sim_converged()}}), both \code{"destination"}
and \code{"attractiveness"} graphics should be almost identical. Only destinations
with an attractiveness above  the \code{cut_off} value are included.
}

For the position based representations and when \code{with_names} is \code{TRUE}, the
names of the destinations are added to the graphical representation . If
\code{with_labels} is \code{TRUE} the names are represented as labels instead of plain
texts (see \code{\link[ggplot2:geom_text]{ggplot2::geom_label()}}). If the \code{ggrepel} package is installed,
its functions are used instead of \code{ggplot2} native functions. When disks are
used to show aggregated flows, the names match the chosen locations: for
destination flow and attractiveness, destination locations are named, while
for production, origin locations are named (they can be both named when the
model is bipartite).
}
\examples{
\dontshow{if (requireNamespace("ggplot2", quietly = TRUE)) withAutoprint(\{ # examplesIf}
positions <- as.matrix(french_cities[1:10, c("th_longitude", "th_latitude")])
distances <- french_cities_distances[1:10, 1:10] / 1000 ## convert to km
production <- rep(1, 10)
attractiveness <- log(french_cities$area[1:10])
flows <- blvim(distances, production, 1.5, 1 / 150, attractiveness,
  origin_data = list(
    names = french_cities$name[1:10],
    positions = positions
  ),
  destination_data = list(
    names = french_cities$name[1:10],
    positions = positions
  ),
  bipartite = FALSE
)
ggplot2::autoplot(flows)
## bar plots should be almost identical if convergence occurred
sim_converged(flows)
ggplot2::autoplot(flows, "destination")
ggplot2::autoplot(flows, "attractiveness")
## names inclusion
ggplot2::autoplot(flows, "destination", with_names = TRUE) +
  ggplot2::coord_flip()
ggplot2::autoplot(flows, with_names = TRUE) +
  ggplot2::theme(axis.text.x = ggplot2::element_text(angle = 90))
## positions
ggplot2::autoplot(flows, "attractiveness", with_positions = TRUE) +
  ggplot2::scale_size_continuous(range = c(0, 6)) +
  ggplot2::coord_sf(crs = "epsg:4326")
ggplot2::autoplot(flows, "destination",
  with_positions = TRUE,
  with_names = TRUE
) +
  ggplot2::scale_size_continuous(range = c(0, 6)) +
  ggplot2::coord_sf(crs = "epsg:4326")
ggplot2::autoplot(flows, "destination",
  with_positions = TRUE,
  with_names = TRUE, with_labels = TRUE
) +
  ggplot2::scale_size_continuous(range = c(0, 6)) +
  ggplot2::coord_sf(crs = "epsg:4326")
ggplot2::autoplot(flows, with_positions = TRUE) +
  ggplot2::scale_linewidth_continuous(range = c(0, 2)) +
  ggplot2::coord_sf(crs = "epsg:4326")
ggplot2::autoplot(flows,
  with_positions = TRUE,
  arrow = ggplot2::arrow(length = ggplot2::unit(0.025, "npc"))
) +
  ggplot2::scale_linewidth_continuous(range = c(0, 2)) +
  ggplot2::coord_sf(crs = "epsg:4326")
## individual flows combined with destination flows
## no arrows
ggplot2::autoplot(flows,
  with_positions = TRUE,
  show_destination = TRUE
) +
  ggplot2::scale_linewidth_continuous(range = c(0, 2)) +
  ggplot2::coord_sf(crs = "epsg:4326")
## readding arrows
ggplot2::autoplot(flows,
  with_positions = TRUE,
  show_destination = TRUE,
  arrow = ggplot2::arrow(length = ggplot2::unit(0.025, "npc"))
) +
  ggplot2::scale_linewidth_continuous(range = c(0, 2)) +
  ggplot2::coord_sf(crs = "epsg:4326")
\dontshow{\}) # examplesIf}
}
\seealso{
\code{\link[=fortify.sim]{fortify.sim()}}
}
