#' Bayesian Nonparametric Sensitivity Analyses of Multiple Testing Procedures for p-values with Arbitrary Dependence (Intercorrelations)
#'
#' Given inputs of *p*-values `p` from `m = length(p)` hypothesis tests and their error rates `alpha`, this `R` package function `bnpMTP()` performs sensitivity analysis and uncertainty quantification for Multiple Testing Procedures (MTPs) based on a mixture of Dirichlet process (DP) prior distribution (Ferguson, 1973) supporting all MTPs providing Family-wise Error Rate (FWER) or False Discovery Rate (FDR) control for *p*-values with arbitrary dependencies, e.g., due to tests performed on shared data and/or correlated variables, etc. From such an analysis, `bnpMTP()` outputs the distribution of the number of significant *p*-values (discoveries); and a *p*-value from a global joint test of all `m`  null hypotheses, based on the probability of significance (discovery) for each *p*-value.\cr 
#' The DP-MTP sensitivity analysis method can analyze a large number of *p*-values obtained from any mix of null hypothesis testing procedures, including one-sample and/or multi-sample tests of: location, scale, higher moments, distribution, or symmetry; correlation, association, regression coefficients, odds ratios; change-points; runs; networks; classification; clustering; posterior distributions; model fit; outlyingness; and/or continuous hypothesis tests (e.g., performed on a realization of a random field); among other tests. Also, this sensitivity analysis method handles *p*-values from traditional offline testing; and from *online testing* performed on a stream of null hypotheses arriving one-by-one (or in blocks) over time (or asynchronously), where each test is based only on previous fixed test decisions and evidence against the current hypothesis, with unknown future data and total number of hypotheses being tested (potentially infinite) (Robertson, 2023).
#' \cr In any case, the DP-MTP sensitivity analysis method assumes that each *p*-value follows a super-uniform distribution under the null hypothesis (i.e., either a Uniform(0,1) distribution under a calibrated test; or a stochastically larger distribution under a conservative test). More **Details** about this method are below (run `bnpMTP` in `R` console to view code) and provided by Karabatsos (2025).
#' @usage bnpMTP( p = NULL , alpha = 0.05 , N = 1000 , mu = 1 )
#' @param p A vector of *p*-values. They can have arbitrary dependence.
#' @param alpha Scalar (default `alpha = 0.05`) or vector of Type I error probabilities specifying the error rate to be spent or invested (Foster & Stine, 2008) on each of the `m = length(p)` null hypothesis tests, with total error rate `sum(alpha)` such as `sum(alpha) = 0.05` (which can be less than the desired total error rate if input `p` excludes *p*-values from future online tests to be done later), as:\cr "Once we have spent this (total) error rate, it is gone" (Tukey, 1991, p.104).
#' \cr Input `alpha` helps define the random significance *thresholds*, by: \cr `Delta_nu(r) = alpha * beta_nu(r)`, \cr for each ordered *p*-value `p(r)` from `sort(p)`, based on a random probability measure `nu` on `[0, m]` from a mixture of Dirichlet process, and on a positive reshaping parameter `beta_nu(.)`, used for MTP sensitivity analysis (see **Details**).
#' \itemize{
#' \item `bnpMTP()` converts any scalar input `alpha` into the following vector input:\cr `alpha <- rep(alpha * (1 / m), m) = alpha * w`, which defines the thresholds of the *Bonferroni* (1936) MTP for `m = length(p) = length(w)` tests, where each *p*-value `p[i]` is assigned a *standard weight* `w[i] = 1 / m`. 
#' \item Input `alpha` can be specified as a vector: `alpha = alpha0 * w` for some small positive number `alpha0` (e.g., `alpha0 = 0.05`), which defines the significance thresholds `alpha` of the *weighted Bonferroni* MTP, based on a prior distribution vector `w` representing the degree of belief for each of the `m = length(p)` null hypotheses (Genovese et al.2006), where `sum(w) = 1` (Tamhane & Gou 2022), or `sum(w) < 1` if `p` excludes *p*-values from future online tests to be done later (Tian & Ramdas, 2021, Section 2).
#' \item Some alternatives for vector input `alpha` are defined by: the *Šidák* (1967), *Fallback* (Wiens & Dmitrienko, 2005), and *Adaptive Discarding* MTPs for offline or online FWER control (Tian & Ramdas, 2021); and *LORD* (Javanmard & Montanari 2018) and other *generalized alpha investing* methods (Aharoni & Rosset, 2014) for online FDR control (Robertson et al. 2023). 
#' }
#' @param N,mu Number of random samples drawn from the mixture of Dirichlet process (`DP(M, nu_0)`) prior distribution for the random probability measure `nu` defined on `[0, m]`, with mass parameter `M` assigned an `Exponential(mu)` hyper-prior distribution with rate `mu`, where `m = length(p)`. Defaults: `N = 1000` and `mu = 1`.
#' @return Output of the DP-MTP sensitivity analysis results, as a list containing the following objects:
#' \item{r.hat_nu}{A vector of `N` samples of the number, `r.hat_nu`, of the smallest *p*-values (from input `p`) that are significant discoveries, based on `N` samples of the random probability measure `nu` defined on `[0, m]` for `m = length(p)` hypothesis tests, with `nu` assigned the mixture of DP prior distribution.}
#' \item{Delta_nu.r,beta_nu.r}{Two `N`-by-`(m + 1)` matrices of `N` mixture of DP samples of the threshold function and the shape function for the sorted *p*-values (`sort(p)`) in `colnames(Delta_nu.r)` and `colnames(beta_nu.r)`, respectively. Using `I = cbind(1:N,r.hat_nu+1)`, the `N` samples of threshold `Delta_nu(r.hat_nu)` and shape `beta_nu(r.hat_nu)` are obtained from `Delta_nu.r[I]` and `beta_nu.r[I]`.}
#' \item{Table}{A `3`-by-`(m + 1)` matrix reporting the probability of significance (`PrSig.p`) for each of the `m = length(p)` total *p*-values in input `p` with respective error rate(s) `alpha`, based on the mixture DP prior. \cr For each ordered *p*-value `p_(r)` from `sort(p)`, the *probability of significance* is estimated by the proportion of `N` samples of `r.hat_nu` which satisfy inequality: \cr `p_(r) <= Delta_nu(r.hat_nu)`, for `r` = `1,...,m = length(p)`. \cr The last column of the output `Table` shows the prior predictive *p*-value from the global joint test that all `m = length(p)` null hypotheses are true; and their total spent error rate `sum(alpha)` and their `max(PrSig.p)`. This *p*-value equals:
#' \cr `min(1 - PrSig.p) = 1 - max(PrSig.p)`
#' \cr                  `=  mean(r.hat_nu == 0) = 1 - mean(r.hat_nu > 0)` 
#' \cr based on the idea that the joint null hypothesis should be rejected if at least one of the `m` null hypotheses is rejected (Simes, 1986).}
# Below, putting #' in front of @keywords adds a list of keywords in the Index section of the R package's user's manual:
# @keywords Dirichlet process, Sensitivity analysis, Multiple testing, False discovery rate, Family-wise error rate.
#
# Below, following: https://roxygen2.r-lib.org/articles/namespace.html
# using @importFrom to import global function definition of 'rexp()' and 'rgamma()' from the 'stats' R package,
# which adds the lines: importFrom(stats,rexp) and importFrom(stats,rgamma)
# into the NAMESPACE file of this R package (similarly as done in NAMESPACE file of 'igraph' R package)
#' @importFrom stats rexp rgamma
#' @export
#' @details The Dirichlet process (`DP`) based MTP sensitivity analysis method (Karabatsos, 2025) assigns a mixture of `DP(M, nu_0)` prior distribution that flexibly supports the entire space of random probability measures `nu` defined on the interval `[0, m]` for `m = length(p)` hypothesis tests, with `Exponential(mu)` hyper-prior distribution assigned to the `DP` mass parameter `M`, and with (mean) baseline probability measure (`nu_0`) defined by the Benjamini & Yekutieli (2001) MTP. In turn, this mixture DP prior also supports the space of all MTPs providing FWER or FDR control for *p* values with arbitrary dependencies, because each of these MTPs can be uniquely characterized by a random probability measure `nu`, based on the *shape function approach* to multiple hypothesis testing (Blanchard & Roquain, 2008, Sections 3.1-3.2; Lemma 3.2, Equation 6, pp.970–972, 976).
#'
#' Specifically, the DP random probability measure, `nu`, drives the random number, `r.hat_nu`, of the smallest *p*-values (from input `p` with `length(p) = m`) that are significant discoveries, defined via the following DP random *step-up procedure* (using inequality <=):
#' 
#'`r.hat_nu = max[r \in {0,1,...,m} | p_(r)` <= `alpha * beta_nu(r)]`,
#' 
#' where for `r` = `0,1,...,m`, the `p_(r)` (with `p_(0) := 0`) are the ordered *p*-values (`sort(p)`) sorted in increasing order, with values of random significance *thresholds*:\cr `Delta_nu(r) = alpha * beta_nu(r)`,\cr based on a random *shape function*:\cr `beta_nu(r) = integral_0^r x d{nu(x)}` \cr which *reshapes* (Ramdas et al. 2019, pp.2795-2796) or modifies `alpha` into new significance thresholds `Delta_nu(r)` to further account for arbitrary dependencies between *p*-values.
#' 
#' Further details are provided by Karabatsos (2025), who illustrated this DP-MTP sensitivity analysis method on over twenty-eight thousand *p*-values of different hypothesis tests performed on observations of 239 variables from a large dataset.
#' @references Aharoni, E., and Rosset, S. (2014). Generalized alpha-investing: Definitions, optimality results and application to public databases. *Journal of the Royal Statistical Society Series B*, **76**, 771–794. <https://www.jstor.org/stable/24774568>
#' @references Benjamini, Y., and Yekutieli, D. (2001). The control of the false discovery rate in multiple testing under dependency. *Annals of Statistics*, **29**, 1165–1188. <https://www.jstor.org/stable/2674075>
#' @references Blanchard, G., and Roquain, E. (2008). Two simple sufficient conditions for FDR control. *Electronic Journal of Statistics*, **2**, 963–992. `DOI: 10.1214/08-EJS180`
#' @references Bonferroni, C. (1936). Teoria statistica delle classi e calcolo delle probabilità. *Pubblicazioni del R Istituto Superiore di Scienze Economiche e Commerciali di Firenze*, **8**, 3–62.
#' @references Ferguson, T. (1973). A Bayesian analysis of some nonparametric problems. *Annals of Statistics*, **1**, 209–230. <https://www.jstor.org/stable/2958008>
#' @references Foster, D., and Stine, R. (2008). Alpha-investing: A procedure for sequential control of expected false discoveries. *Journal of the Royal Statistical Society, Series B*, **70**, 429–444. <https://www.jstor.org/stable/20203833>
#' @references Genovese, C., Roeder, K., and Wasserman, L. (2006). False discovery control with *p*-value weighting, *Biometrika*, **93**, 509–524. <https://www.jstor.org/stable/20441304>
#' @references Javanmard, A., and Montanari, A. (2018). Online rules for control of false discovery rate and false discovery exceedance. *Annals of Statistics*, **46**, 526–554. <https://www.jstor.org/stable/26542797>
#' @references Karabatsos, G. (2025). Bayesian nonparametric sensitivity analysis of multiple test procedures under dependence. *Biometrical Journal*. <https://arxiv.org/abs/2410.08080>. Paper presented in the *14th International Conference on Bayesian Nonparametrics* at UCLA on June 26, 2025.
#' @references Needleman, H., Gunnoe, C., Leviton, A., Reed, R., Presie, H., Maher, C., and Barret, P. (1979). Deficits in psychologic and classroom performance of children with elevated dentine lead levels. *New England Journal of Medicine*, **300**, 689–695. <https://www.nejm.org/doi/10.1056/NEJM197903293001301>
#' @references Ramdas, A., Barber, R., Wainwright, M., and Jordan, M. (2019). A unified treatment of multiple testing with prior knowledge using the *p*-filter. *Annals of Statistics*, **47**, 2790–2821. <https://www.jstor.org/stable/26784046>
#' @references Robertson, D., Wason, J., and Ramdas, A. (2023). Online multiple hypothesis testing. *Statistical Science*, **38**, 557–575. <https://pmc.ncbi.nlm.nih.gov/articles/PMC7615519/>
#' @references Šidák, Z. (1967). Rectangular confidence regions for the means of multivariate normal distributions. *Journal of the American Statistical Association*, **62**, 626–633. <https://www.jstor.org/stable/2283989>
#' @references Simes, R. (1986). An improved Bonferroni procedure for multiple tests of significance. *Biometrika*, **73**, 751–754. <https://www.jstor.org/stable/2336545>
#' @references Tamhane, A., and Gou, J. (2022). Multiple test procedures based on *p*-values. Chapter 2 of *Handbook of Multiple Comparisons*, by X. Cui, T. Dickhaus, Y. Ding, and J. Hsu (Eds.). CRC Press.
#' @references Tian, J., and Ramdas, A. (2021). Online control of the familywise error rate. *Statistical Methods in Medical Research*, **30**, 976–993. <https://pubmed.ncbi.nlm.nih.gov/33413033/>
#' @references Tukey, J. (1991). The philosophy of multiple comparisons. *Statistical Science*, **6**, 100–116. <https://www.jstor.org/stable/2245714>
#' @references Wiens, B., and Dmitrienko, A. (2005). The fallback procedure for evaluating a single family of hypotheses. *Journal of Biopharmaceutical Statistics*, **15**, 929–942. <https://www.tandfonline.com/doi/full/10.1080/10543400500265660>
#' @examples
#' #---------------------------------------------------------------------------------------
#' # Consider a classic data set in the field of multiple hypothesis testing procedures.
#' # Needleman (1979,Table 3) from yes/no responses to 11 Teachers' Behavioral survey items
#' # compared 58 children exposed to high lead and 100 children exposed to low lead levels;
#' # by p-values from 11 chi-square null hypothesis tests of equal group % 'yes' responses;
#' # and a 2-tail p-value (0.02) from ANCOVA F-test of null hypothesis of equal group means
#' # in total sum score on the 11 items, while controlling for mother age at child's birth, 
#' # number of pregnancies & educational level; father's socioeconomic status; parental IQ.
#' #---------------------------------------------------------------------------------------
#'
#' # Below, enter the vector of twelve p-values (and then run this R code line, below):
#' p        = c(0.003, 0.05, 0.05, 0.14, 0.08, 0.01, 0.04, 0.01, 0.05, 0.003, 0.003, 0.02)
#' 
#' # Below, name these p-values (then run the three R code lines, below):
#' names(p) = c( "Distractible"  , "Impersistent"  , "Dependent"   , "Disorganized" ,
#'               "Hyperactive"   , "Impulsive"     , "Frustrated"  , "Daydreamer"   ,
#'               "MissEzDirect"  , "MissSeqDirect" , "LowFunction" , "SumScore"     )
#' 
#' # Get results of DP-MTP sensitivity analysis of the p-values: (Run 2 code lines, below):
#' set.seed(123) # for reproducibility of results of Monte Carlo sampling done by bnpMTP()
#' Result   = bnpMTP( p = p , alpha = 0.05 )
#'
#' # Show probability of significance for each of m = length(p) = 12 p-values in input 'p'
#' # based on mixture of DP(M, nu_0) prior; and prior predictive p-value from global test
#' # of all 12 null hypotheses, and their total error sum(alpha) (run R code line below):
#' Result$Table
#'
#' # Summarize mixture of DP(M, nu_0) prior distribution of number of significant p-values:
#' quantile( Result$r.hat_nu )
#' 
#' #---------------------------------------------------------------------------------------
#' # Now suppose that the p-values were obtained from an online stream of hypothesis tests,
#' # with more hypothesis tests to be performed in the future (possibly infinite). 
#' # Accordingly, we specify the alpha vector based on p-value weights (w) defined 
#' # by the geometric distribution on {1,2,...} with 'success' probability 0.35,
#' # with sum(w) < 1 over the currently available twelve p-values in input p. 
#' #---------------------------------------------------------------------------------------
#'
#' # Get results of DP-MTP sensitivity analysis of the p-values: (Run 5 code lines, below):
#' alpha0   = 0.05
#' w        = dgeom( ( 1 : length(p) ) - 1 , prob = 0.35 ) # specify p-value weights.
#' alpha    = alpha0 * w
#' set.seed(123) # for reproducibility of results of Monte Carlo sampling done by bnpMTP()
#' Online   = bnpMTP( p = p , alpha = alpha )
#'
#' # Show probability of significance for each of m = length(p) = 12 p-values in input 'p'
#' # based on mixture of DP(M, nu_0) prior; and prior predictive p-value from global test
#' # of the 12 null hypotheses so far and their total error sum(alpha) (run line below):
#' Online$Table
#'
#' # Summarize mixture of DP(M, nu_0) prior distribution of number of significant p-values:
#' quantile( Online$r.hat_nu )
#
bnpMTP <- function( p = NULL , alpha = 0.05 , N = 1000 , mu = 1 ){
  	#===============================================================================================================================
  	# Verify the inputs:
  	#===============================================================================================================================
  	# Check inputs of the p-values (vector 'p'):
  	if( is.null(names(p)) ){ names(p) = paste('Test', 1 : length(p), sep = '') }
  	Names 		=	make.unique(names(p))
  	p 			= 	suppressWarnings( as.numeric( p ) )
  	ok.p		=  	all(!is.null(p)) & all(!is.na(p)) & all(!is.nan(p) , na.rm = T)
  	ok.p		=	ok.p  & all( p >= 0 , na.rm = T) & all( p <= 1 , na.rm = T )
  	if(!ok.p){ stop("Input 'p' must be a vector containing one or more p-values from the interval [0,1].")	}
  	names(p)	= 	Names
  	m			=	length(p)
  	#
  	# Check input 'alpha' (error rate(s)):
  	if( is.null(alpha)     ){ alpha =  rep( 0.05  / m , m)           }
  	alpha 		= 	suppressWarnings( as.numeric( alpha ) )
  	if( length(alpha) == 1 ){ alpha =  rep( alpha * (1 / m) , m)     }
  	ok.alpha	= 	all(!is.null(alpha)) & all( !is.na(alpha) ) & all( !is.nan(alpha), na.rm = T)
  	ok.alpha	= 	ok.alpha & all(alpha > 0, na.rm = T) & all(alpha < 1, na.rm = T)
  	sum.alpha	= 	sum(alpha) 
  	ok.alpha	= 	ok.alpha & ( sum.alpha > 0 ) & ( sum.alpha < 1 )
  	ok.alpha	= 	ok.alpha & ( ( length(alpha) == 1 ) | ( length(alpha) == m ) )
  	if(!ok.alpha){ stop("Input 'alpha' must be NULL (leading to default value 0.05), 
       or 
       have non-negative number(s) with 0 < sum(alpha) < 1,
       and with length(alpha) = 1 or length(alpha) = length(p).") }
  	names(alpha)= 	Names
  	#
  	# Check input 'N' (number of Monte Carlo samples from the Dirichlet Process):
  	N 			= 	suppressWarnings( as.integer( N ) )
  	ok.N		= 	all(!is.null(N)) & all(!is.na(N)) & all(!is.nan(N), na.rm = T) & (length(N) == 1)
  	ok.N		= 	ok.N & all(N > 0, na.rm = T) & all(is.finite(N), na.rm = T)
  	if(!ok.N){ stop("Input 'N' must be a finite positive integer.") 							}
  	#
  	# Check input 'mu' (rate parameter of M ~ Exponential(mu) hyper-prior for DP concentration parameter M):
  	mu 			= 	suppressWarnings( as.numeric( mu ) )
  	ok.mu		= 	all(!is.null(mu))  & all(!is.na(mu)) & all(!is.nan(mu), na.rm = T) & ( length(mu) == 1 ) 
  	ok.mu		= 	ok.mu    &   all(mu > 0, na.rm = T)   & 	all(is.finite(mu), na.rm = T) 
  	if(!ok.mu){ stop("Input 'mu' must be a finite positive number.") 				          }
  	#
  	#===============================================================================================================================
  	# If only m = 1 p-value in input p, then can compute results directly without sampling the Dirichlet process:
  	#===============================================================================================================================
	if ( m == 1 ) {
		beta_nu.r			= 1	# beta_nu.r is an integral from x = 0 to 1 (with r = 1)
		Delta_nu.r			= alpha * beta_nu.r	# significance threshold
		PrSig.p				= ifelse( p <= Delta_nu.r, 1, 0 )
		r.hat_nu		    = rep( PrSig.p, N )
		beta_nu.r.hat		= rep( ifelse( PrSig.p, beta_nu.r	, 0), N )
		Delta_nu.r.hat		= rep( ifelse( PrSig.p, Delta_nu.r  , 0), N )
   		# If m = 1 and r = 1, then always, the integral beta_nu.r = 1 
   		# (where beta_nu.r is an integral from 0 to 1 where r = 1).
   		# Also, if r = 0 (referring to the zeroth ordered p-value, p_(0), set to p_(0) = 0 ), 
   		# then: beta_nu.r = 0 (because an integral for 0 to 0 equals zero) and Delta_nu.r = alpha * beta_nu.r = alpha * 0 = 0
		beta_nu.r			= NULL
		Delta_nu.r		  	= NULL
  	}
  	#===============================================================================================================================
  	# Sample the Dirichlet process when there are m > 1 p-values:
  	#===============================================================================================================================
  	if ( m > 1 ) {
   		# ---------------------------------------------------------------------------------------------------------------------------
   		# Organize the p-values and their weights:
   		# ---------------------------------------------------------------------------------------------------------------------------
   		p.sort.			 	= sort(p, index.return = T)
   		p.sort 			 	= p.sort.$x  # the m p-values sorted in increasing order.
   		p.sortI			 	= p.sort.$ix # ordering index vector for sorted p-values.
   		# p.sort[match(names(p), names(p.sort))]	# returns p values (p) in their original order.
   		P.sort 			 	= matrix(p.sort, nrow = 1)[rep(1, N), ] # N by m matrix.
   		colnames(P.sort) 	= names(p.sort)
		P.sort           	= cbind(0, P.sort) # N by m+1 matrix. Adding first column of zeros for first ordered p-value p_(0) := 0.
 		alpha.sort.by.p	= alpha[p.sortI]
 		# alpha.sort.by.p[match(names(alpha), names(alpha.sort.by.p))] # returns alpha values in their original order.
 		ALPHA  			 	= matrix(alpha.sort.by.p, ncol = 1)[, rep(1, N)]#m by N matrix; levels alpha.sort.by.p within each column
   		# ---------------------------------------------------------------------------------------------------------------------------
   		# Draw N samples of the random probability measure nu defined on [0, m]  (for m = length(p) hypothesis tests)
   		# and corresponding shape functions (betas_nu) and thresholds (Deltas_nu)
   		# based on a Dirichlet DP(M, nu_0) process) prior distribution for nu with M ~ Exponential(mu) hyper-prior distribution:
   		# ---------------------------------------------------------------------------------------------------------------------------
   		# Draw N samples of DP mass parameter M from the exponential(mu) hyper-prior distribution:
   		# [extra Nx - N samples used to handle NaNs in beta_nu.r columns caused by small M values producing underflow in rgamma()]
   		Nx               	= N + 100
   		M                	= matrix( rexp( Nx , rate = mu) , nrow = 1 )[ rep(1, m),  ] # m by Nx matrix
   		#
  		# Draw N corresponding samples of nu from DP(M, nu_0) prior distribution:
   		r                	= matrix( 1 : m, nrow = m, ncol = Nx) # m by Nx matrix (1 : m repeat within each column)
   		nu_0             	= (r * sum(1 / (1 : m))) ** (-1)      # m by Nx matrix
  		M.nu_0           	= M * nu_0
  		gamrnd.mNx       	= matrix( rgamma(m * Nx, shape = M.nu_0, rate = 1), m, Nx ) # m by Nx matrix
   		#
 	  	# Get corresponding N DP samples of shape function beta_nu.r values: 
   		beta_nu.r        	= gamrnd.mNx / matrix(colSums(gamrnd.mNx), nrow = 1)[rep(1 , m) ,  ] # m by Nx matrix
   		beta_nu.r        	= ( beta_nu.r[ , apply(!is.na(beta_nu.r), 2, all) ] )[   , (1 : N) ] # m by N  matrix
   		r                	= r[ , 1 : N ]	 # m by N matrix (1 : m repeat within each column)
   		# Below, beta_nu.r is m by N matrix of N DP random shape function integral evaluated over points 1 : m
  		beta_nu.r        	= apply( r * beta_nu.r , 2 , cumsum ) # N integrals from 0 to r for r = 1 : m
   		#
  		# Get N corresponding random samples of Delta_nu.r:
		Delta_nu.r       	= t( ALPHA * beta_nu.r ) # N by m matrix
      	Delta_nu.r       	= cbind(0, Delta_nu.r)   # N by m+1 matrix. Adding delta = zeros for first ordered p-value p_(0) := 0.
      	colnames(Delta_nu.r)= paste('p_(', 0 : m , ')=', c(0, p.sort) , sep = '') # Put sorted p-values as column names
 		beta_nu.r        	= cbind(0, t(beta_nu.r)) # N by m+1 matrix. Adding delta = zeros for first ordered p-value p_(0) := 0.
      	colnames(beta_nu.r) = paste('p_(', 0 : m , ')=', c(0, p.sort) , sep = '') # Put sorted p-values as column names
   		#
   		# Get N corresponding step-up decisions on the R smallest p-values:
  		r.hat_nu      		= max.col( P.sort <= Delta_nu.r , "last" ) - 1  # N of DP samples of r.hat_nu
   		# A first column of zeros in P.sort, in Delta_nu.r, and in beta_nu.r, was NOT included, inconsequentially,
		# for the PISA analysis results presented in the Biometrical Journal article.
		# This R function code bnpMTP() can reproduce the results reported in the Biometrical Journal article,
		# based on using: set.seed(123), and based on using: 
		# Nx = N = 1000 above; and skipping the above code line for beta_nu.r that used apply(!is.na(beta_nu.r), 2, all).
 		R                	= matrix( r.hat_nu , ncol = 1 )[ , rep(1, m) ]# N by m matrix
   		#
   		# Get vector of N DP prior samples of number of discoveries (r.hat_nu) among the m tests:
   		# PrSig.p # DP prior predictive probability of discovery, for each sorted p-value:
   		PrSig.p          	= colMeans( t(r) <= R , na.rm = T) 
   		names(PrSig.p)   	= names(p.sort) # enables verification of the unsorting of PrSig.p done below:
   		# Put PrSig.p in original order of unsorted p-values:
   		PrSig.p          	= PrSig.p[match(names(p), names(p.sort))]# can verify that p and PrSig.p have names in the same order
  	}
  	#===============================================================================================================================
  	# OUTPUTS
  	#===============================================================================================================================
	if (m > 1){Global.label = paste( 'GLOBAL(all ', m , ' hypotheses)' , sep = '' ) }
	if (m ==1){Global.label =     c( 'GLOBAL(1 hypothesis)' ) }	
	# Prior predictive p-value for global test of null hypothesis H0: All m null hypotheses true, is given by:
	# min(1 - PrSig.p) = 1 - max(PrSig.p) = mean(r.hat_nu == 0) = 1 - mean(r.hat_nu > 0)
	Global 					= c    (                min(1 - PrSig.p) ,  sum(alpha) , max(PrSig.p) )	
  	Table 					  	= round( cbind( rbind(  p                ,  alpha      , PrSig.p      ) , Global ) , 3 )
  	colnames(Table)   	   	= c    (                names(p)         ,                                Global.label )
  	rownames(Table)   	   	= c    (               'p-value'         , 'alpha'     , 'PrSig.p'    )
  	names(r.hat_nu) 	    	= "r.hat_nu"
	# Can get the N samples of DP random shape functions beta_nu(r.hat_nu) and thresholds Delta_nu(r.hat_nu), by:
	# I = cbind(1:N, r.hat_nu + 1); beta_nu.r.hat = beta_nu.r[I]; Delta_nu.r.hat = Delta_nu.r[I]
  	return( list( r.hat_nu = r.hat_nu , Delta_nu.r = Delta_nu.r , beta_nu.r = beta_nu.r , Table = Table ) )
  	#===============================================================================================================================
}



# ================================================================================================
# For testing bnpMTP():
# ================================================================================================
#
# ------------------------------------
# Offline testing example:
# ------------------------------------
#  rm( list = ls() )
#  set.seed(123) # for reproducibility
#  p        =  c(0.003, 0.05, 0.05, 0.14, 0.08, 0.01, 0.04, 0.01, 0.05, 0.003, 0.003, 0.02)
# # Below, name these p-values (then run the three R code lines, below):
#  names(p) = c( "Distractible"  , "Impersistent"  , "Dependent"   , "Disorganized" ,
#                "Hyperactive"   , "Impulsive"     , "Frustrated"  , "Daydreamer"   ,
#                "MissEzDirect"  , "MissSeqDirect" , "LowFunction" , "SumScore"     )
#  p = p; alpha = 0.05; N = 1000; mu = 1; 
#
#  Result = bnpMTP(p = p, alpha = alpha, N = N, mu = mu)
#  Result$Table
#
# The prior predictive *p*-value for the global test of 
# null hypothesis H0 that all `m` null hypotheses are true, 
# satisfies the following equalities:
# PrSig.p = Result$Table[ 3, 1 : length(p) ]
# min( 1 - PrSig.p )
# 1 - max( PrSig.p )
# mean( Result$r.hat_nu == 0 )
# 1 - mean( Result$r.hat_nu > 0 )
# 
# quantile( Result$r.hat_nu )
# Result$Delta_nu.r
# Result$beta_nu.r
# 
# The N samples of the threshold Delta_nu(r.hat_nu)
# and shape beta_nu(r.hat_nu), using:
# I = cbind( 1 : N , Result$r.hat_nu + 1)
# are:
# Result$Delta_nu.r[I] 
# and:
# Result$beta_nu.r[I]
#
# ------------------------------------
# Online testing example:
# ------------------------------------
# set.seed(123) # for reproducibility
# w        = ( (1 - 0.35) ^ ( ( 1 : length(p) ) - 1 ) ) * 0.35
# alpha = 0.05 * w # alternative alphas.
# Online = bnpMTP(p = p, alpha = alpha, N = N, mu = mu)
# Online$Table
# quantile( Online$r.hat_nu )
#
# The prior predictive *p*-value for the global test of 
# null hypothesis H0 that all `m` null hypotheses are true, 
# satisfies the following equalities:
# PrSig.p = Online$Table[ 3, 1 : length(p) ]
# min( 1 - PrSig.p )
# 1 - max( PrSig.p )
# mean( Online$r.hat_nu == 0 )
# 1 - mean( Online$r.hat_nu > 0 )
# 
# The N samples of the threshold Delta_nu(r.hat_nu)
# and shape beta_nu(r.hat_nu), using:
# I = cbind( 1 : N , Online$r.hat_nu + 1)
# are:
# Online$Delta_nu.r[I] 
# and:
# Online$beta_nu.r[I]
#
# ------------------------------------
# Trivial example (1 hypothesis test):
# ------------------------------------
# set.seed(123) # for reproducibility
# p = p[1] ; alpha = 0.05; 
# N = 1000 ; mu = 1
#
# Trivial = bnpMTP(p = p, alpha = alpha, N = N, mu = mu)
#
# Trivial$Table
# quantile( Trivial$r.hat_nu )
#
# The prior predictive *p*-value for the global test of 
# null hypothesis H0 that all `m` null hypotheses are true, 
# satisfies the following equalities:
# PrSig.p = Trivial$Table[ 3, 1 : length(p) ]
# min( 1 - PrSig.p )
# 1 - max( PrSig.p )
# mean( Trivial$r.hat_nu == 0 )
# 1 - mean( Trivial$r.hat_nu > 0 )
# ================================================================================================