## 
# ------------------------------------------------------------------------
# 
# "bandw1(y)" --
# 
# Compute the bandwidth for the kernel density estimators.
# 
# ------------------------------------------------------------------------
##
#' @aliases bandw1
#' @title Computing the Bandwidth of the Kernel Density Estimators.
#' @description The function computes the bandwidth of a smoothed kernel density estimators
#'  using Silverman's rule.
#' @param y A numeric vector representing a Markov process of a vector of points.
#' @details The function compute the bandwidth 
#' using the following formula: 
#' \eqn{h=0.9An^{-1,5},}  
#' where \eqn{A=\min{\left\{\hat{\sigma}; IRQ/1.34\right\}}}, where \eqn{\hat{\sigma}} is the empirical
#' standard deviation of the data and \eqn{IRQ} is the empirical interquartile range.
#' @return Value of the Silverman's rule of thumb bandwidth. 
#' @seealso \code{\link{findBestEpsilon}},
#' \code{\link{fastNadaraya}}.
#' @references Tsybakov, A. B. (2018). Introduction to Nonparametric Estimation. \emph{SpringerLink.}
#' @keywords "Regenerative bootstrap" "Markov chains" "Small ensembles"
#' @export
#' @examples 
#' n=200# the length of the process
#' # Generating the AR(1) process
#' coeff=0.75
#' X = arima.sim(n=n, list(ar = c(coeff)))
#' h = bandw1(X)

bandw1 <- function(y) {
  if (!is.numeric(y)) {
    stop("Error: 'y' must be a numeric vector.")
  }
  # IQR with type 4 corresponds to the linear interpolation
  # of the empirical cdf
  C=(2*pi*sqrt(2))^(-0.2)
  iqr <- IQR(y,type=4)
  n <- length(y)
  s <- sd(y)
  return(C*min(s,(iqr/1.34)/n^0.2))
}
