
# ------------------------------------------------------------------------
#
# "block.sub<- function(x,func,length.block,method=c("movingblock","circular"),...) " --
#
# Block subsampling for time series using blocks of observations.
#
# ------------------------------------------------------------------------
##
#' @aliases block.sub
#' @title Block Subsampling 
#' @description Block Subsampling for Time Series Using Blocks of Observations.
#' @param x A numeric vector or time series data.
#' @param func The function to apply to each block.
#' @param length.block A scalar or a vector indicating the lengths of the blocks. 
#' @param method A character string specifying the type of block subsampling method.
#' Can be \code{"movingblock"} or \code{"circular"}:
#'   \describe{
#'     \item{\code{"movingblock"}}{divides the series into overlapping blocks.}
#'     \item{\code{"circular"}}{wraps the time series around a circle to create Circular Moving Blocks.}
#'   }
#' @param ... Optional additional arguments for the \code{func} function.
#' @details  This function performs block subsampling on time series data using
#'  the Moving or Circular Blocks methods (see Politis et al., 1999). 
#' It applies the specified function \code{func} to each subsample block whose
#'  length are defined by \code{length.block}.
#' The function \code{func} is applied to the vector \code{x} and may return either a real number or 
#' a vector. 
#' @return A matrix containing, in rows, the value of \code{func} applied to 
#' each block. The number of columns in the matrix is equal to the length 
#' of the output of the \code{func} function.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Politis, D. N., Romano, J. P., & Wolf, M. (1999). Subsampling. \emph{Springer N.Y.}. 
#' 
#' @seealso \code{\link{best.block.sub.size}}, \code{\link{rate.block.sub}}.
#' @keywords "Subsampling" "Moving block"
#' @export
#' @examples
#' set.seed(123) 
#' data <- rnorm(100) 
#' result <- block.sub(data, mean, length.block = 5, method="movingblock")




block.sub<- function(x,func,length.block,method=c("movingblock","circular"),...) 
  {
  
  if (!is.function(func)) {
    stop("Argument 'func' must be a valid function.")
  }
  
  if (!(is.numeric(length.block) && all(length.block > 0))) {
    stop("Argument 'length.block' must be a positive numeric scalar or vector.")
  }
  method <- match.arg(method)  
  
  if (method[1]=="movingblock") 
    res=block.sub.mb(x,func,length.block,...)
  else 
    res=block.sub.circular(x,func,length.block,...)
  
  return(res)
} 





block.sub.mb<- function(x,func,length.block,...) {
  n <- length(x)
  y <- func(x,...)
  b <- length.block
  len <- length(y)
  
  # Number of blocks
  q <- n-b+1
  res <- matrix(nrow=q,ncol=len)
  
  
  # Number of sections
  s <- ((n-1) %/% b)+1
  # Bootstrapped series
  nx <- numeric(s*b)
  
  for (idx in 1:q) {
    res[idx,] <- func(x[idx:(idx+b-1)],...)
  }
  return(res)
} 




#######################


block.sub.circular<- function(x,func,length.block,...) {
  n <- length(x)
  y <- func(x,...)
  b <- length.block
  len <- length(y)
  
  # Number of blocks
  q <- n
  res <- matrix(nrow=q,ncol=len)
  
  
  # Circular blocks 
  x <- c(x, x[1:(b - 1)])
  s <- ((n - 1)%/%b) + 1
  nx <- numeric(s * b)
  
  # Number of sections
  s <- ((n-1) %/% b)+1
  # Bootstrapped series
  nx <- numeric(s*b)
  pos <- 1
  for (idx in 1:q) {
    res[idx, ] <- func(x[idx:(idx + b - 1)],...)
  }        
  return(res)
} 




