## 
# ------------------------------------------------------------------------
# 
# "ftrunc(x, atom_f, m, M trunc)" --
# 
# Computes the mean size of the blocks, the mean , median, truncated and winzorized mean based on blocks.
# 
# ------------------------------------------------------------------------
##
#' @aliases ftrunc
#' @title Robust Estimators of the Mean Based on Regeneration Blocks.
#' @description
#' The function calculates various statistics (mean, median, truncated, and Winsorized mean)
#' based on regeneration blocks obtained from the data. It also computes the mean block size. 
#' It relies on block-based calculations for robust statistics by eliminating either too large blocks
#' or too large values of the mean on a given block.
#' @param x A vector or time series.
#' @param atom_f A numeric value specifying an atom of the Markov chain.
#' @param m A numeric value; the lower truncation threshold 
#' Default is the 5th percentile of \code{x}.
#' @param M A numeric value; the upper truncation threshold 
#' Default is the 95th percentile of \code{x}.
#' @param trunc A numeric value specifying the truncation threshold for computing the 
#' truncated and Winsorized means of the block length.
#' @details This function uses blocks obtained from the input data \code{x} to compute 
#' several descriptive statistics, including the mean size of blocks, 
#' overall mean, median, and robust estimates like truncated and Winsorized means. 
#' The function internally uses \code{\link{GetBlocks}} to divide the input data 
#' into regenerative blocks when the process hits the atom \code{atom_f}.
#' 
#' The parameters \code{m} and \code{M} represent the lower and upper truncation 
#' thresholds, respectively. By default, these are set to the 5th and 95th percentiles 
#' of the input data, but they can be manually adjusted by the user to perform customized truncation.
#' 
#' The parameter \code{trunc} is used to eliminate blocks which lengths are greater than 
#' \code{trunc}.
#' 
#' @return A numeric vector containing the following elements:
#' 
#'   * \code{"taumean"} - the mean block size,
#'   * \code{"fmean"} - the mean of the input data,
#'   * \code{"fmed"} - the median of the input data,
#'   * \code{"tmean"} - the truncated mean of the blocks based on the truncation threshold,
#'   * \code{"wmean"} - the Winsorized mean of the blocks based on the truncation threshold.
#' @references Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Bertail, P., Clémençon, S. and  Tressou, J. (2015). Bootstrapping Robust Statistics 
#' for Markovian Data Applications to Regenerative R‐Statistics and L‐Statistics. 
#' \emph{Journal of Time Series Analysis}, \bold{36}, 462–480. 
#' 
#' @seealso \code{\link{GetBlocks}}, \code{\link{findBestEpsilon}}, \code{\link{GetPseudoBlocks}}, 
#' \code{\link{smallEnsemble}},
#' \code{\link{regenboot}}.
#' @keywords "Regenerative Block Bootstrap" "Markov chains" "Robust statistics"
#' @export
#' @examples 
#' n=500 # the length of the process
#' lambda=0.6 # arrival rate
#' mu=0.8 # departure rat
#' X = genMM1(n,lambda,mu) # generate MM1 queue
#' atom = 0 # specify the atom
#' trunc = 30 # set truncation threshold
#' result = ftrunc(x=X, atom_f=atom, m=0, trunc = trunc) # apply function
#' print(result)
#'


ftrunc<-function(x,atom_f,m=quantile(x,0.05),M=quantile(x,0.95),trunc){
  if (!is.numeric(x)) {
    stop("Error: 'x' must be a numeric vector.")
  }
  # Check if 'trun' is numeric and not empty
  if (!is.numeric(trunc) || length(trunc) != 1) {
    stop("Error: 'trunc' must be a single numeric value.")
  }
  
  n=length(x)
  GB=GetBlocks(x, atom_f,0)
  Blocks=GB[[2]]
  l_n=GB[[3]]
  
  taumean=n/l_n
  fmed=quantile(x,0.5)
  fmean=mean(x)
  tmean=sum(Blocks[(Blocks[,2]<=trunc),3])/sum(Blocks[(Blocks[,2]<=trunc),4])
  wmean=sum(Blocks[(Blocks[,2]<=trunc),5])/sum(Blocks[(Blocks[,2]<=trunc),2])
  c(taumean,fmean,fmed,tmean,wmean)
}

