
# ------------------------------------------------------------------------
# 
# "jackVar(x, func, ...)" --
# 
# Jackknife estimator for the variance of function 'func'.
# 
# ------------------------------------------------------------------------
##
#' @aliases jackVar
#' @title Jackknife Variance Estimator
#' @description 
#' Estimates the variance of a statistic using the jackknife-variance procedure in the i.i.d case.
#' @param x A vector or a matrix representing the data.
#' @param func The function used to compute the statistic on each sample.
#' @param ...  Optional additional arguments for the \code{func} function.
#' @details 
#' When \code{x} is a vector of length \emph{n} or a matrix with \emph{n} rows, 
#' the function \code{func}, having output size equal to \emph{p}, is applied to \code{x} with 
#' each \emph{i}-th row removed, 
#' resulting in \deqn{T_{n-1}^{i} = func(x[-i]).} The jackknife variance is computed 
#' based on these recalculated statistics and the original statistic \deqn{T_n = func(x).} 
#' The covariance matrix is calculated according to the jackknife formula.
#' 
#' This method is used to estimate the variance of a statistic that is potentially 
#' biased due to the finite sample size.
#' 
#' @return 
#' Returns a scalar or a covariance matrix, depending on whether the function \code{func} 
#' is univariate or multivariate. For a function returning a vector of length
#'  \emph{p}, the output will be a covariance matrix of size \emph{p x p}.
#' 
#' @references 
#' Bertail, P. and Dudek, A. (2025). \emph{Bootstrap for 
#' Dependent Data, with an R package} (by Bernard Desgraupes and Karolina Marek) - submitted.
#' 
#' Efron, B. (1979). Bootstrap methods: another look at the jackknife. 
#' \emph{Ann. Statist.}, \bold{7}, 1-26.
#' 
#' Gray, H., Schucany, W. and Watkins, T. (1972). \emph{The Generalized Jackknife Statistics}. 
#' Marcel Dekker, New York.
#' 
#' Quenouille, M.H. (1949). Approximate tests of correlation in time-series. 
#' \emph{J. Roy. Statist. Soc., Ser. B}, \bold{11}, 68-84.
#' 
#' @seealso 
#' \code{\link{jackFunc}}, 
#' \code{\link{boots}},
#' \code{\link{jackVarBlock}}, 
#' \code{\link{jackFuncBlock}}, 
#' \code{\link{jackFuncRegen}}.
#' 
#' @keywords 
#' Jackknife Resampling "Statistical analysis" "Variance estimation"
#' 
#' @export
#' @examples 
#' set.seed(1)
#' x <- rnorm(101)
#' func <- function(x) { mean(x^2) }
#' jackVar(x, func)
#' # Function returning a vector with the mean and standard deviation of x
#' mfunc <- function(x) { c(mean(x), sd(x)) }
#' jackVar(x, mfunc)
#' # Function to compute the moment of order p with p as additional argument
#' funca <- function(x, p) { mean((x-mean(x))^p)}
#' jackVar(x, funca, 3)
##
##
jackVar <- function(x,func,...) {
  # Test the value returned by func
  Tn <- func(x,...)
  if (!is.vector(Tn)) {
    stop("Function 'func' must return a vector")
  }
  p <- length(Tn)
  Tnames <- names(Tn)
  if (is.matrix(x)) {
    n <- nrow(x)
  } else {
    n <- length(x)
  }
  if (n <= p) {stop("length of x must be greater than length of func(x)")}
  J <- matrix(nrow=n,ncol=p)
  for (i in 1:n) {
    if (is.matrix(x)) {
      Tni <- func(x[-i,],...)
    } else {
      Tni <- func(x[-i],...)
    }
    J[i,] <- n*Tn - (n-1)*Tni
  }
  muJ <- colMeans(J)
  V <- matrix(0,nrow=p,ncol=p)
  for (i in 1:n) {
    L <- J[i,]-muJ
    V <- V + L%*%t(L)
  }
  V <- V/(n*(n-p))
  if (!is.null(Tnames)) {
    rownames(V) <- Tnames
    colnames(V) <- Tnames
  }
  if (p == 1) {V <- as.vector(V)}
  return(V)
}