
#' Plot compensation analysis
#' 
#' Plots the effect of varying compensation from 0 to 1 for each cell subclass,
#' examining the minimum subclass output result following a call to
#' [deconvolute()]. For this function to work, the argument `plot_comp` must be
#' set to `TRUE` during the call to [deconvolute()].
#' 
#' @param x An object of class 'deconv' generated by [deconvolute()].
#' @param overlay Logical whether to overlay compensation curves onto a single
#'   plot.
#' @param mfrow Optional vector of length 2 for organising plot layout. See
#'   `par()`. Only used when `overlay = FALSE`.
#' @param ... Optional graphical arguments passed to [plot()].
#' @return No return value, plots the effect of varying compensation on minimum
#'   subclass output for each cell subclass.
#' @importFrom graphics text
#' @export

plot_comp <- function(x, overlay = TRUE, mfrow = NULL, ...) {
  if (!inherits(x, "deconv")) stop("not a 'deconv' class object")
  ox <- x
  x <- x$comp_check
  if (is.null(x)) stop("missing comp_check analysis")
  
  n <- length(x) -1
  px <- x$px
  x <- x[1:n]
  new.args <- list(...)
  
  if (overlay) {
    yr <- range(x)
    scheme <- hue_pal(h = c(0, 270), c = 120)(n)
    args <- list(x = NA, las = 1, xlim = c(0, 1), ylim = yr,
                 xlab = "Compensation", ylab = "min output")
    if (length(new.args)) args[names(new.args)] <- new.args
    do.call(plot, args)
    abline(h = 0)
    for (i in seq_len(n)) {
      lines(px, x[[i]], col = scheme[i])
    }
  } else {
    nr1 <- ceiling(sqrt(n))
    nr2 <- ceiling(n / nr1)
    if (is.null(mfrow)) mfrow <- c(nr1, nr2)
    op <- par(bty = "l", mgp = c(2.2, 0.6, 0), tcl = -0.3,
              mar = c(3.7, 4.5, 1.5, 1.1), mfrow = mfrow)
    on.exit(par(op))
    for (i in seq_len(n)) {
      ylim <- range(x[[i]])
      if (ylim[1] > 0) ylim[1] <- 0
      if (ylim[2] < 0) ylim[2] <- 0
      plot(px, x[[i]], type = "l", ylim = ylim, las = 1,
           xlab = paste(names(x)[i], "comp"), ylab = "")
      abline(h = 0, col = "darkgrey")
      comp <- ox$subclass$comp_amount[i]
      abline(v = comp, col = "red", lty = 2)
      text(comp, ylim[2], format(comp, digits = 2, nsmall = 1),
           col = "red", adj = c(0.5, -0.5), xpd = NA)
    }
  }
}
