% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/debugr.r
\name{dwatch}
\alias{dwatch}
\title{Printing debug outputs during runtime}
\usage{
dwatch(crit = "", objs = NULL, funs = NULL, args = NULL,
  show.all = FALSE, expr = NULL, msg = "", halt = FALSE,
  unique.id = "", suppress.source = FALSE, show.frame = TRUE,
  filename = "")
}
\arguments{
\item{crit}{An string containing an expression that determines if any debug
outputs shall be displayed at all. Only, if \code{crit} evaluates to
\code{TRUE}, a debug output will be shown.}

\item{objs}{A vector of object names (as strings). The values of these
objects will be displayed in the debug output.}

\item{funs}{A vector of function names (as strings) that shall be applied to
the objects in \code{objs}, one function per object. \code{funs} must have
the same length as \code{objs}. If no function shall be applied to an
object, the respective element in the \code{funs} vector must be
\code{NULL}. The functions in \code{funs} must undertake the task of
printing the object.}

\item{args}{A list of vectors containing additional arguments for the
functions in \code{funs}. It is assumed that the first argument of each
function in \code{funs} is the respective object from \code{objs}.
Additional arguments can then be supplied with \code{args}. The \code{args}
list must have the same number of elements as \code{funs}. If a function
does not receive any additional arguments, the respective element in the
\code{args} list must be \code{NULL}. Each element of \code{args} is a
vector of named elements. The element name is the name of the additions
argument to the respective \code{funs} function, the elements value is the
argument's value.}

\item{show.all}{Prints all objects from the (calling) environment. If set to
\code{TRUE}, \code{objs} is ignored and all objects in the enviroment (with
the exception of functions) are included in the debug output.}

\item{expr}{A vector of strings containing expressions to be evaluated and
displayed in the debug output. This output comes on top of any \code{msg}
or \code{objs} output.}

\item{msg}{A string containing a general message to be displayed.}

\item{halt}{If \code{TRUE}, the execution of the debugged R script is stopped
after printing the output.}

\item{unique.id}{A unqiue string ID that can be chosen by the user. This ID
is displayed in the debug output and is used to identify the code section
that contains the \code{dwatch} call. By default, when a debug output is
displayed, \code{dwatch} tries to show an extract from the code thats
surrounds the \code{dwatch} call (this feature can be turned off by setting
\code{suppress.source} to \code{TRUE}).}

\item{suppress.source}{If \code{TRUE} (default), \code{dwatch} tries to find
the code section that includes the \code{dwatch} call and displays it as
part of the debug output. Requires \code{unique.id} to be set.}

\item{show.frame}{If \code{TRUE} (default), a frame is displayed at the top
and the bottom of the debug output.}

\item{filename}{If a filename is provided, all debug message are only printed
to the file and not shown on the R console.}
}
\description{
Prints a debug output to the console or to a file. A debug output can consist
of a static text message, the values of one or more objects (potentially
transformed by applying some functions) or the value of one or multiple (more
complex) R expressions. Whether or not a debug message is displayed can be
made dependent on the evaluation of a criterion phrased as an R expression.
Generally, debug messages are only shown if the debug mode is activated. The
debug mode is activated and deactivated with \code{\link{debugr_switchOn}}
and \code{\link{debugr_switchOff}}, respectively, which change the logical
\code{debugr.active} value in the global options. Since debug messages are
only displayed in debug mode, the \code{dwatch} function calls can even
remain in the original code as they remain silent and won't have any effect
until the debug mode is switched on again.
}
\examples{

library(debugr)

# --- A simple example to print the value of an object
myfunction <- function(x) {
  justastring <- "Not much information here"
  z <- 1

  for(i in 1:x) {
    # This call can remain in your code; it is only activated when
    # the debug mode is switched on
    dwatch(crit = "z > 40000", objs = c("z"))
    z <- z * i
  }
  invisible(z)
}

# Turn debug mode on
debugr_switchOn()

# Call function for debugging
myfunction(10)



# --- Applying a function to the object that is printed
myfunction <- function(x) {
  justastring <- "Not much information here"
  z <- 1

  for(i in 1:x) {
    dwatch(crit = "z > 40000", objs = c("z"), funs=c("format"),
    args = as.list(c(big.mark = "\\",\\"")))
    z <- z * i
  }
  invisible(z)
}

myfunction(10)



# --- Same thing, this time with a expression
myfunction <- function(x) {
  justastring <- "Not much information here"
  z <- 1

  for(i in 1:x) {
    dwatch(crit = "z > 40000", expr=c("format(z, big.mark = \\",\\")"))
    z <- z * i
  }
  invisible(z)
}

myfunction(10)

}
\seealso{
debugr_switchOn, debugr_switchOff, debugr_isActive
}
