% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/fit_distributions.R
\name{fit_dist}
\alias{fit_dist}
\title{Unified Parameter Estimation for Probability Distributions}
\usage{
fit_dist(
  data,
  dist = "normal",
  method = "mle",
  start = NULL,
  lower = NULL,
  upper = NULL,
  optim_method = "Nelder-Mead",
  custom_functions = NULL,
  tol_spacing = 1e-16,
  tol_param = 1e-06,
  ties_method = "cheng_amin",
  ...
)
}
\arguments{
\item{data}{numeric vector of observed data.}

\item{dist}{character string specifying the distribution. Options include:
"normal", "exponential", "gamma", "weibull", "weibull3", "lognormal",
"lognormal3", "beta", or "custom".}

\item{method}{character string specifying estimation method. Options:
"mle" (Maximum Likelihood - default), "mps" (Maximum Product Spacing),
"mom" (Method of Moments).}

\item{start}{named list or numeric vector of initial parameter values.
Required for "custom" distributions with "mps" or "mle" methods.}

\item{lower}{named list or numeric vector of lower bounds for parameters.}

\item{upper}{named list or numeric vector of upper bounds for parameters.}

\item{optim_method}{optimization method passed to \code{\link{optim}}. Default is "Nelder-Mead".}

\item{custom_functions}{list containing custom distribution functions (for dist="custom"):
\itemize{
  \item \code{pdf}: probability density function f(x, params) (for MLE and ties correction)
  \item \code{cdf}: cumulative distribution function F(x, params) (for MPS)
  \item \code{param_names}: character vector of parameter names
  \item \code{start_fn}: function to compute starting values (optional)
}}

\item{tol_spacing}{numeric tolerance for spacings/densities to avoid log(0). Default is 1e-16.}

\item{tol_param}{numeric tolerance for parameter lower bounds. Default is 1e-6.}

\item{ties_method}{character string for handling ties in MPS. Options:
"cheng_amin" (default - Cheng & Amin 1983), "none", "cheng_stephens" (Cheng & Stephens 1989).
Only applicable when method = "mps".}

\item{...}{Additional arguments passed to \code{\link{optim}}.}
}
\value{
A list with class "fit_dist" containing:
\item{estimate}{Named vector of parameter estimates}
\item{vcov}{Variance-covariance matrix (for mps/mle)}
\item{se}{Standard errors (for mps/mle)}
\item{loglik}{Log-likelihood value}
\item{aic}{Akaike Information Criterion}
\item{bic}{Bayesian Information Criterion}
\item{objective}{Maximum value of objective function (logspacing for mps, loglik for mle)}
\item{ks_statistic}{Kolmogorov-Smirnov test statistic}
\item{ks_pvalue}{Kolmogorov-Smirnov test p-value}
\item{convergence}{Convergence code from optim (0 indicates success)}
\item{message}{Convergence message from optim}
\item{data}{Original data (sorted)}
\item{dist}{Distribution name}
\item{method}{Estimation method used}
\item{n}{Sample size}
\item{k}{Number of parameters}
\item{tol_spacing}{Tolerance used for spacings/densities}
\item{tol_param}{Tolerance used for parameter bounds}
\item{ties_method}{Ties correction method (for MPS only)}
}
\description{
Estimates parameters of probability distributions using various methods:
Maximum Likelihood (MLE), Maximum Product Spacing (MPS),
or Method of Moments (MOM).
}
\examples{
# MLE estimation (default)
set.seed(123)
x <- rweibull(100, shape = 2.5, scale = 1.5)
fit1 <- fit_dist(x, dist = "weibull")
print(fit1)

# MPS estimation with Cheng-Amin ties correction
fit2 <- fit_dist(x, dist = "weibull", method = "mps")

# L-Moments estimation (under development)
# fit3 <- fit_dist(x, dist = "weibull", method = "lm")

# Method of Moments
fit4 <- fit_dist(x, dist = "weibull", method = "mom")

# Compare fits
cat("AIC - MLE:", fit1$aic, "MPS:", fit2$aic, "\n")

}
\references{
Cheng, R. C. H., & Amin, N. A. K. (1983). Estimating parameters in continuous
univariate distributions with a shifted origin. Journal of the Royal Statistical
Society: Series B, 45(3), 394-403.

Cheng, R. C. H., & Stephens, M. A. (1989). A goodness-of-fit test using Moran's
statistic with estimated parameters. Biometrika, 76(2), 385-392.
}
