#' Build a Dynamic Linear Model (DLM) for clniical modeling by using wastewater data
#'
#' Constructs a dynamic linear model (DLM) object using the dlm package.
#'
#'
#' @param formula an object of class "formula" describing the model to be fitted.
#' @param lags a nonnegative integer indicating the lag of latent state in the model.
#' @param prior an optional list of the prior mean vector and covariance structure of the latent state.
#' if not provided, a naive prior is considered.
#' @param equal.state.var logical; if TRUE the variance is the same across all wastewater state.
#' @param equal.obs.var logical; if TRUE the variance is the same across all wastewater observation.
#' @param params a vector of the parameters of the model.
#'
#' @return A DLM object (from \code{dlm::dlm}) with an additional attribute \code{"model_info"}.
#' @noRd




build_pdlm <- function(params,
                      lags,
                      equal.state.var,
                      equal.obs.var,
                      nW,
                      nV,
                      k,
                      prior){

  n.coef <- 1+(lags+1)*2
  Ft <- getFF(params[1:n.coef],k,lags)
  Gt <- getGG(k,lags)
  Wt <- getW(exp(params[n.coef+1:nW]), k, lags)
  Vt <- getV(exp(params[n.coef+nW+1:2]), k)
  # Construct the DLM using the dlm package.
  mod <- dlm::dlm(
    m0 = prior$m0,
    C0 = prior$C0,
    FF = Ft,
    GG = Gt,
    V = Vt,
    W = Wt)
  # Attach additional model info as an attribute.
  attr(mod, "model_info") <- list(
    lags=lags,
    equal.state.var=equal.state.var,
    equal.obs.var=equal.obs.var)
  return(mod)
}

