#' KR-20
#'
#' @description
#' Compute the KR-20 reliability coefficient for
#' dichotomously scored items (e.g., 0/1).
#'
#' @param x A data frame or matrix of item responses, with rows as persons
#'   and columns as items. Items are assumed to be dichotomous (0/1).
#'
#' @details
#' KR-20 is an internal consistency reliability estimate for tests with
#' dichotomously scored items.
#' Rows containing missing values are removed using \code{stats::na.exclude()}.
#'
#' @return A single numeric value: the KR-20 reliability coefficient.
#'
#' @examples
#' data(data.u)
#' kr20(data.u)
#'
#' @export
kr20 <- function(x) {

  if (is.null(x)) stop("`x` must not be NULL.")
  if (!is.data.frame(x) && !is.matrix(x)) {
    stop("`x` must be a data frame or a matrix.")
  }

  x <- stats::na.exclude(x)
  x <- as.matrix(x)

  if (!is.numeric(x)) stop("`x` must contain only numeric values.")

  ni <- ncol(x)
  np <- nrow(x)

  if (ni < 2L) stop("`x` must have at least 2 items.")
  if (np < 2L) stop("`x` must have at least 2 persons.")

  total_scores <- rowSums(x)
  varsum <- stats::var(total_scores)

  if (isTRUE(all.equal(varsum, 0))) {
    stop("Total score variance is zero; KR-20 is undefined.")
  }

  p <- colSums(x) / np
  pqsum <- sum(p * (1 - p))

  kr20 <- ni / (ni - 1) * (1 - pqsum / varsum)

  return(kr20)
}
