#' Stratified Cronbach's Coefficient Alpha
#'
#' @description
#' Compute the stratified Cronbach's coefficient alpha for a test composed of
#' several item strata (e.g., subtests or subscales).
#'
#' @param x A data frame or matrix containing item responses, with rows as
#'   subjects and columns as items. Items are assumed to be ordered by stratum.
#' @param s A numeric vector giving the number of items in each stratum. The
#'   sum of \code{s} must equal \code{ncol(x)}.
#'
#' @details
#' Stratified alpha is an estimate of the internal consistency reliability of a
#' composite test formed by multiple item strata (e.g., subtests). Each stratum
#' reliability is computed using \code{alpha()}, and combined using the
#' classical stratified-alpha formula.
#'
#' @return A named list with:
#' \describe{
#'   \item{stratified_alpha}{Stratified Cronbach's coefficient alpha.}
#' }
#'
#' @examples
#' data(data.m)
#' stratified_alpha(data.m, c(13, 12, 6))
#'
#' @export
stratified_alpha <- function(x, s) {
  # basic checks ---------------------------------------------------------------
  if (is.null(x)) {
    stop("`x` must not be NULL.")
  }
  if (!is.data.frame(x) && !is.matrix(x)) {
    stop("`x` must be a data frame or a matrix.")
  }

  x <- as.matrix(x)
  if (!is.numeric(x)) {
    stop("`x` must contain only numeric values.")
  }

  if (missing(s)) {
    stop("`s` must be supplied as a numeric vector giving items per stratum.")
  }
  if (!is.numeric(s)) {
    stop("`s` must be numeric.")
  }
  if (any(s <= 0L)) {
    stop("All elements of `s` must be positive.")
  }

  # number of strata -----------------------------------------------------------
  ns <- length(s)
  if (ns < 2L) {
    stop("`s` must define at least 2 strata.")
  }

  if (sum(s) != ncol(x)) {
    stop("The sum of `s` must equal the number of columns in `x`.")
  }

  # handle missing values consistently -----------------------------------------
  x <- stats::na.exclude(x)

  # vectors of reliabilities and variances -------------------------------------
  ri <- numeric(ns)
  vi <- numeric(ns)

  start <- 1L
  for (k in seq_len(ns)) {
    end <- start + s[k] - 1L
    subx <- x[, start:end, drop = FALSE]

    # Cronbach's alpha for stratum k
    ri[k] <- alpha(subx)$alpha

    # variance of the stratum total score (sum of covariances)
    vi[k] <- sum(stats::cov(subx))

    start <- end + 1L
  }

  # total variance of the composite score --------------------------------------
  vt <- sum(stats::cov(x))
  if (isTRUE(all.equal(vt, 0))) {
    stop("Total score variance is zero; stratified alpha is undefined.")
  }

  # stratified alpha -----------------------------------------------------------
  stratifiedalpha <- 1 - sum(vi * (1 - ri)) / vt

  return(list(stratified_alpha = stratifiedalpha))
}
