% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/get_obj_name.r
\name{get_obj_name}
\alias{get_obj_name}
\title{Return the name of an object at a given parent generation from an environment}
\usage{
get_obj_name(obj, n = 0, eval = FALSE, silent = TRUE)
}
\arguments{
\item{obj}{object whose name at a given parent generation is of interest.}

\item{n}{number of parent generations to go back from the calling environment
to retrieve the name of the object that leads to \code{obj}
in the function calling chain. See details for more information.}

\item{eval}{whether to evaluate \code{obj} in the \code{n}-th parent generation before
getting the object's name in that environment. See details for more information.}

\item{silent}{when \code{FALSE}, the names of the environments and objects in those environments are printed
as those environments are traversed by this function.}
}
\value{
The name of the object in the \code{n}-th parent generation environment.
}
\description{
A practical use of this function is to retrieve the name of the object leading to
a function's parameter in the function calling chain, at any parent generation.
}
\details{
In particular, it provides a handy way of retrieving the name of a function's parameter
and use it in e.g. messages to the user describing the arguments received by the function.
In this context, it is a shortcut to calling \code{as.list(environment())}, which returns
a list of parameter names and parameter values.
See the Examples section for an illustration.


This function goes back to each parent generation from the calling function's environment
and at each of those parent generations it retrieves the name of the object that is part of
the parameter chain leading to the calling function's parameter.

To illustrate: suppose we call a function \code{f <- function(x)} by running the piece of code \code{f(z)},
and that \code{f} calls another function \code{g <- function(y)} by running the piece of code \code{g(x)}.  

That is, we have the parameter chain:  
\code{z -> x -> y}

If, inside function \code{g()}, we call \code{get_obj_name()} as follows, we obtain respectively:  
\code{get_obj_name(y, n=1)} yields \code{"x"}
\code{get_obj_name(y, n=2)} yields \code{"z"}

because these calls are telling "give me the name of object \code{y} as it was called
\code{n} levels up from the calling environment --i.e. from the environment of \code{g()}.

Note that the results of these two calls are different from making the following two
\code{deparse(substitute())} calls:  
\code{deparse(substitute(y, parent.frame(n=1)))}   
\code{deparse(substitute(y, parent.frame(n=2)))}
because these calls simply \code{substitute} or evaluate \code{y} at the \code{n}-th parent generation.
If \code{y} is not defined at those parent generations, the \code{substitute()} calls return
simply \code{"y"}.

On the contrary, the previous two calls to \code{get_obj_name()} return the name of the object
in the parameter chain (\code{z -> x -> y}) \emph{leading} to \code{y}, which is a quite different
piece of information.

When eval=TRUE, the result is the same as the result of \code{deparse()}
except for the following three cases:
\itemize{
\item if the object passed to \code{get_obj_name()} evaluates to a name, it returns that name,
without any added quotes. For example, if \code{v = "x"} then \code{get_obj_name(v, eval=TRUE)} returns \code{"x"}
while \code{deparse(v)} returns \code{"\"x\""}.
\item the result of \code{NULL} is \code{NULL} instead of \code{"NULL"} which is the case with \code{deparse()}.
\item the result of a non-existent object is \code{NULL}, while \code{deparse()} returns an error stating
that the object does not exist.
}

When \code{get_obj_name} operates on non-existent objects it works at follows:
\itemize{
\item when \code{eval=FALSE} it returns the name of the non-existent object
enclosed in quotes (e.g. \code{get_obj_name(nonexistent)} returns \code{"nonexistent"}, assuming \code{nonexistent}
does not exist).
\item when \code{eval=TRUE} it returns NULL.
}

Finally \code{get_obj_name(NULL)} returns \code{NULL}, while \code{as.character(NULL)} returns \code{as.character(0)}.
}
\examples{
# Example 1:
# This example shows the difference between using get_obj_name() and deparse(substitute())
g <- function(y) { return(list(obj_name=get_obj_name(y, n=2, silent=FALSE), 
                               substitute=deparse(substitute(y, parent.frame(n=2))) )) }
f <- function(x) { g(x) }
z = 3; 
f(z)           # After showing the names of objects as they
               # are traversed in the parameter chain (silent=FALSE),
               # this function returns a list where
               # the first element (result of get_obj_name()) is "z"
               # and the second element (result of deparse(substitute())) is "y".
               # Note that 'z' is the object leading to object 'y'
               # inside function g() if we follow the parameter names
               # leading to 'y' in the function calling chain.

# Example 2:
# When eval=TRUE, get_obj_name() behaves the same way as deparse()
# (except for the cases noted in the Details section)
# because the values of all objects linked by the parameter chain
# are ALL the same.
g <- function(y) { return(list(obj_name=get_obj_name(y, n=2, eval=TRUE),
                               deparse=deparse(y))) }
f <- function(x) { g(x) }
z = 3 
f(z)           # Returns a list where both elements are equal to "3"
               # because the output of get_obj_name() with eval=TRUE
               # and deparse() are the same.

# Example 3:
# This example shows how we can use get_obj_name() to get the parameter names
# of non '...' parameters, which are then used in messages to the user.
# The advantage of using get_obj_name() as opposed to the hard-coded parameter name
# is that an error is raised if the parameter does not exist.
# An example is also shown that uses as.list(environment()), which clearly is more
# general... get_obj_name() should be used when referring to a couple of specific
# parameters.
f <- function(x, y, ...) {
  cat("Arguments received by the function (using get_obj_name()) (explicit listing):\n")
  cat(get_obj_name(x), ":", x, "\n")
  cat(get_obj_name(y), ":", y, "\n")
  cat("Arguments received by the function (using as.list(environment())) (automatic listing):\n")
  paramsList = as.list(environment())
  paramsNames = names(paramsList)
  sapply(paramsNames, get_obj_name)
  for (p in paramsNames) {
    cat(p, ":", paramsList[[p]], "\n")
  }
}
z = 5
extra_param = "a '...' parameter" 
  ## Note: this exra parameter is NOT shown neither by get_obj_name()
  ## nor by as.list(environment())
f("test", z, extra_param)
}
\seealso{
\link{get_obj_value}
}
