% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/confounding.R
\name{confounders}
\alias{confounders}
\alias{confounders_emm}
\alias{confounders_poly}
\alias{probsens_conf}
\title{Uncontrolled confounding}
\usage{
confounders(
  case,
  exposed,
  type = c("RR", "OR", "RD"),
  bias_parms = NULL,
  alpha = 0.05
)

confounders_emm(
  case,
  exposed,
  type = c("RR", "OR", "RD"),
  bias_parms = NULL,
  alpha = 0.05
)

confounders_poly(
  case,
  exposed,
  type = c("RR", "OR", "RD"),
  bias_parms = NULL,
  alpha = 0.05
)

probsens_conf(
  case,
  exposed,
  reps = 1000,
  prev_exp = list(dist = c("constant", "uniform", "triangular", "trapezoidal", "normal",
    "beta"), parms = NULL),
  prev_nexp = list(dist = c("constant", "uniform", "triangular", "trapezoidal", "normal",
    "beta"), parms = NULL),
  risk = list(dist = c("constant", "uniform", "triangular", "trapezoidal",
    "log-logistic", "log-normal"), parms = NULL),
  corr_p = NULL,
  alpha = 0.05
)
}
\arguments{
\item{case}{Outcome variable. If a variable, this variable is tabulated against.}

\item{exposed}{Exposure variable.}

\item{type}{Choice of implementation, with no effect measure modification for
ratio measures (relative risk -- RR; odds ratio -- OR) or difference measures
(risk difference -- RD).}

\item{bias_parms}{Numeric vector defining the 3, 4, or 6 necessary bias parameters.
\itemize{
\item This vector has 3 elements for the \code{confounders()} function, in the
following order:
\enumerate{
\item the association between the confounder and the outcome among those who
were not exposed (RR, OR, or RD according to choice of implementation),
\item the prevalence of the confounder among the exposed (between 0 and 1), and
\item the prevalence of the confounder among the unexposed (between 0 and 1).
}
\item This vector has 4 elements for the \code{confounders_emm()} function, in the
following order:
\enumerate{
\item the association between the confounder and the outcome among those who
were exposed,
\item the association between the confounder and the outcome among those who
were not exposed,
\item the prevalence of the confounder among the exposed (between 0 and 1), and
\item the prevalence of the confounder among the unexposed (between 0 and 1).
}
\item This vector has 6 elements for the \code{confounders_poly()} function, in the
following order:
\enumerate{
\item the association between the highest level confounder and the outcome,
\item the association between the mid-level confounder and the outcome,
\item the prevalence of the highest level confounder among the exposed (between 0 and 1),
\item the prevalence of the highest level confounder among the unexposed (between 0 and 1),
\item the prevalence of the mid-level confounder among the exposed (between 0 and 1), and
\item the prevalence of the mid-level confounder among the unexposed (between 0 and 1).
}
}}

\item{alpha}{Significance level.}

\item{reps}{Number of replications to run.}

\item{prev_exp}{List defining the prevalence of exposure among the exposed.
The first argument provides the probability distribution function (constant,
uniform, triangular, trapezoidal, truncated normal, or beta) and the second
its parameters as a vector. Lower bound of the truncated normal cannot be
less than zero. Upper bound is Inf by default.
\enumerate{
\item constant: constant value,
\item uniform: min, max,
\item triangular: lower limit, upper limit, mode,
\item trapezoidal: min, lower mode, upper mode, max.
\item normal: lower bound, upper bound, mean, sd.
\item beta: alpha, beta.
}}

\item{prev_nexp}{List defining the prevalence of exposure among the unexposed.}

\item{risk}{List defining the confounder-disease relative risk or the
confounder-exposure odds ratio. The first argument provides the probability
distribution function (constant, uniform, triangular, trapezoidal,
log-logistic, or log-normal) and the second its parameters as a vector:
\enumerate{
\item constant: constant value,
\item uniform: min, max,
\item triangular: lower limit, upper limit, mode,
\item trapezoidal: min, lower mode, upper mode, max.
\item log-logistic: shape, rate. Must be strictly positive,
\item log-normal: meanlog, sdlog. This is the mean and standard deviation on the log scale.
}}

\item{corr_p}{Correlation between the exposure-specific confounder prevalences.}
}
\value{
A list with elements:
\item{obs_data}{The analyzed 2 x 2 table from the observed data.}
\item{cfder_data}{The same table for Confounder +.}
\item{cfder1.data}{The same table for Mid-level Confounder + (for \code{confounders_poly()}).}
\item{cfder2.data}{The same table for Highest-level Confounder + (for \code{confounders_poly()}).}
\item{nocfder_data}{The same table for Confounder -.}
\item{obs_measures}{A table of relative risk with confidence intervals; for
Total, Confounder +, and Confounder -.}
\item{adj_measures}{A table of Standardized Morbidity Ratio and Mantel-Haenszel
estimates.}
\item{bias_parms}{Input bias parameters.}

A list with elements (for \code{probsens_conf()}):
\item{obs_data}{The analyzed 2 x 2 table from the observed data.}
\item{obs_measures}{A table of observed relative risk and odds ratio with confidence intervals.}
\item{adj_measures}{A table of corrected relative risks and odds ratios.}
\item{sim_df}{Data frame of random parameters and computed values.}
\item{reps}{Number of replications.}
}
\description{
\code{confounders()} and \code{probsens_conf()} allow to provide adjusted measures of
association corrected for unknown or unmeasured confounding without effect
modification.
}
\details{
\code{confounders_emm()} allows to provide for adjusted measures of association
corrected for unknown or unmeasured confounding in the presence of effect
modification.

\code{confounders_poly()} allows to provide for adjusted measures of association
corrected for unknown or unmeasured polychotomous (3-level) confounding without
effect modification.
}
\section{Simple bias analysis with \code{confounders()}}{

\code{confounders()} allows you to run a simple sensitivity analysis to correct for
unknown or unmeasured confounding without effect modification. Implementation
for ratio measures (relative risk -- RR, or odds ratio -- OR) and difference
measures (risk difference -- RD).

The analytic approach uses the "relative risk due to confounding" as defined by
Miettinen (1972), i.e. \eqn{RR_{adj} = \frac{RR_{crude}}{RR_{conf}}} where
\eqn{RR_{adj}} is the standardized (adjusted) risk ratio, \eqn{RR_{crude}} is
the crude risk ratio, and \eqn{RR_{conf}} is the relative risk component
attributable to confounding by the stratification factors. The output provides
both \eqn{RR_{adj}} (SMR or Mantel-Haenszel) and the \eqn{RR_{conf}} (i.e., RR,
OR or RD due to confounding from the unmeasured confounder).
}

\section{Probabilistic sensitivity analysis with \code{probsens_conf()}}{

\code{probsens_conf()} performs a summary-level probabilistic sensitivity analysis
to correct for unknown or unmeasured confounding and random error simultaneously.
It returns the Mantel-Haenszel risk ratio.

Correlations between prevalences of exposure classification among cases and
controls can be specified and use the NORmal To Anything (NORTA)
transformation (Li & Hammond, 1975).
}

\section{Simple bias analysis with \code{confounders_emm()}}{

\code{confounders_emm()} allows you to run a simple sensitivity analysis to correct
for unknown or unmeasured confounding in the presence of effect modification.
Implementation for ratio measures (relative risk -- RR, or odds ratio -- OR)
and difference measures (risk difference -- RD).
}

\section{Simple bias analysis with \code{confounders_poly()}}{

\code{confounders_poly()} allows you to run a simple sensitivity analysis to correct
for unknown or unmeasured polychotomous (3-level) confounding without effect
modification. Implementation for ratio measures (relative risk -- RR, or odds
ratio -- OR) and difference measures (risk difference -- RD).
}

\section{Updated calculations}{

episensr 2.0.0 introduced updated calculations of probabilistic bias analyses
by (1) using the NORTA transformation to define a correlation between
distributions, and (2) sampling true prevalences and then sampling the
adjusted cell counts rather than just using the expected cell counts from a
simple quantitative bias analysis. This updated version should be preferred
but if you need to run an old analysis, you can easily revert to the
computation using \code{\link[=probsens.conf_legacy]{probsens.conf_legacy()}} as follows:

\if{html}{\out{<div class="sourceCode">}}\preformatted{library(episensr)
probsens_conf <- probsens.conf_legacy
}\if{html}{\out{</div>}}
}

\examples{
# The data for this example come from:
# Tyndall M.W., Ronald A.R., Agoki E., Malisa W., Bwayo J.J., Ndinya-Achola J.O.
# et al.
# Increased risk of infection with human immunodeficiency virus type 1 among
# uncircumcised men presenting with genital ulcer disease in Kenya.
# Clin Infect Dis 1996;23:449-53.
confounders(matrix(c(105, 85, 527, 93),
dimnames = list(c("HIV+", "HIV-"), c("Circ+", "Circ-")),
nrow = 2, byrow = TRUE),
type = "RR",
bias_parms = c(.63, .8, .05))

confounders(matrix(c(105, 85, 527, 93),
dimnames = list(c("HIV+", "HIV-"), c("Circ+", "Circ-")),
nrow = 2, byrow = TRUE),
type = "OR",
bias_parms = c(.63, .8, .05))

confounders(matrix(c(105, 85, 527, 93),
dimnames = list(c("HIV+", "HIV-"), c("Circ+", "Circ-")),
nrow = 2, byrow = TRUE),
type = "RD",
bias_parms = c(-.37, .8, .05))
#
confounders_emm(matrix(c(105, 85, 527, 93),
dimnames = list(c("HIV+", "HIV-"), c("Circ+", "Circ-")),
nrow = 2, byrow = TRUE),
type = "RR",
bias_parms = c(.4, .7, .8, .05))

confounders_emm(matrix(c(105, 85, 527, 93),
dimnames = list(c("HIV+", "HIV-"), c("Circ+", "Circ-")),
nrow = 2, byrow = TRUE),
type = "OR",
bias_parms = c(.4, .7, .8, .05))

confounders_emm(matrix(c(105, 85, 527, 93),
dimnames = list(c("HIV+", "HIV-"), c("Circ+", "Circ-")),
nrow = 2, byrow = TRUE),
type = "RD",
bias_parms = c(-.6, -.3, .8, .05))
#
# The data for this example come from:
# Tyndall M.W., Ronald A.R., Agoki E., Malisa W., Bwayo J.J., Ndinya-Achola J.O.
# et al.
# Increased risk of infection with human immunodeficiency virus type 1 among
# uncircumcised men presenting with genital ulcer disease in Kenya.
# Clin Infect Dis 1996;23:449-53.
confounders_poly(matrix(c(105, 85, 527, 93),
dimnames = list(c("HIV+", "HIV-"), c("Circ+", "Circ-")),
nrow = 2, byrow = TRUE),
type = "RR",
bias_parms = c(.4, .8, .6, .05, .2, .2))

confounders_poly(matrix(c(105, 85, 527, 93),
dimnames = list(c("HIV+", "HIV-"), c("Circ+", "Circ-")),
nrow = 2, byrow = TRUE),
type = "OR",
bias_parms = c(.4, .8, .6, .05, .2, .2))

confounders_poly(matrix(c(105, 85, 527, 93),
dimnames = list(c("HIV+", "HIV-"), c("Circ+", "Circ-")),
nrow = 2, byrow = TRUE),
type = "RD",
bias_parms = c(-.4, -.2, .6, .05, .2, .2))
#
# The data for this example come from:
# Tyndall M.W., Ronald A.R., Agoki E., Malisa W., Bwayo J.J., Ndinya-Achola J.O. et al.
# Increased risk of infection with human immunodeficiency virus type 1 among
# uncircumcised men presenting with genital ulcer disease in Kenya.
# Clin Infect Dis 1996;23:449-53.
tyndall <- matrix(c(105, 85, 527, 93),
dimnames = list(c("HIV+", "HIV-"), c("Circ+", "Circ-")), nrow = 2, byrow = TRUE)
set.seed(1234)
probsens_conf(tyndall, reps = 100000,
prev_exp = list("trapezoidal", c(.7, .75, .85, .9)),
prev_nexp = list("trapezoidal", c(.03, .04, .07, .1)),
risk = list("trapezoidal", c(.5, .6, .7, .8)))

set.seed(123)
probsens_conf(tyndall, reps = 20000,
prev_exp = list("beta", c(200, 56)),
prev_nexp = list("beta", c(10, 16)),
risk = list("triangular", c(.6, .7, .63)),
corr_p = .8)

set.seed(123)
probsens_conf(tyndall, reps = 20000,
prev_exp = list("normal", c(.01, .12, 0.03, 0.005)),
prev_nexp = list("normal", c(0, Inf, 0.01, 0.0001)),
risk = list("triangular", c(.6, .7, .63)), corr_p = .8)

# Fox M.P., MacLehose R.F., Lash T.L.
# SAS and R code for probabilistic quantitative bias analysis for
# misclassified binary variables and binary unmeasured confounders
# Int J Epidemiol 2023:1624-1633.
fox <- matrix(c(40, 20, 60, 80),
dimnames = list(c("Diseased", "Non-diseased"), c("Exposed", "Unexposed")),
nrow = 2, byrow = TRUE)
set.seed(1234)
probsens_conf(fox, reps = 10^5,
prev_exp = list("beta", c(10, 20)),
prev_nexp = list("beta", c(5, 20)),
risk = list("trapezoidal", c(1.5, 1.7, 2.3, 2.5)))

set.seed(1234)
probsens_conf(fox, reps = 20000,
prev_exp = list("beta", c(10, 20)),
prev_nexp = list("beta", c(5, 20)),
risk = list("log-normal", c(log(2), .23)))
}
\references{
Fox, M.P, MacLehose, R.F., Lash, T.L., 2021 \emph{Applying Quantitative
Bias Analysis to Epidemiologic Data}, pp.105--140, 256--262,
Springer.

Miettinen, 1971. Components of the Crude Risk Ratio. \emph{Am J Epidemiol}
96(2):168-172.

Li, S.T., Hammond, J.L., 1975. \emph{Generation of Pseudorandom Numbers
with Specified Univariate Distributions and Correlation Coefficients}.
IEEE Trans Syst Man Cybern 5:557-561.
}
\seealso{
Other confounding: 
\code{\link{confounders_array}()},
\code{\link{confounders_evalue}()},
\code{\link{confounders_ext}()},
\code{\link{confounders_limit}()},
\code{\link{probsens_irr_conf}()}
}
\concept{confounding}
