#' Plot L-Moment Ratio Diagram
#'
#' Generates a plot of L-moment ratios with the L-skewness on the x-axis and L-kurtosis
#' on the y-axis. Plots the sample and log-sample L-moment ratios alongside the 
#' theoretical L-moment ratios for a set of candidate distributions. Also includes
#' a small inset around the L-moment ratios of the recommended distribution.
#'
#' @param results List of distribution selection results generated by [select_ldistance()], 
#'  [select_lkurtosis()], or [select_zstatistic()].
#'
#' @param ... Optional named arguments: 'title', 'xlabel', and 'ylabel'.
#'
#' @return \code{ggplot}; plot object containing the L-moment ratio diagram, with:
#' - L-moment ratio curves for each 3-parameter distribution.
#' - Points for the L-moment ratios of each 2-parameter distribution.
#' - Sample and log-sample L-moment ratio \eqn{(t_3, t_4)} points.
#'
#' @seealso [select_ldistance()], [select_lkurtosis()], [select_zstatistic()]
#'
#' @examples
#' data <- rnorm(n = 100, mean = 100, sd = 10)
#' results <- select_ldistance(data)
#' plot_lmom_diagram(results)
#'
#' @import ggplot2
#' @importFrom grDevices pdf
#' @export

plot_lmom_diagram <- function(results, ...) {

	# Get method name from the results
	method <- results$method

	# Create dataframes for the sample L-moments
	reg_moments = utils_sample_lmoments(results$decomposed_data)
	reg_sample_t3 = reg_moments[3]
	reg_sample_t4 = reg_moments[4]
	reg_lm <- data.frame(x = reg_sample_t3, y = reg_sample_t4)

	log_moments = utils_sample_lmoments(log(results$decomposed_data))
	log_sample_t3 = log_moments[3]
	log_sample_t4 = log_moments[4]
	log_lm <- data.frame(x = log_sample_t4, y = log_sample_t4)

	# Generate a dataframe with the moments for each distribution
	dlm <- list()

	for (distribution in c("GUM", "NOR", "LNO", "GEV", "GLO", "GNO", "PE3", "LP3", "WEI")) {

		info <- model_info(distribution)

		if (info$n_params == 2) {
			dm <- theoretical_lmoments_fast(distribution, c(0, 1))
			dlm[[distribution]] = data.frame(x = dm[3], y = dm[4])
		}

		else {

			# Generate a sequence of parameter sets to pass to a 3-parameter distribution
			k_seq <- seq(info$k_bounds[1], info$k_bounds[2], 0.001)
			params <- lapply(k_seq, function(i) c(0, 1, i))

			# Get a matrix of likelihood moment ratios for each parameter set in params
			lmr <- do.call(rbind, lapply(params, function(p) { 
				suppressWarnings(theoretical_lmoments_fast(distribution, p))
			}))

			# Return the t3 and t4 values as a dataframe
			df = data.frame(x = lmr[, 3], y = lmr[, 4]) 
			dlm[[distribution]] <- df[(!is.na(df$x) & !is.na(df$y)), ]

		}

	}

	# Initilaize the legend information
	labels <- c(
		"Sample",
		"Log-Sample",
		"GEV",
		"GUM",
		"GLO",
		"GNO",
		"NOR/LNO",
		"PE3/LP3",
		"WEI"
	)

	colors <- c(
		"#000000",
		"#000000",
		"#e69f00",
		"#e69f00",
		"#56b4e9",
		"#009e73",
		"#009e73",
		"#0072b2",
		"#d55e00"
	)

	shapes <- c(17, 15, NA, 16, NA, NA, 16, NA, NA)

	# Capture optional arguments
	args <- list(...)

    # Set default values, overriding if necessary
	title <- args$title %||% paste(method, "Model Selection", sep = " ")
	xlabel <- args$xlabel %||% expression("L-skewness (" * tau[3] * ")")
	ylabel <- args$ylabel %||% expression("L-kurtosis (" * tau[4] * ")")

	# Generate the plot
	p1 <- ggplot(mapping = aes(x = .data$x, y = .data$y)) +
		geom_line(data = dlm$GEV, aes(color = "3"), linewidth = 1) +
		geom_line(data = dlm$GLO, aes(color = "5"), linewidth = 1) +
		geom_line(data = dlm$GNO, aes(color = "6"), linewidth = 1) +
		geom_line(data = dlm$PE3, aes(color = "8"), linewidth = 1) +
		geom_line(data = dlm$WEI, aes(color = "9"), linewidth = 1) +
		geom_point(data = dlm$GUM, aes(color = "4"), shape = 16, size = 4) +
		geom_point(data = dlm$NOR, aes(color = "7"), shape = 16, size = 4) +
		geom_point(data = reg_lm, aes(color = "1"), shape = 17, size = 5) +
		geom_point(data = log_lm, aes(color = "2"), shape = 15, size = 5) +
		scale_color_manual(labels = labels, values = colors) +
		guides(color = guide_legend(override.aes = list(shape = shapes)))

	# Add the theme
	p1 <- add_theme(add_scales(p1))

	# Add an inset showing the the selected distribution
	point <- if (results$recommendation %in% c("LNO", "LP3")) log_lm else reg_lm
	df <- dlm[[results$recommendation]]

	# Customize the inset depending on the metric
	if (method == "L-distance") {
		df$distance <- sqrt((df$x - point$x)^2 + (df$y - point$y)^2)
		df <- df[which.min(df$distance), ]
		p1 <- p1 + geom_segment(data = point, aes(xend = df$x, yend = df$y))
	} else if (method == "L-kurtosis") {
		df <- df[which.min(abs(df$x - point$x)), ]
		df$distance <- abs(df$y - point$y)
		p1 <- p1 + geom_segment(data = point, aes(xend = df$x, yend = df$y))
	} else {
		df$distance <- sqrt((df$x - point$x)^2 + (df$y - point$y)^2)
		df <- df[which.min(df$distance), ]
	}

	# Dynamically set the radius of the magnified section
	radius <- df$distance * 3

	# Draw the inset plot: same layers, zoomed to [x +/- r, y +/- r]
	p2 <- p1 +
		coord_cartesian(
			xlim = c(point$x - radius, point$x + radius),
			ylim = c(point$y - radius, point$y + radius)
		) +
		theme_void() + 
		theme(
			legend.position = "none",
			panel.border = element_rect(colour = "black", fill = NA, linewidth = 1),
			plot.background = element_rect(colour = NA, fill = "white")
		) + 
		labs(title = paste0("Recommendation: ", results$recommendation))

	# Turn the inset plot into a grob
	p2_grob <- ggplotGrob(p2)

	# Place inset on main plot using annotation_custom()
	p1 <- p1 + 
		annotation_custom(
			grob = p2_grob,
			xmin = -0.25, 
			xmax = 0.25,
			ymin = 0.35, 
			ymax = 0.60
		) + 
		coord_fixed(ratio = 2, xlim = c(-0.7, 0.7), ylim = c(0, 0.7)) +
		labs(x = xlabel, y = ylabel, color = "Legend", title = title)

	# Return the plot
	p1

}
