% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{simplify_network}
\alias{simplify_network}
\title{Simplify Network}
\usage{
simplify_network(
  graph_df,
  nodes,
  method = c("shortest-paths", "cluster"),
  directed = FALSE,
  cost.column = "cost",
  by = NULL,
  radius_km = list(nodes = 7, cluster = 20),
  ...
)
}
\arguments{
\item{graph_df}{A data.frame with columns \code{from} and \code{to} representing the graph edges.
For the cluster method, the graph must also have columns \code{FX}, \code{FY}, \code{TX}, \code{TY}
representing node coordinates.}

\item{nodes}{For \code{method = "shortest-paths"}: either an atomic vector of node IDs, or a
data.frame with columns \code{from} and \code{to} specifying origin-destination pairs.
For \code{method = "cluster"}: an atomic vector of node IDs to preserve. These nodes will
be kept as cluster centroids, and nearby nodes (within \code{radius_km$nodes}) will be
assigned to their clusters. Remaining nodes are clustered using \code{\link[leaderCluster]{leaderCluster}}.}

\item{method}{Character string (default: "shortest-paths"). Method to use for simplification:
\code{"shortest-paths"} computes shortest paths between nodes and keeps only traversed edges;
\code{"cluster"} clusters nodes using the \code{\link[leaderCluster]{leaderCluster}} algorithm and contracts the graph.}

\item{directed}{Logical (default: FALSE). Whether the graph is directed.
For \code{method = "shortest-paths"}: controls path computation direction.
For \code{method = "cluster"}: if TRUE, A->B and B->A remain as separate edges after
contraction; if FALSE, edges are normalized so that \code{from < to} before grouping.}

\item{cost.column}{Character string (default: "cost"). Name of the cost column in \code{graph_df}.
Alternatively, a numeric vector of edge costs with length equal to \code{nrow(graph_df)}.
With \code{method = "cluster"}, a numeric vector of node weights matching \code{nodes_from_graph(graph_df)} can be provided.}

\item{by}{Link characteristics to preserve/not simplify across, passed as a one-sided
formula or character vector of column names. Typically includes attributes like
\emph{mode}, \emph{type}, or \emph{capacity}.
For \code{method = "shortest-paths"}: paths are computed separately for each group
defined by \code{by}, with edges not in the current group penalized (cost multiplied by 100) to compel mode-specific routes.
For \code{method = "cluster"}: edges are grouped by \code{from}, \code{to}, AND
\code{by} columns, preventing consolidation across different modes/types.}

\item{radius_km}{Named list with elements \code{nodes} (default: 7) and \code{cluster} (default: 20).
Only used for \code{method = "cluster"}.
\code{nodes}: radius in kilometers around preserved nodes. Graph nodes within this radius
will be assigned to the nearest preserved node's cluster.
\code{cluster}: radius in kilometers for clustering remaining nodes using leaderCluster.}

\item{\dots}{For \code{method = "cluster"}: additional arguments passed to
\code{\link[collapse]{collap}} for edge attribute aggregation.}
}
\value{
A data.frame containing the simplified graph with:
  \itemize{
    \item For \code{method = "shortest-paths"}:
      \itemize{
        \item All columns from the input \code{graph_df} (for edges that were kept)
        \item Attribute \code{"edges"}: integer vector of edge indices from the original graph
        \item Attribute \code{"edge_counts"}: integer vector indicating how many times each edge was traversed
      }
    \item For \code{method = "cluster"}:
      \itemize{
        \item \code{edge} - New edge identifier
        \item \code{from}, \code{to} - Cluster centroid node IDs
        \item \code{FX}, \code{FY}, \code{TX}, \code{TY} - Coordinates of cluster centroid nodes
        \item Aggregated edge attributes from the original graph
        \item Attribute \code{"group.id"}: mapping from original edges to simplified edges
        \item Attribute \code{"group.starts"}: start indices of each group
        \item Attribute \code{"group.sizes"}: number of original edges per simplified edge
      }
  }
}
\description{
Spatially simplify a network graph using shortest paths or node clustering methods. This further simplifies the network topology but does not preserve full connectivity. It should ideally be called after \code{\link{consolidate_graph}()} if the network is still too large/complex.
}
\details{
\code{simplify_network()} provides two methods to simplify large transport networks:

\strong{Method: "shortest-paths"}
\itemize{
  \item Validates that all origin and destination nodes exist in the network
  \item Computes shortest paths from each origin to all destinations using igraph
  \item Marks all edges that are traversed by at least one shortest path
  \item Returns only the subset of edges that were traversed
  \item If \code{nodes} is a data frame with \code{from} and \code{to} columns, paths are computed
    from each unique origin to its specified destinations
}

\strong{Method: "cluster"}
\itemize{
  \item Requires the graph to have spatial coordinates (\code{FX}, \code{FY}, \code{TX}, \code{TY})
  \item If \code{nodes} is provided, these nodes are preserved as cluster centroids
  \item Nearby nodes (within \code{radius_km$nodes} km) are assigned to the nearest preserved node
  \item Remaining nodes are clustered using \code{\link[leaderCluster]{leaderCluster}} with
    \code{radius_km$cluster} as the clustering radius
  \item For each cluster, the node closest to the cluster centroid is selected as representative
  \item The graph is contracted by mapping all nodes to their cluster representatives
  \item Self-loops (edges where both endpoints map to the same cluster) are dropped
  \item For undirected graphs (\code{directed = FALSE}), edges are normalized so \code{from < to},
    merging opposite-direction edges; for directed graphs, A->B and B->A remain separate
  \item Edge attributes are aggregated using \code{\link[collapse]{collap}} (default: mean for
    numeric, mode for categorical); customize via \code{\dots}
}
}
\examples{
library(flownet)
library(sf)

# Convert segments to undirected graph
graph <- africa_segments |>
  linestrings_from_graph() |>
  linestrings_to_graph() |>
  create_undirected_graph(FUN = "fsum")

# Get city/port nodes to preserve
nodes_df <- nodes_from_graph(graph, sf = TRUE)
nearest_nodes <- nodes_df$node[st_nearest_feature(africa_cities_ports, nodes_df)]

# Initial consolidation
graph <- consolidate_graph(graph, keep = nearest_nodes, w = ~ passes)

# Method 1: Shortest-paths simplification (keeps only traversed edges)
graph_simple <- simplify_network(graph, nearest_nodes,
                                 method = "shortest-paths",
                                 cost.column = ".length")
nrow(graph_simple)  # Reduced number of edges

\donttest{
# Method 2: Cluster-based simplification (contracts graph spatially)
# Compute node weights for clustering
node_weights <- collapse::rowbind(
  collapse::fselect(graph, node = from, gravity_rd),
  collapse::fselect(graph, to, gravity_rd),
  use.names = FALSE) |>
  collapse::collap(~ node, "fsum")

graph_cluster <- simplify_network(graph, nearest_nodes,
                                  method = "cluster",
                                  cost.column = node_weights$gravity_rd,
                                  radius_km = list(nodes = 30, cluster = 27),
                                  w = ~ passes)
nrow(graph_cluster)
}

}
