% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/simulate-gctsc.R
\name{sim_gctsc}
\alias{sim_gctsc}
\alias{sim_poisson}
\alias{sim_negbin}
\alias{sim_zip}
\alias{sim_binom}
\alias{sim_bbinom}
\alias{sim_zib}
\alias{sim_zibb}
\title{Simulate from Gaussian Copula Time Series Models}
\usage{
sim_poisson(mu, tau, arma_order, nsim = 100, seed = NULL)

sim_negbin(mu, dispersion, tau, arma_order, nsim = 100, seed = NULL)

sim_zip(mu, pi0, tau, arma_order, nsim = 100, seed = NULL)

sim_binom(prob, size, tau, arma_order, nsim = 100, seed = NULL)

sim_bbinom(prob, rho, size, tau, arma_order, nsim = 100, seed = NULL)

sim_zib(prob, pi0, size, tau, arma_order, nsim = 100, seed = NULL)

sim_zibb(prob, rho, pi0, size, tau, arma_order, nsim = 100, seed = NULL)
}
\arguments{
\item{mu}{Mean parameter(s):
\itemize{
\item For Poisson, ZIP, and negative binomial: \eqn{\mu > 0}.
\item Scalar or vector of length \code{nsim}.
}}

\item{tau}{Numeric vector of ARMA coefficients, ordered as
\code{c(phi_1, ..., phi_p, theta_1, ..., theta_q)}. ARMA(0,0) is not supported.}

\item{arma_order}{Integer vector \code{c(p, q)} giving AR and MA orders.}

\item{nsim}{Positive integer; number of time points to simulate.}

\item{seed}{Optional integer to set the random seed.}

\item{dispersion}{Overdispersion parameter for negative binomial marginals
(\eqn{\kappa > 0} in \eqn{\mathrm{Var}(Y) = \mu + \kappa \mu^2}).}

\item{pi0}{Zero-inflation probability for ZIP, ZIB, and ZIBB marginals:
\eqn{0 \le \pi_0 < 1}, scalar or length \code{nsim}.}

\item{prob}{Probability parameter(s) for binomial-type marginals:
\eqn{0 < p < 1}, scalar or length \code{nsim}.}

\item{size}{Number of trials for binomial-type marginals; positive integer
(scalar).}

\item{rho}{Intra-cluster correlation for beta-binomial and ZIBB marginals
(\eqn{0 < \rho < 1} in \eqn{\mathrm{Var}(Y) = n p (1-p) [1 + (n-1)\rho]}) where \eqn{n} is the number of trials.}
}
\value{
A list with components:
\itemize{
\item \code{y}: Simulated time series data.
\item \code{z}: Latent Gaussian process values.
\item \code{marginal}: Marginal distribution name.
\item \code{parameters}: List of parameters used.
\item \code{cormat}: ARMA structure.
}
}
\description{
These functions simulate time series data from Gaussian copula models
with various discrete marginals and an ARMA dependence structure.
}
\details{
\strong{Marginal types:}
\itemize{
\item {Poisson}: Counts with mean  \eqn{\mu}.
\item {Negative binomial (NB)}: Overdispersed counts with mean  \eqn{\mu} and dispersion parameter \eqn{\kappa}.
\item {Binomial}: Number of successes in \eqn{n} trials with success probability \eqn{p}.
\item {Beta–-binomial (BB)}: Binomial with success probability \eqn{p} following a beta distribution, allowing intra-cluster correlation  \eqn{\rho}.
\item {Zero--inflated Poisson (ZIP)}: Poisson with extra probability \eqn{\pi_0} of an excess zero.
\item {Zero--inflated binomial (ZIB)}: Binomial with extra probability \eqn{\pi_0} of an excess zero.
\item {Zero--inflated beta–binomial (ZIBB)}: Beta–binomial with extra probability \eqn{\pi_0} of an excess zero.
}

\strong{Parameterization notes:}
\itemize{
\item Negative binomial uses \code{dispersion} (\eqn{\kappa}) to model
overdispersion: larger \code{dispersion} increases variance for a fixed mean.
\item Beta--binomial and ZIBB use \code{rho} as the overdispersion parameter:
\eqn{\rho} is the intra-class correlation, with \eqn{\rho \to 0}
giving the binomial model.
\item Zero--inflated marginals include a separate \code{pi0} parameter that
controls the extra probability mass at zero.
}
}
\examples{
# Poisson example
sim_poisson(mu = 10, tau = c(0.2, 0.2), arma_order = c(1, 1), nsim = 100, seed = 42)

# Negative Binomial example
sim_negbin(mu = 10, dispersion = 2, tau = c(0.5, 0.5), arma_order = c(1, 1))

# Beta-Binomial example with seasonal covariates
n <- 100
xi <- numeric(n)
zeta <- rnorm(n)
for (j in 3:n) {
  xi[j] <- 0.6 * xi[j - 1] - 0.4 * xi[j - 2] + zeta[j]
}
prob <- plogis(0.2 + 0.3 * sin(2 * pi * (1:n) / 12) +
             0.5 * cos(2 * pi * (1:n) / 12) + 0.3 * xi)
sim_zibb(prob, rho = 1/6, pi0 = 0.2, size = 24, tau = 0.5, arma_order = c(1, 0))

}
