#' Download spatial data of official metropolitan areas in Brazil
#'
#' @description
#' The function returns the shapes of municipalities grouped by their respective
#' metro areas. Metropolitan areas are created by each state in Brazil. The data
#' set includes the municipalities that belong to all metropolitan areas in the
#' country according to state legislation in each year. Original data were
#' generated by Institute of Geography. Data at scale 1:250,000, using Geodetic
#' reference system "SIRGAS2000" and CRS(4674).
#'
#' @param year Numeric. Year of the data in YYYY format. Defaults to `2018`.
#' @param code_state The two-digit code of a state or a two-letter uppercase
#'                   abbreviation (e.g. 33 or "RJ"). If `code_state="all"` (the
#'                   default), the function downloads all states.
#' @template simplified
#' @template showProgress
#' @template cache
#'
#' @return An `"sf" "data.frame"` object
#'
#' @export
#' @family area functions
#'
#' @examplesIf identical(tolower(Sys.getenv("NOT_CRAN")), "true")
#' # Read all official metropolitan areas for a given year
#'   m <- read_metro_area(2005)
#'
#'   m <- read_metro_area(2018)
read_metro_area <- function(year = 2018,
                            code_state = "all",
                            simplified = TRUE,
                            showProgress = TRUE,
                            cache = TRUE){

  # Get metadata with data url addresses
  temp_meta <- select_metadata(geography="metropolitan_area", year=year, simplified=simplified)

  # list paths of files to download
  file_url <- as.character(temp_meta$download_path)

  # download files
  temp_sf <- download_gpkg(file_url = file_url,
                           showProgress = showProgress,
                           cache = cache)

  # check if download failed
  if (is.null(temp_sf)) { return(invisible(NULL)) }

  # filter state
  temp_sf <- filter_state(temp_sf, code = code_state)

  return(temp_sf)
  }
