% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/blend.R
\name{with_blend}
\alias{with_blend}
\title{Blend a layer with a reference}
\usage{
with_blend(
  x,
  bg_layer,
  blend_type = "over",
  flip_order = FALSE,
  alpha = NA,
  ...
)
}
\arguments{
\item{x}{A ggplot2 layer object, a ggplot, a grob, or a character string
naming a filter}

\item{bg_layer}{The background layer to use. Can either be a string
identifying a registered filter, or a raster object. The map will be resized
to match the dimensions of x.}

\item{blend_type}{The type of blending to perform. See \emph{Details}}

\item{flip_order}{Should the order of the background and the overlay be
flipped so that \code{bg_layer} is treated as being on top and \code{x} being below.}

\item{alpha}{For non-Duff-Porter blends the alpha channel may become modified.
This argument can be used to set the resulting alpha channel to that of the
source (\code{"src"}) or destination (\code{"dst"})}

\item{...}{Arguments to be passed on to methods. See
\link[=object_support]{the documentation of supported object} for a description of
object specific arguments.}
}
\value{
Depending on the input, either a \code{grob}, \code{Layer}, list of \code{Layer}s,
\code{guide}, or \code{element} object. Assume the output can be used in the same
context as the input.
}
\description{
This filter blends the layer with a reference according to one of many rules
as laid out in the \emph{Details} section.
}
\details{
Two images can be blended in a variety of ways as described below. In the
following \emph{source} will refer to the top-most image, and \emph{destination}  to
the bottom-most image. Note that which is which can be controlled with the
\code{flip_order} argument.
\subsection{Duff-Porter alpha blend modes}{

This is a set of well-defined blend types for composing two images, taking
their opacity into account:
\itemize{
\item \code{"source"}: Completely disregards the destination, leaving only the source
\item \code{"destination"}: Completely disregards the source, leaving only the
destination
\item \code{"clear"}: Disregards both destination and source
\item \code{"xor"}: Composes source on top of destination, setting shared areas to
transparent
\item \code{"over"}: Composes source on top of destination
\item \code{"in"}: Shows source, but only where the destination is opaque
\item \code{"out"}: Shows source but only where the destination is transparent
\item \code{"atop"}: Composes source on top of destination, keeping the transparency
of destination
\item \code{"copy"}: Like source, but will only affect the area occupied by the source
image
}
}

\subsection{Mathmathical blend modes}{

These blend modes perform often complex channel operations based on the
different channel values in the source and destination:
\itemize{
\item \code{"multiply"}: Multiplies the channel values of source and destination
together (after scaling them to 0-1) to obtain new channel values
\item \code{"screen"}: As multiply except that the channels are scaled to 1-0 before
multiplication, and the result is reversed again before being used
\item \code{"bumpmap"}: Like multiple, except source is converted to greyscale first
\item \code{"divide"}: Divide the channel values in source by the channel values in
destination
\item \code{"plus"}: Add the channel values together \emph{including the alpha channel}
\item \code{"minus"}: Subtracts the destination channels from the source channels
\item \code{"modulus_plus"}: As plus, but overflow will wrap around instead of being
capped
\item \code{"modulus_minus"}: As minus but overflow (underflow) will wrap around
instead of being capped
\item \code{"difference"}: Takes the absolute difference in channel values between
source and destination
\item \code{"exclusion"}: \code{source + destination - 2*source*destination}. A sort of
averaged difference
\item \code{"lighten"}: Will pick the lightest pixel at each pixel
\item \code{"darken"}: Will pick the darkest pixel at each pixel
\item \code{"lighten_intensity"}: Will pick the most intense colour at each pixel
\item \code{"darken_intensity"}: Will pick the least intense colour at each pixel
}
}

\subsection{Lighting blend modes}{

These blend modes are designed to provide different lighting effects:
\itemize{
\item \code{"overlay"}: Simultaneously multiplies and screens at the same time based
on the colour values of the destination. Will colorize midtones in the
destination with the source
\item \code{"hard_light"}: The inverse of overlay (i.e. the source acts as the
destination and vice versa)
\item \code{"soft_light"}: Like overlay but will extent the range of colorization past
the midtones
\item \code{"pegtop_light"}: Like soft-light, but without any discontinuity in the
blend
\item \code{"linear_light"}: Combines dodging and burning so that the destination will
be dodged (lightened) when the source is light and burned (darkened) when
the source is dark
\item \code{"vivid_light"}: A refinement of linear-light that better avoids shading
intense colours
\item \code{"pin_light"}: Preserves midtones of the destination and only shades
lighter and darker parts, resulting in harsh, contrasty lightning.
\item \code{"linear_dodge"}: Lighten the destination if the source is light
\item \code{"linear_burn"}: Darked the destination if the source is dark
\item \code{"color_dodge"}: Like linear-dodge, but preserves blacks in the destination
image
\item \code{"color_burn"}: Like linear-burn but preserve whites in the destination
image
}
}

\subsection{Channel copying blends}{

These blend modes copies a single channel from the source to the destination
\itemize{
\item \code{"copy_opacity"}: Will set the opacity of destination to the grayscale
version of source. To copy the opacity of source into destination use
\code{blend_type = "in"} with \code{flip_order = TRUE}.
\item \code{"copy_red"}: Copies the red channel in source into the red channel in
destination
\item \code{"copy_green"}: Copies the green channel in source into the green channel in
destination
\item \code{"copy_blue"}: Copies the blue channel in source into the blue channel in
destination
\item \code{"hue"}: Replaces the hue of the destination with the hue of the source
\item \code{"saturate"}: Replaces the saturation of the destination with the
saturation of the source
\item \code{"luminize"}: Replaces the luminance of the destination with the luminance
of the source
\item \code{"colorize"}: Combines hue and saturate
}
}

\subsection{Special blends}{
\itemize{
\item \code{"unique"}: Only keep pixels in the source that differ from the
destination.
}

The above is obviously a very quick overview. More information can be found
in \url{https://legacy.imagemagick.org/Usage/compose/}
}
}
\examples{
library(ggplot2)
ggplot() +
  as_reference(
    geom_text(aes(0.5, 0.5, label = 'Blend Modes!'), size = 10, fontface = 'bold'),
    id = "text"
  ) +
  with_blend(
    geom_polygon(aes(c(0, 1, 1), c(0, 0, 1)), colour = NA, fill = 'magenta'),
    bg_layer = "text",
    blend_type = 'xor'
  )

}
\seealso{
Other blend filters: 
\code{\link{with_blend_custom}()},
\code{\link{with_interpolate}()},
\code{\link{with_mask}()}
}
\concept{blend filters}
