% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gipsmult_class.R
\name{plot.gipsmult}
\alias{plot.gipsmult}
\title{Plot optimized matrix or optimization \code{gipsmult} object}
\usage{
\method{plot}{gipsmult}(
  x,
  type = NA,
  logarithmic_y = TRUE,
  logarithmic_x = FALSE,
  color = NULL,
  title_text = "Convergence plot",
  xlabel = NULL,
  ylabel = NULL,
  show_legend = TRUE,
  ylim = NULL,
  xlim = NULL,
  ...
)
}
\arguments{
\item{x}{Object of a \code{gipsmult} class.}

\item{type}{A character vector of length 1. One of
\code{c("heatmap", "MLE", "best", "all", "both", "n0", "block_heatmap")}:
\itemize{
\item \code{"heatmap"}, \code{"MLE"} - Plots heatmaps of the Maximum Likelihood
Estimator of the covariance matrices given the permutation.
That is, the \code{Ss} matrices inside the \code{gipsmult} object
projected on the permutation in the \code{gipsmult} object.
\item \code{"best"} - Plots the line of the biggest a posteriori found over time.
\item \code{"all"} - Plots the line of a posteriori for all visited states.
\item \code{"both"} - Plots both lines from "all" and "best".
\item \code{"n0"} - Plots the line of \code{n0}s that were spotted during optimization
(only for "MH" optimization).
\item \code{"block_heatmap"} - Plots heatmaps of diagonally block representation of \code{Ss}.
Non-block entries (equal to 0) are white for better clarity.
}

The default value is \code{NA}, which will be changed to "heatmap" for
non-optimized \code{gipsmult} objects and to "both" for optimized ones.
Using the default produces a warning.
All other arguments are ignored for
the \code{type = "heatmap"}, \code{type = "MLE"}, or \code{type = "block_heatmap"}.}

\item{logarithmic_y, logarithmic_x}{A boolean.
Sets the axis of the plots in logarithmic scale.}

\item{color}{Vector of colors to be used to plot lines.}

\item{title_text}{Text to be in the title of the plot.}

\item{xlabel}{Text to be on the bottom of the plot.}

\item{ylabel}{Text to be on the left of the plot.}

\item{show_legend}{A boolean. Whether or not to show a legend.}

\item{ylim}{Limits of the y axis. When \code{NULL},
the minimum, and maximum of the \code{\link[=log_posteriori_of_gipsmult]{log_posteriori_of_gipsmult()}} are taken.}

\item{xlim}{Limits of the x axis. When \code{NULL},
the whole optimization process is shown.}

\item{...}{Additional arguments passed to
other various elements of the plot.}
}
\value{
When \code{type} is one of \code{"best"}, \code{"all"}, \code{"both"} or \code{"n0"},
returns an invisible \code{NULL}.
When \code{type} is one of \code{"heatmap"}, \code{"MLE"} or \code{"block_heatmap"},
returns an object of class \code{ggplot}.
}
\description{
Plot heatmaps of the MAP covariance matrices estimator
or the convergence of the optimization method.
The plot depends on the \code{type} argument.
}
\examples{
require("MASS") # for mvrnorm()

perm_size <- 6
mu1 <- runif(6, -10, 10)
mu2 <- runif(6, -10, 10) # Assume we don't know the means
sigma1 <- matrix(
  data = c(
    1.0, 0.8, 0.6, 0.4, 0.6, 0.8,
    0.8, 1.0, 0.8, 0.6, 0.4, 0.6,
    0.6, 0.8, 1.0, 0.8, 0.6, 0.4,
    0.4, 0.6, 0.8, 1.0, 0.8, 0.6,
    0.6, 0.4, 0.6, 0.8, 1.0, 0.8,
    0.8, 0.6, 0.4, 0.6, 0.8, 1.0
  ),
  nrow = perm_size, byrow = TRUE
)
sigma2 <- matrix(
  data = c(
    1.0, 0.5, 0.2, 0.0, 0.2, 0.5,
    0.5, 1.0, 0.5, 0.2, 0.0, 0.2,
    0.2, 0.5, 1.0, 0.5, 0.2, 0.0,
    0.0, 0.2, 0.5, 1.0, 0.5, 0.2,
    0.2, 0.0, 0.2, 0.5, 1.0, 0.5,
    0.5, 0.2, 0.0, 0.2, 0.5, 1.0
  ),
  nrow = perm_size, byrow = TRUE
)
# sigma1 and sigma2 are matrices invariant under permutation (1,2,3,4,5,6)
numbers_of_observations <- c(21, 37)
Z1 <- MASS::mvrnorm(numbers_of_observations[1], mu = mu1, Sigma = sigma1)
Z2 <- MASS::mvrnorm(numbers_of_observations[2], mu = mu2, Sigma = sigma2)
S1 <- cov(Z1)
S2 <- cov(Z2) # Assume we have to estimate the mean

g <- gipsmult(list(S1, S2), numbers_of_observations)
if (require("graphics")) {
  plot(g, type = "MLE")
}

g_map <- find_MAP(g, max_iter = 30, show_progress_bar = FALSE, optimizer = "hill_climbing")
if (require("graphics")) {
  plot(g_map, type = "both", logarithmic_x = TRUE)
}

if (require("graphics")) {
  plot(g_map, type = "MLE")
}
# Now, the output is (most likely) different because the permutation
# `g_map[[1]]` is (most likely) not an identity permutation.

g_map_MH <- find_MAP(g, max_iter = 30, show_progress_bar = FALSE, optimizer = "MH")
if (require("graphics")) {
  plot(g_map_MH, type = "n0")
}
}
\seealso{
\itemize{
\item \code{\link[=find_MAP]{find_MAP()}} - Usually, the \code{plot.gipsmult()}
is called on the output of \code{find_MAP()}.
\item \code{\link[=gipsmult]{gipsmult()}} - The constructor of a \code{gipsmult} class.
The \code{gipsmult} object is used as the \code{x} parameter.
}
}
