% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/de.R
\name{diffevo}
\alias{diffevo}
\title{Differential Evolution Algorithms for Minimization Problems}
\usage{
diffevo(
  objFunc,
  lower,
  upper,
  init = NULL,
  fixed = NULL,
  DE_INFO = NULL,
  seed = NULL,
  verbose = TRUE,
  ...
)
}
\arguments{
\item{objFunc}{The R or Rcpp compiled objective function. See the example.}

\item{lower}{The vector of finite lower bounds of the search domain.
Its length should be equal to the dimension of the domain space.}

\item{upper}{The vector of finite upper bounds of the search domain.
Its length should be equal to the dimension of the domain space.}

\item{init}{The vector of initial population.
Its length should be equal to the dimension of the domain space.
When there are more than one initial vectors, specify \code{init} as
a matrix. Each row vector represents one initial point.
The default for \code{init} is \code{NULL}.}

\item{fixed}{The vector of real values and NA values that controls DE to search only 
for the NA-valued components.}

\item{DE_INFO}{The list of DE parameters generated by \code{getDEInfo()}.}

\item{seed}{The random seed that controls initial population of DE. The default is \code{NULL}.}

\item{verbose}{The logical value controls if DE would reports the updating progress. The default is \code{TRUE}.}

\item{...}{Further arguments to be passed to \code{objFunc}.}
}
\value{
An List.
\describe{
\item{par}{ the global best particle.}
\item{val}{ the objective function value of the global best particle.}
\item{history}{ a vector of objective function values of the global best particle in DE search history.}
\item{cputime}{ the computational time in seconds.}
}
}
\description{
The general-purpose implementation of differential evolution algorithms for minimizing an user-defined objective function.
}
\examples{
# three-dimensional function
objf <- function(x, loc) {
  val <- 0
  for (i in 1:length(x)) {
    val <- val + (x[i] - loc)^2
  }
  return(val)
}

upp_bound <- rep(5, 3)
low_bound <- rep(-5, 3)
loc_shift <- 1

alg_setting <- getDEInfo(nPop = 32, maxIter = 100, deType = "rand-1", 
                         sf = 0.5, cr = 0.1)
res <- diffevo(objFunc = objf, lower = low_bound, upper = upp_bound, 
               DE_INFO = alg_setting, loc = loc_shift)
res$par
res$val

# C++ function example
\donttest{
library(Rcpp)
library(RcppArmadillo)
objf_c <- cppFunction('double objf_c(SEXP x, SEXP loc) {
    double val = 0;
    double loc_c = (double)Rcpp::as<double>(loc);
   arma::rowvec x_c = (arma::rowvec)Rcpp::as<arma::rowvec>(x);
   for (arma::uword i = 0; i < x_c.n_elem; i++) {
     val += (x_c(i) - loc_c)*(x_c(i) - loc_c);
   }
    return val;
 }', depends = "RcppArmadillo")

alg_setting <- getDEInfo(nPop = 32, maxIter = 100, deType = "rand-1", 
                         sf = 0.5, cr = 0.1)
res_c <- diffevo(objFunc = objf_c, lower = low_bound, upper = upp_bound, 
                       DE_INFO = alg_setting, loc = 1)
res_c$par
res_c$val
}
}
\references{
Storn, R., & Price, K. (1997). Differential evolution-a simple and efficient heuristic for global optimization over continuous spaces. Journal of global optimization, 11, 341-359.
}
