% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/survival_adapt.R
\name{survival_adapt}
\alias{survival_adapt}
\title{Simulate and execute a single adaptive clinical trial design with a
time-to-event endpoint}
\usage{
survival_adapt(
  hazard_treatment,
  hazard_control = NULL,
  cutpoints = 0,
  N_total,
  lambda = 0.3,
  lambda_time = 0,
  interim_look = NULL,
  end_of_study,
  prior = c(0.1, 0.1),
  block = 2,
  rand_ratio = c(1, 1),
  prop_loss = 0,
  alternative = "greater",
  h0 = 0,
  Fn = 0.05,
  Sn = 0.9,
  prob_ha = 0.95,
  N_impute = 10,
  N_mcmc = 10,
  method = "logrank",
  imputed_final = FALSE
)
}
\arguments{
\item{hazard_treatment}{vector. Constant hazard rates under the treatment
arm.}

\item{hazard_control}{vector. Constant hazard rates under the control arm.}

\item{cutpoints}{vector. Times at which the baseline hazard changes. Default
is \code{cutpoints = 0}, which corresponds to a simple (non-piecewise)
exponential model.}

\item{N_total}{integer. Maximum sample size allowable}

\item{lambda}{vector. Enrollment rates across simulated enrollment times. See
\code{\link{enrollment}} for more details.}

\item{lambda_time}{vector. Enrollment time(s) at which the enrollment rates
change. Must be same length as lambda. See \code{\link{enrollment}} for
more details.}

\item{interim_look}{vector. Sample size for each interim look. Note: the
maximum sample size should not be included.}

\item{end_of_study}{scalar. Length of the study; i.e. time at which endpoint
will be evaluated.}

\item{prior}{vector. The prior distributions for the piecewise hazard rate
parameters are each \eqn{Gamma(a_0, b_0)}, with specified (known)
hyper-parameters \eqn{a_0} and \eqn{b_0}. The default non-informative prior
distribution used is Gamma(0.1, 0.1), which is specified by setting
\code{prior = c(0.1, 0.1)}.}

\item{block}{scalar. Block size for generating the randomization schedule.}

\item{rand_ratio}{vector. Randomization allocation for the ratio of control
to treatment. Integer values mapping the size of the block. See
\code{\link{randomization}} for more details.}

\item{prop_loss}{scalar. Overall proportion of subjects lost to follow-up.
Defaults to zero.}

\item{alternative}{character. The string specifying the alternative
hypothesis, must be one of \code{"greater"} (default), \code{"less"} or
\code{"two.sided"}.}

\item{h0}{scalar. Null hypothesis value of \eqn{p_\textrm{treatment} -
  p_\textrm{control}} when \code{method = "bayes"}. Default is \code{h0 = 0}.
The argument is ignored when \code{method = "logrank"} or \code{= "cox"};
in those cases the usual test of non-equal hazards is assumed.}

\item{Fn}{vector of \code{[0, 1]} values. Each element is the probability
threshold to stop at the \eqn{i}-th look early for futility. If there are
no interim looks (i.e. \code{interim_look = NULL}), then \code{Fn} is not
used in the simulations or analysis. The length of \code{Fn} should be the
same as \code{interim_look}, else the values are recycled.}

\item{Sn}{vector of \code{[0, 1]} values. Each element is the probability
threshold to stop at the \eqn{i}-th look early for expected success. If
there are no interim looks (i.e. \code{interim_look = NULL}), then
\code{Sn} is not used in the simulations or analysis. The length of
\code{Sn} should be the same as \code{interim_look}, else the values are
recycled.}

\item{prob_ha}{scalar \code{[0, 1]}. Probability threshold of alternative
hypothesis.}

\item{N_impute}{integer. Number of imputations for Monte Carlo simulation of
missing data.}

\item{N_mcmc}{integer. Number of samples to draw from the posterior
distribution when using a Bayesian test (\code{method = "bayes"}).}

\item{method}{character. For an imputed data set (or the final data set after
follow-up is complete), whether the analysis should be a log-rank
(\code{method = "logrank"}) test, Cox proportional hazards regression model
Wald test (\code{method = "cox"}), a fully-Bayesian analysis (\code{method
  = "bayes"}), or a chi-square test (\code{method = "chisq"}). See Details
section.}

\item{imputed_final}{logical. Should the final analysis (after all subjects
have been followed-up to the study end) be based on imputed outcomes for
subjects who were LTFU (i.e. right-censored with time
\code{<end_of_study})? Default is \code{TRUE}. Setting to \code{FALSE}
means that the final analysis would incorporate right-censoring.}
}
\value{
A data frame containing some input parameters (arguments) as well as
statistics from the analysis, including:

\describe{
\item{\code{N_treatment:}}{
integer. The number of patients enrolled in the treatment arm for
each simulation.}
\item{\code{N_control:}}{
integer. The number of patients enrolled in the control arm for
each simulation.}
\item{\code{est_interim:}}{
scalar. The treatment effect that was estimated at the time of the
interim analysis. Note this is not actually used in the final
analysis.}
\item{\code{est_final:}}{
scalar. The treatment effect that was estimated at the final analysis.
Final analysis occurs when either the maximum sample size is reached
and follow-up complete, or the interim analysis triggered an early
stopping of enrollment/accrual and follow-up for those subjects is
complete.}
\item{\code{post_prob_ha:}}{
scalar. The corresponding posterior probability from the final
analysis. If \code{imputed_final} is true, this is calculated as the
posterior probability of efficacy (or equivalent, depending on how
\code{alternative:} and \code{h0} were specified) for each imputed
final analysis dataset, and then averaged over the \code{N_impute}
imputations. If \code{method = "logrank"}, \code{post_prob_ha} is
calculated in the same fashion, but value represents \eqn{1 - P},
where \eqn{P} denotes the frequentist \eqn{P}-value.}
\item{\code{stop_futility:}}{
integer. A logical indicator of whether the trial was stopped early for
futility.}
\item{\code{stop_expected_success:}}{
integer. A logical indicator of whether the trial was stopped early for
expected success.}
}
}
\description{
Simulate and execute a single adaptive clinical trial design with a
time-to-event endpoint
}
\details{
Implements the Goldilocks design method described in Broglio et al.
(2014). At each interim analysis, two probabilities are computed:
\enumerate{
\item \strong{The posterior predictive probability of eventual success.} This is
calculated as the proportion of imputed datasets at the \emph{current} sample
size that would go on to be success at the specified threshold. At each
interim analysis it is compared to the corresponding element of
\code{Sn}, and if it exceeds the threshold,
accrual/enrollment is suspended and the outstanding follow-up allowed to
complete before conducting the pre-specified final analysis.
\item \strong{The posterior predictive probability of final success}. This is
calculated as the proportion of imputed datasets at the \emph{maximum}
threshold that would go on to be successful. Similar to above, it is
compared to the corresponding element of \code{Fn}, and if it
is less than the threshold, accrual/enrollment is suspended and the
trial terminated. Typically this would be a binding decision. If it is
not a binding decision, then one should also explore the simulations
with \code{Fn = 0}.
}

Hence, at each interim analysis look, 3 decisions are allowed:
\enumerate{
\item \strong{Stop for expected success}
\item \strong{Stop for futility}
\item \strong{Continue to enroll} new subjects, or if at maximum sample size,
proceed to final analysis.
}

At each interim (and final) analysis methods as:
\itemize{
\item Log-rank test (\code{method = "logrank"}).
Each (imputed) dataset with both treatment and control arms can be
compared using a standard log-rank test. The output is a \emph{P}-value,
and there is no treatment effect reported. The function returns \eqn{1 -
     P}, which is reported in \code{post_prob_ha}. Whilst not a posterior
probability, it can be contrasted in the same manner. For example, if
the success threshold is \eqn{P < 0.05}, then one requires
\code{post_prob_ha} \eqn{> 0.95}. The reason for this is to enable
simple switching between Bayesian and frequentist paradigms for
analysis.
\item Cox proportional hazards regression Wald test (\code{method = "cox"}).
Similar to the log-rank test, a \emph{P}-value is calculated based on a
two-sided test. However, for consistency, \eqn{1 - P}, which is
reported in \code{post_prob_ha}. Whilst not a posterior probability, it
can be contrasted in the same manner. For example, if the success
threshold is \eqn{P < 0.05}, then one requires \code{post_prob_ha}
\eqn{> 0.95}.
\item Bayesian absolute difference (\code{method = "bayes"}).
Each imputed dataset is used to update the conjugate Gamma prior
(defined by \code{prior}), yielding a posterior distribution for the
piecewise exponential rate parameters. In turn, the posterior
distribution of the cumulative incidence function (\eqn{1 - S(t)}, where
\eqn{S(t)} is the survival function) evaluated at time
\code{end_of_study} is calculated. If a single arm study, then this
summarizes the treatment effect, else, if a two-armed study, the
independent posteriors are used to estimate the posterior distribution
of the difference. A posterior probability is calculated according to
the specification of the test type (\code{alternative}) and the value of
the null hypothesis (\code{h0}).
\item Chi-square test (\code{method = "chisq"}).
Each (imputed) dataset with both treatment and control arms can be
compared using a standard chi-square test on the final event status,
which discards the event time information. The output is a
\emph{P}-value, and there is no treatment effect reported. The function
returns \eqn{1 - P}, which is reported in \code{post_prob_ha}. Whilst
not a posterior probability, it can be contrasted in the same manner.
For example, if the success threshold is \eqn{P < 0.05}, then one
requires \code{post_prob_ha} \eqn{> 0.95}. The reason for this is to
enable simple switching between Bayesian and frequentist paradigms for
analysis.
\item Imputed final analysis (\code{imputed_final}).
The overall final analysis conducted after accrual is suspended and
follow-up is complete can be analyzed on imputed datasets (default) or
on the non-imputed dataset. Since the imputations/predictions used
during the interim analyses assume all subjects are imputed (since loss
to follow-up is not yet known), it would seem most appropriate to
conduct the trial in the same manner, especially if loss to follow-up
rates are appreciable. Note, this only applies to subjects who are
right-censored due to loss to follow-up, which we assume is a
non-informative process. This can be used with any \code{method}.
}
}
\examples{
# RCT with exponential hazard (no piecewise breaks)
# Note: the number of imputations is small to enable this example to run
#       quickly on CRAN tests. In practice, much larger values are needed.
survival_adapt(
 hazard_treatment = -log(0.85) / 36,
 hazard_control = -log(0.7) / 36,
 cutpoints = 0,
 N_total = 600,
 lambda = 20,
 lambda_time = 0,
 interim_look = 400,
 end_of_study = 36,
 prior = c(0.1, 0.1),
 block = 2,
 rand_ratio = c(1, 1),
 prop_loss = 0.30,
 alternative = "less",
 h0 = 0,
 Fn = 0.05,
 Sn = 0.9,
 prob_ha = 0.975,
 N_impute = 10,
 N_mcmc = 10,
 method = "bayes")
}
\references{
Broglio KR, Connor JT, Berry SM. Not too big, not too small: a Goldilocks
approach to sample size selection. \emph{Journal of Biopharmaceutical
Statistics}, 2014; 24(3): 685–705.
}
