% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geks.R
\name{geks}
\alias{geks}
\alias{tornqvist_geks}
\alias{fisher_geks}
\alias{walsh_geks}
\title{GEKS index}
\usage{
geks(f, r = 0)

tornqvist_geks(
  p,
  q,
  period,
  product,
  window = nlevels(period),
  n = window - 1L,
  na.rm = FALSE,
  match_method = c("all", "back-price")
)

fisher_geks(
  p,
  q,
  period,
  product,
  window = nlevels(period),
  n = window - 1L,
  na.rm = FALSE,
  match_method = c("all", "back-price")
)

walsh_geks(
  p,
  q,
  period,
  product,
  window = nlevels(period),
  n = window - 1L,
  na.rm = FALSE,
  match_method = c("all", "back-price")
)
}
\arguments{
\item{f}{A \link[=price_indexes]{price index function} that uses information on
both base and current-period prices and quantities, and satisfies the
time-reversal test. Usually a Törnqvist, Fisher, or Walsh index.}

\item{r}{A finite number giving the order of the generalized mean used to
average price indexes over the rolling window. The default uses a
geometric mean.}

\item{p}{A numeric vector of prices, the same length as \code{q}.}

\item{q}{A numeric vector of quantities, the same length as \code{p}.}

\item{period}{A factor, or something that can be coerced into one, that
gives the corresponding time period for each element in \code{p} and
\code{q}. The ordering of time periods follows the levels of \code{period}
to agree with \code{\link[=cut.Date]{cut()}}.}

\item{product}{A factor, or something that can be coerced into one, that
gives the corresponding product identifier for each element in \code{p} and
\code{q}.}

\item{window}{A positive integer giving the length of the rolling window.
The default is a window that encompasses all periods in \code{period}.
Non-integers are truncated towards zero.}

\item{n}{A positive integer giving the length of the index series for each
window, starting from the end of the window. For example, if there are 13
periods in \code{window}, setting \code{n = 1} gives the index for period 13. The
default gives an index for each period in \code{window}. Non-integers are
truncated towards zero.}

\item{na.rm}{Passed to \code{f} to control if missing values are removed.}

\item{match_method}{Either 'all' to match all products against each other
(the default) or 'back-price' to match only back prices. The later can be
faster when there is lots of product imbalanced, but should be used with
a balanced index-number formula \code{f}.}
}
\value{
\code{geks()} returns a function:

\preformatted{function(p,
         q,
         period,
         product,
         window = nlevels(period),
         n = window - 1,
         na.rm = FALSE,
         match_method = c("all", "back-price")){...}}

This calculates a period-over-period GEKS index with the desired
index-number formula, returning a list for each window with a named-numeric
vector of index values.

\code{tornqvist_geks()}, \code{fisher_geks()}, and \code{walsh_geks()} each return a list
with a named numeric vector giving the value of the respective
period-over-period GEKS index for each window.
}
\description{
Calculate a generalized inter-temporal GEKS price index over a rolling
window.
}
\note{
Like \code{\link[=back_period]{back_period()}}, if multiple prices
correspond to a period-product pair, then the back price at a point in time
is always the first price for that product in the previous period. Unlike a
bilateral index, however, duplicated period-product pairs can have more
subtle implications for a multilateral index.
}
\examples{
price <- 1:10
quantity <- 10:1
period <- rep(1:5, 2)
product <- rep(letters[1:2], each = 5)

cumprod(tornqvist_geks(price, quantity, period, product)[[1]])

# Calculate the index over a rolling window.

(tg <- tornqvist_geks(price, quantity, period, product, window = 3))

# Use a movement splice to combine the indexes in each window.

splice_index(tg, 2)

# ... or use a mean splice.

splice_index(tg)

# Use all non-missing data.

quantity[2] <- NA
fisher_geks(price, quantity, period, product, na.rm = TRUE)

# Remove records with any missing data.

fg <- geks(balanced(fisher_index))
fg(price, quantity, period, product, na.rm = TRUE)

# Make a Jevons GEKS index.

jevons_geks <- geks(\(p1, p0, ..., na.rm) jevons_index(p1, p0, na.rm))
jevons_geks(price, quantity, period, product)

}
\references{
Balk, B. M. (2008). \emph{Price and Quantity Index Numbers}.
Cambridge University Press.

IMF, ILO, Eurostat, UNECE, OECD, and World Bank. (2020).
\emph{Consumer Price Index Manual: Concepts and Methods}.
International Monetary Fund.

Ivancic, L., Diewert, W. E., and Fox, K. J. (2011). Scanner data, time
aggregation and the construction of price indexes.
\emph{Journal of Econometrics}, 161(1): 24--35.
}
\seealso{
\code{GEKSIndex()} in the \pkg{IndexNumR} package for an implementation of the
GEKS index with more options.

Other price index functions: 
\code{\link{index_weights}()},
\code{\link{price_indexes}},
\code{\link{splice_index}()}
}
\concept{price index functions}
