/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Pmodule.h" 

#include "H5private.h"   
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5Fprivate.h"  
#include "H5Iprivate.h"  
#include "H5MMprivate.h" 
#include "H5Ppkg.h"      
#include "H5VMprivate.h" 

#define H5D_ACS_DATA_CACHE_NUM_SLOTS_SIZE sizeof(size_t)
#define H5D_ACS_DATA_CACHE_NUM_SLOTS_DEF  H5D_CHUNK_CACHE_NSLOTS_DEFAULT
#define H5D_ACS_DATA_CACHE_NUM_SLOTS_ENC  H5P__encode_chunk_cache_nslots
#define H5D_ACS_DATA_CACHE_NUM_SLOTS_DEC  H5P__decode_chunk_cache_nslots

#define H5D_ACS_DATA_CACHE_BYTE_SIZE_SIZE sizeof(size_t)
#define H5D_ACS_DATA_CACHE_BYTE_SIZE_DEF  H5D_CHUNK_CACHE_NBYTES_DEFAULT
#define H5D_ACS_DATA_CACHE_BYTE_SIZE_ENC  H5P__encode_chunk_cache_nbytes
#define H5D_ACS_DATA_CACHE_BYTE_SIZE_DEC  H5P__decode_chunk_cache_nbytes

#define H5D_ACS_PREEMPT_READ_CHUNKS_SIZE sizeof(double)
#define H5D_ACS_PREEMPT_READ_CHUNKS_DEF  H5D_CHUNK_CACHE_W0_DEFAULT
#define H5D_ACS_PREEMPT_READ_CHUNKS_ENC  H5P__encode_double
#define H5D_ACS_PREEMPT_READ_CHUNKS_DEC  H5P__decode_double

#define H5D_ACS_VDS_VIEW_SIZE sizeof(H5D_vds_view_t)
#define H5D_ACS_VDS_VIEW_DEF  H5D_VDS_LAST_AVAILABLE
#define H5D_ACS_VDS_VIEW_ENC  H5P__dacc_vds_view_enc
#define H5D_ACS_VDS_VIEW_DEC  H5P__dacc_vds_view_dec

#define H5D_ACS_VDS_PRINTF_GAP_SIZE sizeof(hsize_t)
#define H5D_ACS_VDS_PRINTF_GAP_DEF  (hsize_t)0
#define H5D_ACS_VDS_PRINTF_GAP_ENC  H5P__encode_hsize_t
#define H5D_ACS_VDS_PRINTF_GAP_DEC  H5P__decode_hsize_t

#define H5D_ACS_VDS_PREFIX_SIZE  sizeof(char *)
#define H5D_ACS_VDS_PREFIX_DEF   NULL 
#define H5D_ACS_VDS_PREFIX_SET   H5P__dapl_vds_file_pref_set
#define H5D_ACS_VDS_PREFIX_GET   H5P__dapl_vds_file_pref_get
#define H5D_ACS_VDS_PREFIX_ENC   H5P__dapl_vds_file_pref_enc
#define H5D_ACS_VDS_PREFIX_DEC   H5P__dapl_vds_file_pref_dec
#define H5D_ACS_VDS_PREFIX_DEL   H5P__dapl_vds_file_pref_del
#define H5D_ACS_VDS_PREFIX_COPY  H5P__dapl_vds_file_pref_copy
#define H5D_ACS_VDS_PREFIX_CMP   H5P__dapl_vds_file_pref_cmp
#define H5D_ACS_VDS_PREFIX_CLOSE H5P__dapl_vds_file_pref_close

#define H5D_ACS_APPEND_FLUSH_SIZE sizeof(H5D_append_flush_t)
#define H5D_ACS_APPEND_FLUSH_DEF                                                                             \
    {                                                                                                        \
        0, {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0}, \
            NULL, NULL                                                                                       \
    }

#define H5D_ACS_EFILE_PREFIX_SIZE  sizeof(char *)
#define H5D_ACS_EFILE_PREFIX_DEF   NULL 
#define H5D_ACS_EFILE_PREFIX_SET   H5P__dapl_efile_pref_set
#define H5D_ACS_EFILE_PREFIX_GET   H5P__dapl_efile_pref_get
#define H5D_ACS_EFILE_PREFIX_ENC   H5P__dapl_efile_pref_enc
#define H5D_ACS_EFILE_PREFIX_DEC   H5P__dapl_efile_pref_dec
#define H5D_ACS_EFILE_PREFIX_DEL   H5P__dapl_efile_pref_del
#define H5D_ACS_EFILE_PREFIX_COPY  H5P__dapl_efile_pref_copy
#define H5D_ACS_EFILE_PREFIX_CMP   H5P__dapl_efile_pref_cmp
#define H5D_ACS_EFILE_PREFIX_CLOSE H5P__dapl_efile_pref_close

#define H5D_ACS_USE_TREE_SIZE sizeof(bool)
#define H5D_ACS_USE_TREE_DEF  true
#define H5D_ACS_USE_TREE_ENC  H5P__encode_bool
#define H5D_ACS_USE_TREE_DEC  H5P__decode_bool

static herr_t H5P__dacc_reg_prop(H5P_genclass_t *pclass);
static herr_t H5P__encode_chunk_cache_nslots(const void *value, void **_pp, size_t *size);
static herr_t H5P__decode_chunk_cache_nslots(const void **_pp, void *_value);
static herr_t H5P__encode_chunk_cache_nbytes(const void *value, void **_pp, size_t *size);
static herr_t H5P__decode_chunk_cache_nbytes(const void **_pp, void *_value);

static herr_t H5P__dacc_vds_view_enc(const void *value, void **pp, size_t *size);
static herr_t H5P__dacc_vds_view_dec(const void **pp, void *value);
static herr_t H5P__dapl_vds_file_pref_set(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dapl_vds_file_pref_get(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dapl_vds_file_pref_enc(const void *value, void **_pp, size_t *size);
static herr_t H5P__dapl_vds_file_pref_dec(const void **_pp, void *value);
static herr_t H5P__dapl_vds_file_pref_del(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dapl_vds_file_pref_copy(const char *name, size_t size, void *value);
static int    H5P__dapl_vds_file_pref_cmp(const void *value1, const void *value2, size_t size);
static herr_t H5P__dapl_vds_file_pref_close(const char *name, size_t size, void *value);

static herr_t H5P__dapl_efile_pref_set(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dapl_efile_pref_get(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dapl_efile_pref_enc(const void *value, void **_pp, size_t *size);
static herr_t H5P__dapl_efile_pref_dec(const void **_pp, void *value);
static herr_t H5P__dapl_efile_pref_del(hid_t prop_id, const char *name, size_t size, void *value);
static herr_t H5P__dapl_efile_pref_copy(const char *name, size_t size, void *value);
static int    H5P__dapl_efile_pref_cmp(const void *value1, const void *value2, size_t size);
static herr_t H5P__dapl_efile_pref_close(const char *name, size_t size, void *value);

const H5P_libclass_t H5P_CLS_DACC[1] = {{
    "dataset access",        
    H5P_TYPE_DATASET_ACCESS, 

    &H5P_CLS_LINK_ACCESS_g,       
    &H5P_CLS_DATASET_ACCESS_g,    
    &H5P_CLS_DATASET_ACCESS_ID_g, 
    &H5P_LST_DATASET_ACCESS_ID_g, 
    H5P__dacc_reg_prop,           

    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL, 
    NULL  
}};

static const H5D_append_flush_t H5D_def_append_flush_g =
    H5D_ACS_APPEND_FLUSH_DEF; 
static const char *H5D_def_efile_prefix_g =
    H5D_ACS_EFILE_PREFIX_DEF;                                     
static const char *H5D_def_vds_prefix_g = H5D_ACS_VDS_PREFIX_DEF; 
static const bool  H5D_def_tree_g = H5D_ACS_USE_TREE_DEF; 

static herr_t
H5P__dacc_reg_prop(H5P_genclass_t *pclass)
{
    size_t rdcc_nslots = H5D_ACS_DATA_CACHE_NUM_SLOTS_DEF;    
    size_t rdcc_nbytes = H5D_ACS_DATA_CACHE_BYTE_SIZE_DEF;    
    double rdcc_w0     = H5D_ACS_PREEMPT_READ_CHUNKS_DEF;     
    H5D_vds_view_t virtual_view = H5D_ACS_VDS_VIEW_DEF;       
    hsize_t        printf_gap   = H5D_ACS_VDS_PRINTF_GAP_DEF; 
    herr_t         ret_value    = SUCCEED;                    

    FUNC_ENTER_PACKAGE

    
    if (H5P__register_real(pclass, H5D_ACS_DATA_CACHE_NUM_SLOTS_NAME, H5D_ACS_DATA_CACHE_NUM_SLOTS_SIZE,
                           &rdcc_nslots, NULL, NULL, NULL, H5D_ACS_DATA_CACHE_NUM_SLOTS_ENC,
                           H5D_ACS_DATA_CACHE_NUM_SLOTS_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_ACS_DATA_CACHE_BYTE_SIZE_NAME, H5D_ACS_DATA_CACHE_BYTE_SIZE_SIZE,
                           &rdcc_nbytes, NULL, NULL, NULL, H5D_ACS_DATA_CACHE_BYTE_SIZE_ENC,
                           H5D_ACS_DATA_CACHE_BYTE_SIZE_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_ACS_PREEMPT_READ_CHUNKS_NAME, H5D_ACS_PREEMPT_READ_CHUNKS_SIZE,
                           &rdcc_w0, NULL, NULL, NULL, H5D_ACS_PREEMPT_READ_CHUNKS_ENC,
                           H5D_ACS_PREEMPT_READ_CHUNKS_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_ACS_VDS_VIEW_NAME, H5D_ACS_VDS_VIEW_SIZE, &virtual_view, NULL, NULL,
                           NULL, H5D_ACS_VDS_VIEW_ENC, H5D_ACS_VDS_VIEW_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_ACS_VDS_PRINTF_GAP_NAME, H5D_ACS_VDS_PRINTF_GAP_SIZE, &printf_gap,
                           NULL, NULL, NULL, H5D_ACS_VDS_PRINTF_GAP_ENC, H5D_ACS_VDS_PRINTF_GAP_DEC, NULL,
                           NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_ACS_VDS_PREFIX_NAME, H5D_ACS_VDS_PREFIX_SIZE, &H5D_def_vds_prefix_g,
                           NULL, H5D_ACS_VDS_PREFIX_SET, H5D_ACS_VDS_PREFIX_GET, H5D_ACS_VDS_PREFIX_ENC,
                           H5D_ACS_VDS_PREFIX_DEC, H5D_ACS_VDS_PREFIX_DEL, H5D_ACS_VDS_PREFIX_COPY,
                           H5D_ACS_VDS_PREFIX_CMP, H5D_ACS_VDS_PREFIX_CLOSE) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    
    if (H5P__register_real(pclass, H5D_ACS_APPEND_FLUSH_NAME, H5D_ACS_APPEND_FLUSH_SIZE,
                           &H5D_def_append_flush_g, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_ACS_EFILE_PREFIX_NAME, H5D_ACS_EFILE_PREFIX_SIZE,
                           &H5D_def_efile_prefix_g, NULL, H5D_ACS_EFILE_PREFIX_SET, H5D_ACS_EFILE_PREFIX_GET,
                           H5D_ACS_EFILE_PREFIX_ENC, H5D_ACS_EFILE_PREFIX_DEC, H5D_ACS_EFILE_PREFIX_DEL,
                           H5D_ACS_EFILE_PREFIX_COPY, H5D_ACS_EFILE_PREFIX_CMP,
                           H5D_ACS_EFILE_PREFIX_CLOSE) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

    
    if (H5P__register_real(pclass, H5D_ACS_USE_TREE_NAME, H5D_ACS_USE_TREE_SIZE, &H5D_def_tree_g, NULL, NULL,
                           NULL, H5D_ACS_USE_TREE_ENC, H5D_ACS_USE_TREE_DEC, NULL, NULL, NULL, NULL) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTINSERT, FAIL, "can't insert property into class");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dapl_vds_file_pref_set(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                            size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);

    
    *(char **)value = H5MM_xstrdup(*(const char **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dapl_vds_file_pref_get(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                            size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);

    
    *(char **)value = H5MM_xstrdup(*(const char **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dapl_vds_file_pref_enc(const void *value, void **_pp, size_t *size)
{
    const char *vds_file_pref = *(const char *const *)value;
    uint8_t   **pp            = (uint8_t **)_pp;
    size_t      len           = 0;
    uint64_t    enc_value;
    unsigned    enc_size;

    FUNC_ENTER_PACKAGE_NOERR

    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));

    
    if (NULL != vds_file_pref)
        len = strlen(vds_file_pref);

    enc_value = (uint64_t)len;
    enc_size  = H5VM_limit_enc_size(enc_value);
    assert(enc_size < 256);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)enc_size;
        UINT64ENCODE_VAR(*pp, enc_value, enc_size);

        
        if (NULL != vds_file_pref) {
            H5MM_memcpy(*(char **)pp, vds_file_pref, len);
            *pp += len;
        } 
    }     

    *size += (1 + enc_size);
    if (NULL != vds_file_pref)
        *size += len;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dapl_vds_file_pref_dec(const void **_pp, void *_value)
{
    char          **vds_file_pref = (char **)_value;
    const uint8_t **pp            = (const uint8_t **)_pp;
    size_t          len;
    uint64_t        enc_value; 
    unsigned        enc_size;  
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(pp);
    assert(*pp);
    assert(vds_file_pref);
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));

    
    enc_size = *(*pp)++;
    assert(enc_size < 256);

    
    UINT64DECODE_VAR(*pp, enc_value, enc_size);
    len = (size_t)enc_value;

    if (0 != len) {
        
        if (NULL == (*vds_file_pref = (char *)H5MM_malloc(len + 1)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTINIT, FAIL, "memory allocation failed for prefix");
        strncpy(*vds_file_pref, *(const char **)pp, len);
        (*vds_file_pref)[len] = '\0';

        *pp += len;
    } 
    else
        *vds_file_pref = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dapl_vds_file_pref_del(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                            size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(value);

    H5MM_xfree(*(void **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dapl_vds_file_pref_copy(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(value);

    *(char **)value = H5MM_xstrdup(*(const char **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static int
H5P__dapl_vds_file_pref_cmp(const void *value1, const void *value2, size_t H5_ATTR_UNUSED size)
{
    const char *pref1     = *(const char *const *)value1;
    const char *pref2     = *(const char *const *)value2;
    int         ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    if (NULL == pref1 && NULL != pref2)
        HGOTO_DONE(1);
    if (NULL != pref1 && NULL == pref2)
        HGOTO_DONE(-1);
    if (NULL != pref1 && NULL != pref2)
        ret_value = strcmp(pref1, pref2);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dapl_vds_file_pref_close(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(value);

    H5MM_xfree(*(void **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dapl_efile_pref_set(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);

    
    *(char **)value = H5MM_xstrdup(*(const char **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dapl_efile_pref_get(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(value);

    
    *(char **)value = H5MM_xstrdup(*(const char **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dapl_efile_pref_enc(const void *value, void **_pp, size_t *size)
{
    const char *efile_pref = *(const char *const *)value;
    uint8_t   **pp         = (uint8_t **)_pp;
    size_t      len        = 0;
    uint64_t    enc_value;
    unsigned    enc_size;

    FUNC_ENTER_PACKAGE_NOERR

    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));

    
    if (NULL != efile_pref)
        len = strlen(efile_pref);

    enc_value = (uint64_t)len;
    enc_size  = H5VM_limit_enc_size(enc_value);
    assert(enc_size < 256);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)enc_size;
        UINT64ENCODE_VAR(*pp, enc_value, enc_size);

        
        if (NULL != efile_pref) {
            H5MM_memcpy(*(char **)pp, efile_pref, len);
            *pp += len;
        } 
    }     

    *size += (1 + enc_size);
    if (NULL != efile_pref)
        *size += len;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dapl_efile_pref_dec(const void **_pp, void *_value)
{
    char          **efile_pref = (char **)_value;
    const uint8_t **pp         = (const uint8_t **)_pp;
    size_t          len;
    uint64_t        enc_value; 
    unsigned        enc_size;  
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_PACKAGE

    assert(pp);
    assert(*pp);
    assert(efile_pref);
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));

    
    enc_size = *(*pp)++;
    assert(enc_size < 256);

    
    UINT64DECODE_VAR(*pp, enc_value, enc_size);
    len = (size_t)enc_value;

    if (0 != len) {
        
        if (NULL == (*efile_pref = (char *)H5MM_malloc(len + 1)))
            HGOTO_ERROR(H5E_RESOURCE, H5E_CANTINIT, FAIL, "memory allocation failed for prefix");
        strncpy(*efile_pref, *(const char **)pp, len);
        (*efile_pref)[len] = '\0';

        *pp += len;
    } 
    else
        *efile_pref = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dapl_efile_pref_del(hid_t H5_ATTR_UNUSED prop_id, const char H5_ATTR_UNUSED *name,
                         size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(value);

    H5MM_xfree(*(void **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dapl_efile_pref_copy(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(value);

    *(char **)value = H5MM_xstrdup(*(const char **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static int
H5P__dapl_efile_pref_cmp(const void *value1, const void *value2, size_t H5_ATTR_UNUSED size)
{
    const char *pref1     = *(const char *const *)value1;
    const char *pref2     = *(const char *const *)value2;
    int         ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    if (NULL == pref1 && NULL != pref2)
        HGOTO_DONE(1);
    if (NULL != pref1 && NULL == pref2)
        HGOTO_DONE(-1);
    if (NULL != pref1 && NULL != pref2)
        ret_value = strcmp(pref1, pref2);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5P__dapl_efile_pref_close(const char H5_ATTR_UNUSED *name, size_t H5_ATTR_UNUSED size, void *value)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(value);

    H5MM_xfree(*(void **)value);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_chunk_cache(hid_t dapl_id, size_t rdcc_nslots, size_t rdcc_nbytes, double rdcc_w0)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (rdcc_w0 > 1.0)
        HGOTO_ERROR(
            H5E_ARGS, H5E_BADVALUE, FAIL,
            "raw data cache w0 value must be between 0.0 and 1.0 inclusive, or H5D_CHUNK_CACHE_W0_DEFAULT");

    
    if (NULL == (plist = H5P_object_verify(dapl_id, H5P_DATASET_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5D_ACS_DATA_CACHE_NUM_SLOTS_NAME, &rdcc_nslots) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set data cache number of chunks");
    if (H5P_set(plist, H5D_ACS_DATA_CACHE_BYTE_SIZE_NAME, &rdcc_nbytes) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set data cache byte size");
    if (H5P_set(plist, H5D_ACS_PREEMPT_READ_CHUNKS_NAME, &rdcc_w0) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set preempt read chunks");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_chunk_cache(hid_t dapl_id, size_t *rdcc_nslots , size_t *rdcc_nbytes ,
                   double *rdcc_w0 )
{
    H5P_genplist_t *plist;               
    H5P_genplist_t *def_plist;           
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(dapl_id, H5P_DATASET_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (NULL == (def_plist = (H5P_genplist_t *)H5I_object(H5P_FILE_ACCESS_DEFAULT)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for default fapl ID");

    
    if (rdcc_nslots) {
        if (H5P_get(plist, H5D_ACS_DATA_CACHE_NUM_SLOTS_NAME, rdcc_nslots) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get data cache number of slots");
        if (*rdcc_nslots == H5D_CHUNK_CACHE_NSLOTS_DEFAULT)
            if (H5P_get(def_plist, H5F_ACS_DATA_CACHE_NUM_SLOTS_NAME, rdcc_nslots) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get default data cache number of slots");
    } 
    if (rdcc_nbytes) {
        if (H5P_get(plist, H5D_ACS_DATA_CACHE_BYTE_SIZE_NAME, rdcc_nbytes) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get data cache byte size");
        if (*rdcc_nbytes == H5D_CHUNK_CACHE_NBYTES_DEFAULT)
            if (H5P_get(def_plist, H5F_ACS_DATA_CACHE_BYTE_SIZE_NAME, rdcc_nbytes) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get default data cache byte size");
    } 
    if (rdcc_w0) {
        if (H5P_get(plist, H5D_ACS_PREEMPT_READ_CHUNKS_NAME, rdcc_w0) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get preempt read chunks");
        if (*rdcc_w0 < 0)
            if (H5P_get(def_plist, H5F_ACS_PREEMPT_READ_CHUNKS_NAME, rdcc_w0) < 0)
                HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get default preempt read chunks");
    } 

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5P__encode_chunk_cache_nslots(const void *value, void **_pp, size_t *size)
{
    uint64_t  enc_value = 0; 
    uint8_t **pp        = (uint8_t **)_pp;
    unsigned  enc_size; 

    FUNC_ENTER_PACKAGE_NOERR

    
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    assert(size);

    
    if (*(const size_t *)value == H5D_ACS_DATA_CACHE_NUM_SLOTS_DEF) {
        enc_size = 0;
        *size += 1;
    } 
    else {
        enc_value = (uint64_t) * (const size_t *)value;
        enc_size  = H5VM_limit_enc_size(enc_value);
        assert(enc_size > 0);
        *size += (1 + enc_size);
    } 

    assert(enc_size < 256);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)enc_size;

        
        if (enc_size != 0) {
            UINT64ENCODE_VAR(*pp, enc_value, enc_size);
        } 
    }     

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__decode_chunk_cache_nslots(const void **_pp, void *_value)
{
    size_t         *value = (size_t *)_value; 
    const uint8_t **pp    = (const uint8_t **)_pp;
    uint64_t        enc_value; 
    unsigned        enc_size;  

    FUNC_ENTER_PACKAGE_NOERR

    
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    assert(pp);
    assert(*pp);
    assert(value);

    
    enc_size = *(*pp)++;
    assert(enc_size < 256);

    
    if (enc_size == 0)
        *value = H5D_ACS_DATA_CACHE_NUM_SLOTS_DEF;
    else {
        
        UINT64DECODE_VAR(*pp, enc_value, enc_size);
        H5_CHECKED_ASSIGN(*value, uint64_t, enc_value, size_t);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__encode_chunk_cache_nbytes(const void *value, void **_pp, size_t *size)
{
    uint64_t  enc_value = 0; 
    uint8_t **pp        = (uint8_t **)_pp;
    unsigned  enc_size; 

    FUNC_ENTER_PACKAGE_NOERR

    
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    assert(size);

    
    if (*(const size_t *)value == H5D_ACS_DATA_CACHE_BYTE_SIZE_DEF) {
        enc_size = 0;
        *size += 1;
    } 
    else {
        enc_value = (uint64_t) * (const size_t *)value;
        enc_size  = H5VM_limit_enc_size(enc_value);
        assert(enc_size > 0);
        *size += (1 + enc_size);
    } 

    assert(enc_size < 256);

    if (NULL != *pp) {
        
        *(*pp)++ = (uint8_t)enc_size;

        
        if (enc_size != 0) {
            UINT64ENCODE_VAR(*pp, enc_value, enc_size);
        } 
    }     

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__decode_chunk_cache_nbytes(const void **_pp, void *_value)
{
    size_t         *value = (size_t *)_value; 
    const uint8_t **pp    = (const uint8_t **)_pp;
    uint64_t        enc_value; 
    unsigned        enc_size;  

    FUNC_ENTER_PACKAGE_NOERR

    
    HDcompile_assert(sizeof(size_t) <= sizeof(uint64_t));
    assert(pp);
    assert(*pp);
    assert(value);

    
    enc_size = *(*pp)++;
    assert(enc_size < 256);

    
    if (enc_size == 0)
        *value = H5D_ACS_DATA_CACHE_BYTE_SIZE_DEF;
    else {
        
        UINT64DECODE_VAR(*pp, enc_value, enc_size);
        H5_CHECKED_ASSIGN(*value, uint64_t, enc_value, size_t);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_virtual_view(hid_t plist_id, H5D_vds_view_t view)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if ((view != H5D_VDS_FIRST_MISSING) && (view != H5D_VDS_LAST_AVAILABLE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a valid bounds option");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5D_ACS_VDS_VIEW_NAME, &view) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_virtual_view(hid_t plist_id, H5D_vds_view_t *view )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (view)
        if (H5P_get(plist, H5D_ACS_VDS_VIEW_NAME, view) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");

done:
    FUNC_LEAVE_API(ret_value)
} 

static herr_t
H5P__dacc_vds_view_enc(const void *value, void **_pp, size_t *size)
{
    const H5D_vds_view_t *view = (const H5D_vds_view_t *)value; 
    uint8_t             **pp   = (uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(view);
    assert(size);

    if (NULL != *pp)
        
        *(*pp)++ = (uint8_t)*view;

    
    (*size)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5P__dacc_vds_view_dec(const void **_pp, void *_value)
{
    H5D_vds_view_t *view = (H5D_vds_view_t *)_value;
    const uint8_t **pp   = (const uint8_t **)_pp;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(pp);
    assert(*pp);
    assert(view);

    
    *view = (H5D_vds_view_t) * (*pp)++;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5Pset_virtual_printf_gap(hid_t plist_id, hsize_t gap_size)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (gap_size == HSIZE_UNDEF)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "not a valid Rprintf gap size");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5D_ACS_VDS_PRINTF_GAP_NAME, &gap_size) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "unable to set value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_virtual_printf_gap(hid_t plist_id, hsize_t *gap_size )
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (gap_size)
        if (H5P_get(plist, H5D_ACS_VDS_PRINTF_GAP_NAME, gap_size) < 0)
            HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "unable to get value");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_append_flush(hid_t plist_id, unsigned ndims, const hsize_t *boundary, H5D_append_cb_t func,
                    void *udata)
{
    H5P_genplist_t    *plist;               
    H5D_append_flush_t info;                
    unsigned           u;                   
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (0 == ndims)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "dimensionality cannot be zero");
    if (ndims > H5S_MAX_RANK)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "dimensionality is too large");
    if (!boundary)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "no boundary dimensions specified");

    
    if (!func && udata)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "callback is NULL while user data is not");

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    info.ndims = ndims;
    info.func  = func;
    info.udata = udata;

    memset(info.boundary, 0, sizeof(info.boundary));
    
    for (u = 0; u < ndims; u++) {
        if (boundary[u] != (boundary[u] & 0xffffffff)) 
            HGOTO_ERROR(H5E_ARGS, H5E_BADRANGE, FAIL, "all boundary dimensions must be less than 2^32");
        info.boundary[u] = boundary[u]; 
    }                                   

    
    if (H5P_set(plist, H5D_ACS_APPEND_FLUSH_NAME, &info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set append flush");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_append_flush(hid_t plist_id, unsigned ndims, hsize_t boundary[], H5D_append_cb_t *func ,
                    void **udata )
{
    H5P_genplist_t    *plist; 
    H5D_append_flush_t info;
    unsigned           u;                   
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_get(plist, H5D_ACS_APPEND_FLUSH_NAME, &info) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get object flush callback");

    
    if (boundary) {
        memset(boundary, 0, ndims * sizeof(hsize_t));
        if (info.ndims > 0)
            for (u = 0; u < info.ndims && u < ndims; u++)
                boundary[u] = info.boundary[u];
    } 
    if (func)
        *func = info.func;
    if (udata)
        *udata = info.udata;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_efile_prefix(hid_t plist_id, const char *prefix)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5D_ACS_EFILE_PREFIX_NAME, &prefix) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set prefix info");

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Pget_efile_prefix(hid_t plist_id, char *prefix , size_t size)
{
    H5P_genplist_t *plist;     
    char           *my_prefix; 
    size_t          len;       
    ssize_t         ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_ACS_EFILE_PREFIX_NAME, &my_prefix) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get external file prefix");

    
    if (my_prefix) {
        
        len = strlen(my_prefix);
        if (prefix) {
            strncpy(prefix, my_prefix, size);
            if (len >= size)
                prefix[size - 1] = '\0';
        } 
    }     
    else
        len = 0;

    
    ret_value = (ssize_t)len;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_virtual_prefix(hid_t plist_id, const char *prefix)
{
    H5P_genplist_t *plist;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_ACCESS, false)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_set(plist, H5D_ACS_VDS_PREFIX_NAME, &prefix) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set prefix info");

done:
    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5Pget_virtual_prefix(hid_t plist_id, char *prefix , size_t size)
{
    H5P_genplist_t *plist;     
    char           *my_prefix; 
    size_t          len;       
    ssize_t         ret_value; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (plist = H5P_object_verify(plist_id, H5P_DATASET_ACCESS, true)))
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    
    if (H5P_peek(plist, H5D_ACS_VDS_PREFIX_NAME, &my_prefix) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get vds file prefix");

    
    if (my_prefix) {
        
        len = strlen(my_prefix);
        if (prefix) {
            strncpy(prefix, my_prefix, size);
            if (len >= size)
                prefix[size - 1] = '\0';
        } 
    }     
    else
        len = 0;

    
    ret_value = (ssize_t)len;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pget_virtual_spatial_tree(hid_t dcpl_id, bool *use_tree)
{
    bool            setting   = false;
    H5P_genplist_t *plist     = NULL;
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    if (NULL == use_tree)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "receiving pointer cannot be NULL");

    plist = H5P_object_verify(dcpl_id, H5P_DATASET_ACCESS, true);
    if (NULL == plist)
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    if (H5P_peek(plist, H5D_ACS_USE_TREE_NAME, &setting) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get dset use spatial tree flag value");

    *use_tree = setting;

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5Pset_virtual_spatial_tree(hid_t dapl_id, bool use_tree)
{
    H5P_genplist_t *plist     = NULL;
    bool            prev_set  = false;
    herr_t          ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    plist = H5P_object_verify(dapl_id, H5P_DATASET_ACCESS, false);
    if (NULL == plist)
        HGOTO_ERROR(H5E_ID, H5E_BADID, FAIL, "can't find object for ID");

    if (H5P_peek(plist, H5D_ACS_USE_TREE_NAME, &prev_set) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTGET, FAIL, "can't get extant dset use spatial tree flag value");

    if (H5P_poke(plist, H5D_ACS_USE_TREE_NAME, &use_tree) < 0)
        HGOTO_ERROR(H5E_PLIST, H5E_CANTSET, FAIL, "can't set dset use spatial tree flag value");

done:
    FUNC_LEAVE_API(ret_value)
} 
