/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Dmodule.h" 

#include "H5private.h"   
#include "H5Dpkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MMprivate.h" 
#include "H5VMprivate.h" 
#include "H5WBprivate.h" 

static herr_t H5D__fill_release(H5D_fill_buf_info_t *fb_info);

H5FL_BLK_EXTERN(type_conv);

H5FL_BLK_DEFINE_STATIC(non_zero_fill);

H5FL_BLK_DEFINE_STATIC(zero_fill);

H5FL_EXTERN(H5S_sel_iter_t);

herr_t
H5D__fill(const void *fill, const H5T_t *fill_type, void *buf, const H5T_t *buf_type, H5S_t *space)
{
    H5S_sel_iter_t *mem_iter      = NULL;  
    bool            mem_iter_init = false; 
    H5WB_t         *elem_wb       = NULL;  
    uint8_t         elem_buf[H5T_ELEM_BUF_SIZE];     
    H5WB_t         *bkg_elem_wb = NULL;              
    uint8_t         bkg_elem_buf[H5T_ELEM_BUF_SIZE]; 
    uint8_t        *bkg_buf = NULL;                  
    uint8_t        *tmp_buf = NULL;                  
    size_t          dst_type_size;                   
    herr_t          ret_value = SUCCEED;             

    FUNC_ENTER_PACKAGE

    
    assert(fill_type);
    assert(buf);
    assert(buf_type);
    assert(space);

    
    if (!(H5S_has_extent(space)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "dataspace extent has not been set");

    
    dst_type_size = H5T_get_size(buf_type);

    
    if (fill == NULL) {
        void *elem_ptr; 

        
        if (NULL == (elem_wb = H5WB_wrap(elem_buf, sizeof(elem_buf))))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't wrap buffer");

        
        if (NULL == (elem_ptr = H5WB_actual_clear(elem_wb, dst_type_size)))
            HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't get actual buffer");

        
        if (H5S_select_fill(elem_ptr, dst_type_size, space, buf) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_CANTENCODE, FAIL, "filling selection failed");
    } 
    else {
        H5T_path_t *tpath;         
        size_t      src_type_size; 
        size_t      buf_size;      

        
        src_type_size = H5T_get_size(fill_type);

        
        buf_size = MAX(src_type_size, dst_type_size);

        
        if (NULL == (tpath = H5T_path_find(fill_type, buf_type)))
            HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL,
                        "unable to convert between src and dest datatype");

        
        if (true == H5T_detect_class(fill_type, H5T_VLEN, false)) {
            hsize_t nelmts; 

            
            nelmts = H5S_GET_SELECT_NPOINTS(space);
            H5_CHECK_OVERFLOW(nelmts, hsize_t, size_t);

            
            if (NULL == (tmp_buf = H5FL_BLK_MALLOC(type_conv, (size_t)nelmts * buf_size)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed");

            
            if (H5T_path_bkg(tpath) &&
                NULL == (bkg_buf = H5FL_BLK_CALLOC(type_conv, (size_t)nelmts * buf_size)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed");

            
            H5VM_array_fill(tmp_buf, fill, src_type_size, (size_t)nelmts);

            
            if (H5T_convert(tpath, fill_type, buf_type, (size_t)nelmts, (size_t)0, (size_t)0, tmp_buf,
                            bkg_buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "data type conversion failed");

            
            if (NULL == (mem_iter = H5FL_MALLOC(H5S_sel_iter_t)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't allocate memory selection iterator");

            
            if (H5S_select_iter_init(mem_iter, space, dst_type_size, 0) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                            "unable to initialize memory selection information");
            mem_iter_init = true;

            
            if (H5D__scatter_mem(tmp_buf, mem_iter, (size_t)nelmts, buf ) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_READERROR, FAIL, "scatter failed");
        } 
        else {
            const uint8_t *fill_buf; 

            
            if (!H5T_path_noop(tpath)) {
                void *elem_ptr;       
                void *bkg_ptr = NULL; 

                
                if (NULL == (elem_wb = H5WB_wrap(elem_buf, sizeof(elem_buf))))
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't wrap buffer");

                
                if (NULL == (elem_ptr = H5WB_actual(elem_wb, buf_size)))
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't get actual buffer");

                
                H5MM_memcpy(elem_ptr, fill, src_type_size);

                
                if (H5T_path_bkg(tpath)) {
                    
                    if (NULL == (bkg_elem_wb = H5WB_wrap(bkg_elem_buf, sizeof(bkg_elem_buf))))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL, "can't wrap buffer");

                    
                    if (NULL == (bkg_ptr = H5WB_actual_clear(bkg_elem_wb, buf_size)))
                        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "can't get actual buffer");
                } 

                
                if (H5T_convert(tpath, fill_type, buf_type, (size_t)1, (size_t)0, (size_t)0, elem_ptr,
                                bkg_ptr) < 0)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "data type conversion failed");

                
                fill_buf = (const uint8_t *)elem_ptr;
            } 
            else
                fill_buf = (const uint8_t *)fill;

            
            if (H5S_select_fill(fill_buf, dst_type_size, space, buf) < 0)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTENCODE, FAIL, "filling selection failed");
        } 
    }     

done:
    if (mem_iter_init && H5S_SELECT_ITER_RELEASE(mem_iter) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release selection iterator");
    if (mem_iter)
        mem_iter = H5FL_FREE(H5S_sel_iter_t, mem_iter);
    if (tmp_buf)
        tmp_buf = H5FL_BLK_FREE(type_conv, tmp_buf);
    if (elem_wb && H5WB_unwrap(elem_wb) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close wrapped buffer");
    if (bkg_elem_wb && H5WB_unwrap(bkg_elem_wb) < 0)
        HDONE_ERROR(H5E_DATASET, H5E_CLOSEERROR, FAIL, "can't close wrapped buffer");
    if (bkg_buf)
        bkg_buf = H5FL_BLK_FREE(type_conv, bkg_buf);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__fill_init(H5D_fill_buf_info_t *fb_info, void *caller_fill_buf, H5MM_allocate_t alloc_func,
               void *alloc_info, H5MM_free_t free_func, void *free_info, const H5O_fill_t *fill,
               const H5T_t *dset_type, size_t total_nelmts, size_t max_buf_size)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(fb_info);
    assert(fill);
    assert(dset_type);

    
    memset(fb_info, 0, sizeof(*fb_info));

    
    fb_info->fill            = fill;
    fb_info->file_type       = dset_type;
    fb_info->fill_alloc_func = alloc_func;
    fb_info->fill_alloc_info = alloc_info;
    fb_info->fill_free_func  = free_func;
    fb_info->fill_free_info  = free_info;

    
    if (fill->buf) {
        htri_t has_vlen_type; 

        
        if ((has_vlen_type = H5T_detect_class(dset_type, H5T_VLEN, false)) < 0)
            HGOTO_ERROR(H5E_DATASET, H5E_BADVALUE, FAIL, "unable to detect vlen datatypes?");
        fb_info->has_vlen_fill_type = (bool)has_vlen_type;

        
        if (fb_info->has_vlen_fill_type) {
            
            if (NULL == (fb_info->mem_type = H5T_copy(dset_type, H5T_COPY_TRANSIENT)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTCOPY, FAIL, "unable to copy file datatype");

            
            fb_info->mem_elmt_size = H5T_get_size(fb_info->mem_type);
            assert(fb_info->mem_elmt_size > 0);
            fb_info->file_elmt_size = H5T_get_size(dset_type);
            assert(fb_info->file_elmt_size == (size_t)fill->size);

            
            fb_info->max_elmt_size = MAX(fb_info->mem_elmt_size, fb_info->file_elmt_size);

            
            if (total_nelmts > 0)
                fb_info->elmts_per_buf = MIN(total_nelmts, MAX(1, (max_buf_size / fb_info->max_elmt_size)));
            else
                fb_info->elmts_per_buf = max_buf_size / fb_info->max_elmt_size;
            assert(fb_info->elmts_per_buf > 0);

            
            fb_info->fill_buf_size = MIN(max_buf_size, (fb_info->elmts_per_buf * fb_info->max_elmt_size));

            
            if (caller_fill_buf) {
                fb_info->fill_buf            = caller_fill_buf;
                fb_info->use_caller_fill_buf = true;
            } 
            else {
                if (alloc_func) {
                    
                    H5_BEFORE_USER_CB(FAIL)
                        {
                            fb_info->fill_buf = alloc_func(fb_info->fill_buf_size, alloc_info);
                        }
                    H5_AFTER_USER_CB(FAIL)
                }
                else
                    fb_info->fill_buf = H5FL_BLK_MALLOC(non_zero_fill, fb_info->fill_buf_size);
                if (NULL == fb_info->fill_buf)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed for fill buffer");
            } 

            
            if (NULL == (fb_info->fill_to_mem_tpath = H5T_path_find(dset_type, fb_info->mem_type)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                            "unable to convert between src and dst datatypes");

            
            if (NULL == (fb_info->mem_to_dset_tpath = H5T_path_find(fb_info->mem_type, dset_type)))
                HGOTO_ERROR(H5E_DATASET, H5E_CANTINIT, FAIL,
                            "unable to convert between src and dst datatypes");

            
            if (H5T_path_bkg(fb_info->fill_to_mem_tpath) || H5T_path_bkg(fb_info->mem_to_dset_tpath)) {
                
                
                if (H5T_path_bkg(fb_info->mem_to_dset_tpath))
                    fb_info->bkg_buf_size = fb_info->elmts_per_buf * fb_info->max_elmt_size;
                else
                    fb_info->bkg_buf_size = fb_info->max_elmt_size;

                
                if (NULL == (fb_info->bkg_buf = H5FL_BLK_MALLOC(type_conv, fb_info->bkg_buf_size)))
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed");
            } 
        }     
        else {
            
            assert(fill->size >= 0);
            fb_info->max_elmt_size = fb_info->file_elmt_size = fb_info->mem_elmt_size = (size_t)fill->size;

            
            if (total_nelmts > 0)
                fb_info->elmts_per_buf = MIN(total_nelmts, MAX(1, (max_buf_size / fb_info->max_elmt_size)));
            else
                fb_info->elmts_per_buf = max_buf_size / fb_info->max_elmt_size;
            assert(fb_info->elmts_per_buf > 0);

            
            fb_info->fill_buf_size = MIN(max_buf_size, fb_info->elmts_per_buf * fb_info->max_elmt_size);

            
            if (caller_fill_buf) {
                fb_info->fill_buf            = caller_fill_buf;
                fb_info->use_caller_fill_buf = true;
            } 
            else {
                if (alloc_func) {
                    
                    H5_BEFORE_USER_CB(FAIL)
                        {
                            fb_info->fill_buf = alloc_func(fb_info->fill_buf_size, alloc_info);
                        }
                    H5_AFTER_USER_CB(FAIL)
                }
                else
                    fb_info->fill_buf = H5FL_BLK_MALLOC(non_zero_fill, fb_info->fill_buf_size);
                if (NULL == fb_info->fill_buf)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed for fill buffer");
            } 

            
            H5VM_array_fill(fb_info->fill_buf, fill->buf, fb_info->max_elmt_size, fb_info->elmts_per_buf);
        }  
    }      
    else { 
        
        fb_info->max_elmt_size = fb_info->file_elmt_size = fb_info->mem_elmt_size = H5T_get_size(dset_type);
        assert(fb_info->max_elmt_size > 0);

        
        if (total_nelmts > 0)
            fb_info->elmts_per_buf = MIN(total_nelmts, MAX(1, (max_buf_size / fb_info->max_elmt_size)));
        else
            fb_info->elmts_per_buf = max_buf_size / fb_info->max_elmt_size;
        assert(fb_info->elmts_per_buf > 0);

        
        fb_info->fill_buf_size = MIN(max_buf_size, (fb_info->elmts_per_buf * fb_info->max_elmt_size));

        
        if (caller_fill_buf) {
            fb_info->fill_buf            = caller_fill_buf;
            fb_info->use_caller_fill_buf = true;

            memset(fb_info->fill_buf, 0, fb_info->fill_buf_size);
        } 
        else {
            if (alloc_func) {
                
                H5_BEFORE_USER_CB(FAIL)
                    {
                        fb_info->fill_buf = alloc_func(fb_info->fill_buf_size, alloc_info);
                    }
                H5_AFTER_USER_CB(FAIL)
                if (NULL == fb_info->fill_buf)
                    HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed for fill buffer");

                memset(fb_info->fill_buf, 0, fb_info->fill_buf_size);
            } 
            else {
                htri_t buf_avail = H5FL_BLK_AVAIL(
                    zero_fill,
                    fb_info->fill_buf_size); 
                assert(buf_avail != FAIL);

                
                if (!buf_avail)
                    fb_info->fill_buf = H5FL_BLK_CALLOC(zero_fill, fb_info->fill_buf_size);
                else
                    fb_info->fill_buf = H5FL_BLK_MALLOC(zero_fill, fb_info->fill_buf_size);
            } 
            if (fb_info->fill_buf == NULL)
                HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed for fill buffer");
        } 
    }     

done:
    
    if (ret_value < 0)
        if (H5D__fill_term(fb_info) < 0)
            HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "Can't release fill buffer info");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__fill_refill_vl(H5D_fill_buf_info_t *fb_info, size_t nelmts)
{
    herr_t ret_value = SUCCEED; 
    void  *buf       = NULL;    

    FUNC_ENTER_PACKAGE

    
    assert(fb_info);
    assert(fb_info->has_vlen_fill_type);
    assert(fb_info->fill_buf);

    
    H5MM_memcpy(fb_info->fill_buf, fb_info->fill->buf, fb_info->file_elmt_size);

    
    if (H5T_path_bkg(fb_info->fill_to_mem_tpath))
        memset(fb_info->bkg_buf, 0, fb_info->max_elmt_size);

    
    if (H5T_convert(fb_info->fill_to_mem_tpath, fb_info->file_type, fb_info->mem_type, (size_t)1, (size_t)0,
                    (size_t)0, fb_info->fill_buf, fb_info->bkg_buf) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "data type conversion failed");

    
    if (nelmts > 1)
        H5VM_array_fill((void *)((unsigned char *)fb_info->fill_buf + fb_info->mem_elmt_size),
                        fb_info->fill_buf, fb_info->mem_elmt_size, (nelmts - 1));

    
    if (H5T_path_bkg(fb_info->mem_to_dset_tpath))
        memset(fb_info->bkg_buf, 0, fb_info->bkg_buf_size);

    
    if (fb_info->fill_alloc_func) {
        
        H5_BEFORE_USER_CB(FAIL)
            {
                buf = fb_info->fill_alloc_func(fb_info->fill_buf_size, fb_info->fill_alloc_info);
            }
        H5_AFTER_USER_CB(FAIL)
    }
    else
        buf = H5FL_BLK_MALLOC(non_zero_fill, fb_info->fill_buf_size);
    if (!buf)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTALLOC, FAIL, "memory allocation failed for temporary fill buffer");

    H5MM_memcpy(buf, fb_info->fill_buf, fb_info->fill_buf_size);

    
    if (H5T_convert(fb_info->mem_to_dset_tpath, fb_info->mem_type, fb_info->file_type, nelmts, (size_t)0,
                    (size_t)0, fb_info->fill_buf, fb_info->bkg_buf) < 0)
        HGOTO_ERROR(H5E_DATASET, H5E_CANTCONVERT, FAIL, "data type conversion failed");

done:
    if (buf) {
        
        if (fb_info->fill->type) {
            if (H5T_vlen_reclaim_elmt(buf, fb_info->fill->type) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "can't reclaim vlen element");
        } 
        else {
            if (H5T_vlen_reclaim_elmt(buf, fb_info->mem_type) < 0)
                HDONE_ERROR(H5E_DATASET, H5E_CANTFREE, FAIL, "can't reclaim vlen element");
        } 

        
        if (fb_info->fill_free_func) {
            
            H5_BEFORE_USER_CB_NOERR(FAIL)
                {
                    fb_info->fill_free_func(buf, fb_info->fill_free_info);
                }
            H5_AFTER_USER_CB_NOERR(FAIL)
        }
        else
            buf = H5FL_BLK_FREE(non_zero_fill, buf);
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5D__fill_release(H5D_fill_buf_info_t *fb_info)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(fb_info);
    assert(fb_info->fill);

    
    if (!fb_info->use_caller_fill_buf && fb_info->fill_buf) {
        if (fb_info->fill_free_func) {
            
            H5_BEFORE_USER_CB_NOERR(FAIL)
                {
                    fb_info->fill_free_func(fb_info->fill_buf, fb_info->fill_free_info);
                }
            H5_AFTER_USER_CB_NOERR(FAIL)
        }
        else {
            if (fb_info->fill->buf)
                fb_info->fill_buf = H5FL_BLK_FREE(non_zero_fill, fb_info->fill_buf);
            else
                fb_info->fill_buf = H5FL_BLK_FREE(zero_fill, fb_info->fill_buf);
        } 
        fb_info->fill_buf = NULL;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5D__fill_term(H5D_fill_buf_info_t *fb_info)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(fb_info);

    
    H5D__fill_release(fb_info);

    
    if (fb_info->has_vlen_fill_type) {
        if (fb_info->mem_type)
            (void)H5T_close_real(fb_info->mem_type);
        if (fb_info->bkg_buf)
            fb_info->bkg_buf = H5FL_BLK_FREE(type_conv, fb_info->bkg_buf);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
