/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5ESmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5ESpkg.h"    

void
H5ES__list_append(H5ES_event_list_t *el, H5ES_event_t *ev)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(el);
    assert(ev);

    ev->next = NULL;

    
    if (NULL == el->tail)
        el->head = el->tail = ev;
    else {
        ev->prev       = el->tail;
        el->tail->next = ev;
        el->tail       = ev;
    } 

    
    el->count++;

    FUNC_LEAVE_NOAPI_VOID
} 

H5_ATTR_PURE size_t
H5ES__list_count(const H5ES_event_list_t *el)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(el);

    FUNC_LEAVE_NOAPI(el->count)
} 

int
H5ES__list_iterate(H5ES_event_list_t *el, H5_iter_order_t order, H5ES_list_iter_func_t cb, void *ctx)
{
    H5ES_event_t *ev;                       
    int           ret_value = H5_ITER_CONT; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(el);
    assert(cb);

    
    ev = (order == H5_ITER_DEC) ? el->tail : el->head;
    while (ev) {
        H5ES_event_t *tmp; 

        
        tmp = (order == H5_ITER_DEC) ? ev->prev : ev->next;

        
        if ((ret_value = (*cb)(ev, ctx)) != H5_ITER_CONT) {
            if (ret_value < 0)
                HERROR(H5E_EVENTSET, H5E_CANTNEXT, "iteration operator failed");
            break;
        } 

        
        ev = tmp;
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5ES__list_remove(H5ES_event_list_t *el, const H5ES_event_t *ev)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(el);
    assert(el->head);
    assert(ev);

    
    if (ev == el->head)
        el->head = ev->next;
    if (NULL != ev->next)
        ev->next->prev = ev->prev;
    if (NULL != ev->prev)
        ev->prev->next = ev->next;
    if (NULL == el->head)
        el->tail = NULL;

    
    el->count--;

    FUNC_LEAVE_NOAPI_VOID
} 
