/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 
#define H5G_TESTING    

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5Gpkg.h"      
#include "H5HLprivate.h" 
#include "H5Iprivate.h"  
#include "H5VLprivate.h" 

htri_t
H5G__is_empty_test(hid_t gid)
{
    H5G_t      *grp            = NULL;        
    htri_t      msg_exists     = false;       
    htri_t      linfo_exists   = false;       
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    htri_t      ret_value      = true;        

    FUNC_ENTER_PACKAGE

    
    if (NULL == (grp = (H5G_t *)H5VL_object_verify(gid, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    

    
    if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_LINK_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
    if (msg_exists > 0) {
        
        if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_STAB_ID)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
        if (msg_exists > 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "both symbol table and link messages found");

        HGOTO_DONE(false);
    } 

    
    if ((linfo_exists = H5O_msg_exists(&(grp->oloc), H5O_LINFO_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
    if (linfo_exists > 0) {
        H5O_linfo_t linfo; 

        
        if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_STAB_ID)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
        if (msg_exists > 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "both symbol table and link info messages found");

        
        if (H5G__obj_get_linfo(&(grp->oloc), &linfo) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADMESG, FAIL, "can't get link info");

        
        if (H5_addr_defined(linfo.fheap_addr))
            HGOTO_DONE(false);
        if (H5_addr_defined(linfo.name_bt2_addr))
            HGOTO_DONE(false);
        if (H5_addr_defined(linfo.corder_bt2_addr))
            HGOTO_DONE(false);

        
        if (linfo.nlinks > 0)
            HGOTO_DONE(false);
    } 

    

    
    if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_STAB_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
    if (msg_exists > 0) {
        H5O_stab_t stab;   
        hsize_t    nlinks; 

        
        if (linfo_exists > 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "both symbol table and link info messages found");
        if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_GINFO_ID)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
        if (msg_exists > 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "both symbol table and group info messages found");

        
        if (NULL == H5O_msg_read(&(grp->oloc), H5O_STAB_ID, &stab))
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "unable to read symbol table message");

        
        if (H5G__stab_count(&(grp->oloc), &nlinks) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "unable to count links");

        
        if (nlinks > 0)
            HGOTO_DONE(false);
    } 

done:
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5G__has_links_test(hid_t gid, unsigned *nmsgs)
{
    H5G_t      *grp            = NULL;        
    htri_t      msg_exists     = 0;           
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    htri_t      ret_value      = true;        

    FUNC_ENTER_PACKAGE

    
    if (NULL == (grp = (H5G_t *)H5VL_object_verify(gid, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_LINK_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
    if (msg_exists == 0)
        HGOTO_DONE(false);

    
    if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_STAB_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
    if (msg_exists > 0)
        HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "both symbol table and link messages found");

    
    if (nmsgs) {
        int msg_count; 

        
        if ((msg_count = H5O_msg_count(&(grp->oloc), H5O_LINK_ID)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTCOUNT, FAIL, "unable to count link messages");
        *nmsgs = (unsigned)msg_count;
    } 

done:
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5G__has_stab_test(hid_t gid)
{
    H5G_t      *grp            = NULL;        
    htri_t      msg_exists     = 0;           
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    htri_t      ret_value      = true;        

    FUNC_ENTER_PACKAGE

    
    if (NULL == (grp = (H5G_t *)H5VL_object_verify(gid, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_STAB_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
    if (msg_exists == 0)
        HGOTO_DONE(false);

    
    if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_LINK_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
    if (msg_exists > 0)
        HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "both symbol table and link messages found");

done:
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5G__is_new_dense_test(hid_t gid)
{
    H5G_t      *grp            = NULL;        
    htri_t      msg_exists     = 0;           
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    htri_t      ret_value      = true;        

    FUNC_ENTER_PACKAGE

    
    if (NULL == (grp = (H5G_t *)H5VL_object_verify(gid, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_STAB_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
    if (msg_exists > 0)
        HGOTO_DONE(false);

    
    if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_LINK_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
    if (msg_exists > 0)
        HGOTO_DONE(false);

    
    if ((msg_exists = H5O_msg_exists(&(grp->oloc), H5O_LINFO_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");
    if (msg_exists > 0) {
        H5O_linfo_t linfo; 

        
        if (H5G__obj_get_linfo(&(grp->oloc), &linfo) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADMESG, FAIL, "can't get link info");

        
        if (!H5_addr_defined(linfo.fheap_addr))
            HGOTO_DONE(false);
        if (!H5_addr_defined(linfo.name_bt2_addr))
            HGOTO_DONE(false);
    } 

done:
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__new_dense_info_test(hid_t gid, hsize_t *name_count, hsize_t *corder_count)
{
    H5B2_t     *bt2_name   = NULL;            
    H5B2_t     *bt2_corder = NULL;            
    H5O_linfo_t linfo;                        
    H5G_t      *grp            = NULL;        
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    herr_t      ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    if (NULL == (grp = (H5G_t *)H5VL_object_verify(gid, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    H5_BEGIN_TAG(grp->oloc.addr)

    
    if (H5G__obj_get_linfo(&(grp->oloc), &linfo) < 0)
        HGOTO_ERROR_TAG(H5E_SYM, H5E_BADMESG, FAIL, "can't get link info");

    
    if (!H5_addr_defined(linfo.fheap_addr))
        HGOTO_DONE_TAG(FAIL);
    if (!H5_addr_defined(linfo.name_bt2_addr))
        HGOTO_DONE_TAG(FAIL);

    
    if (NULL == (bt2_name = H5B2_open(grp->oloc.file, linfo.name_bt2_addr, NULL)))
        HGOTO_ERROR_TAG(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

    
    if (H5B2_get_nrec(bt2_name, name_count) < 0)
        HGOTO_ERROR_TAG(H5E_SYM, H5E_CANTCOUNT, FAIL, "unable to retrieve # of records from name index");

    
    if (H5_addr_defined(linfo.corder_bt2_addr)) {
        
        if (NULL == (bt2_corder = H5B2_open(grp->oloc.file, linfo.corder_bt2_addr, NULL)))
            HGOTO_ERROR_TAG(H5E_SYM, H5E_CANTOPENOBJ, FAIL,
                            "unable to open v2 B-tree for creation order index");

        
        if (H5B2_get_nrec(bt2_corder, corder_count) < 0)
            HGOTO_ERROR_TAG(H5E_SYM, H5E_CANTCOUNT, FAIL,
                            "unable to retrieve # of records from creation order index");
    } 
    else
        *corder_count = 0;

    
    H5_END_TAG

done:
    
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for name index");
    if (bt2_corder && H5B2_close(bt2_corder) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTCLOSEOBJ, FAIL, "can't close v2 B-tree for creation order index");
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__lheap_size_test(hid_t gid, size_t *lheap_size)
{
    H5G_t      *grp = NULL;                   
    H5O_stab_t  stab;                         
    H5CX_node_t api_ctx        = {{0}, NULL}; 
    bool        api_ctx_pushed = false;       
    herr_t      ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    if (NULL == (grp = (H5G_t *)H5VL_object_verify(gid, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    if (NULL == H5O_msg_read(&(grp->oloc), H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read symbol table message");

    
    if (H5HL_get_size(grp->oloc.file, stab.heap_addr, lheap_size) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGETSIZE, FAIL, "can't query local heap size");

done:
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__user_path_test(hid_t obj_id, char *user_path, size_t *user_path_len, unsigned *obj_hidden)
{
    void             *obj_ptr;                      
    const H5G_name_t *obj_path;                     
    H5CX_node_t       api_ctx        = {{0}, NULL}; 
    bool              api_ctx_pushed = false;       
    herr_t            ret_value      = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    assert(user_path_len);
    assert(obj_hidden);

    
    if (NULL == (obj_ptr = H5VL_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "can't get object for ID");

    
    if (H5CX_push(&api_ctx) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTSET, FAIL, "can't set API context");
    api_ctx_pushed = true;

    
    switch (H5I_get_type(obj_id)) {
        case H5I_GROUP:
            obj_path = H5G_nameof((H5G_t *)obj_ptr);
            break;

        case H5I_DATASET:
            obj_path = H5D_nameof((H5D_t *)obj_ptr);
            break;

        case H5I_DATATYPE:
            
            if (!H5T_is_named((H5T_t *)obj_ptr))
                HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a named datatype");

            obj_path = H5T_nameof((H5T_t *)obj_ptr);
            break;

        case H5I_MAP:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "maps not supported in native VOL connector");

        case H5I_UNINIT:
        case H5I_BADID:
        case H5I_FILE:
        case H5I_DATASPACE:
        case H5I_ATTR:
        case H5I_VFL:
        case H5I_VOL:
        case H5I_GENPROP_CLS:
        case H5I_GENPROP_LST:
        case H5I_ERROR_CLASS:
        case H5I_ERROR_MSG:
        case H5I_ERROR_STACK:
        case H5I_SPACE_SEL_ITER:
        case H5I_EVENTSET:
        case H5I_NTYPES:
        default:
            HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "unknown data object type");
    } 
    assert(obj_path);

    
    if (obj_path->user_path_r) {
        size_t len = H5RS_len(obj_path->user_path_r);

        
        if (user_path)
            strncpy(user_path, H5RS_get_str(obj_path->user_path_r), (len + 1));

        
        *user_path_len = len;

        
        *obj_hidden = obj_path->obj_hidden;
    } 
    else {
        *user_path_len = 0;
        *obj_hidden    = 0;
    } 

done:
    if (api_ctx_pushed && H5CX_pop(false) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTRESET, FAIL, "can't reset API context");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__verify_cached_stab_test(H5O_loc_t *grp_oloc, H5G_entry_t *ent)
{
    H5O_stab_t stab;                
    H5HL_t    *heap      = NULL;    
    herr_t     ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(grp_oloc->addr)

    
    if (ent->type != H5G_CACHED_STAB)
        HGOTO_ERROR(H5E_SYM, H5E_BADTYPE, FAIL, "symbol table information is not cached");

    
    if (NULL == H5O_msg_read(grp_oloc, H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_BADMESG, FAIL, "unable to read symbol table message");

    
    if ((ent->cache.stab.btree_addr != stab.btree_addr) || (ent->cache.stab.heap_addr != stab.heap_addr))
        HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "cached stab info does not match object header");

    
    if (H5B_valid(grp_oloc->file, H5B_SNODE, stab.btree_addr) < 0)
        HGOTO_ERROR(H5E_BTREE, H5E_NOTFOUND, FAIL, "b-tree address is invalid");

    
    if (NULL == (heap = H5HL_protect(grp_oloc->file, stab.heap_addr, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_HEAP, H5E_NOTFOUND, FAIL, "heap address is invalid");

done:
    
    if (heap && H5HL_unprotect(heap) < 0)
        HDONE_ERROR(H5E_SYM, H5E_PROTECT, FAIL, "unable to unprotect symbol table heap");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static int
H5G__verify_cached_stabs_test_cb(H5F_t *f, const void H5_ATTR_UNUSED *_lt_key, haddr_t addr,
                                 const void H5_ATTR_UNUSED *_rt_key, void H5_ATTR_UNUSED *udata)
{
    H5G_node_t *sn = NULL;
    H5O_loc_t   targ_oloc;
    H5O_t      *targ_oh = NULL;
    htri_t      stab_exists;
    H5O_stab_t  stab;
    unsigned    i;
    int         ret_value = H5_ITER_CONT;

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(H5_addr_defined(addr));

    
    if (NULL == (sn = (H5G_node_t *)H5AC_protect(f, H5AC_SNODE, addr, f, H5AC__READ_ONLY_FLAG)))
        HGOTO_ERROR(H5E_SYM, H5E_CANTLOAD, H5_ITER_ERROR, "unable to load symbol table node");

    
    
    targ_oloc.file         = f;
    targ_oloc.holding_file = false;

    
    for (i = 0; i < sn->nsyms; i++) {
        
        targ_oloc.addr = sn->entry[i].header;

        
        if (NULL == (targ_oh = H5O_protect(&targ_oloc, H5AC__READ_ONLY_FLAG, false)))
            HGOTO_ERROR(H5E_SYM, H5E_CANTPROTECT, H5_ITER_ERROR, "unable to protect target object header");

        
        if ((stab_exists = H5O_msg_exists_oh(targ_oh, H5O_STAB_ID)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, H5_ITER_ERROR, "unable to check for STAB message");

        if (stab_exists) {
            
            if (NULL == H5O_msg_read_oh(f, targ_oh, H5O_STAB_ID, &stab))
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, H5_ITER_ERROR, "unable to read STAB message");

            
            if (sn->entry[i].type != H5G_CACHED_STAB)
                HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, H5_ITER_ERROR, "STAB message is not cached in group node");

            if ((sn->entry[i].cache.stab.btree_addr != stab.btree_addr) ||
                (sn->entry[i].cache.stab.heap_addr != stab.heap_addr))
                HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, H5_ITER_ERROR,
                            "cached symbol table information is incorrect");
        }
        else if (sn->entry[i].type == H5G_CACHED_STAB)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, H5_ITER_ERROR, "nonexistent STAB message is cached");

        
        if (H5O_unprotect(&targ_oloc, targ_oh, H5AC__NO_FLAGS_SET) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTUNPROTECT, H5_ITER_ERROR, "unable to release object header");
        targ_oh = NULL;
    } 

done:
    if (sn && H5AC_unprotect(f, H5AC_SNODE, addr, sn, H5AC__NO_FLAGS_SET) < 0)
        HDONE_ERROR(H5E_SYM, H5E_PROTECT, H5_ITER_ERROR, "unable to release object header");

    if (targ_oh) {
        assert(ret_value == H5_ITER_ERROR);
        if (H5O_unprotect(&targ_oloc, targ_oh, H5AC__NO_FLAGS_SET) < 0)
            HDONE_ERROR(H5E_SYM, H5E_CANTUNPROTECT, H5_ITER_ERROR, "unable to release object header");
    } 

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__verify_cached_stabs_test(hid_t gid)
{
    H5G_t          *grp = NULL; 
    htri_t          stab_exists;
    H5O_stab_t      stab;                        
    H5G_bt_common_t udata     = {NULL, NULL, 0}; 
    haddr_t         prev_tag  = HADDR_UNDEF;     
    herr_t          ret_value = SUCCEED;         

    FUNC_ENTER_PACKAGE

    
    assert(gid >= 0);

    
    if (NULL == (grp = (H5G_t *)H5VL_object_verify(gid, H5I_GROUP)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a group");

    
    H5AC_tag(grp->oloc.addr, &prev_tag);

    
    
    if ((stab_exists = H5O_msg_exists(&(grp->oloc), H5O_STAB_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to read object header");

    
    if (!stab_exists)
        HGOTO_DONE(SUCCEED);

    
    if (NULL == H5O_msg_read(&(grp->oloc), H5O_STAB_ID, &stab))
        HGOTO_ERROR(H5E_SYM, H5E_BADMESG, FAIL, "can't get symbol table info");

    
    if ((ret_value = H5B_iterate(grp->oloc.file, H5B_SNODE, stab.btree_addr, H5G__verify_cached_stabs_test_cb,
                                 &udata)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTNEXT, FAIL, "iteration operator failed");

    
    H5AC_tag(prev_tag, NULL);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
