/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5HFmodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5HFpkg.h"     
#include "H5MMprivate.h" 
#include "H5VMprivate.h" 

herr_t
H5HF__dtable_init(H5HF_dtable_t *dtable)
{
    hsize_t tmp_block_size;      
    hsize_t acc_block_off;       
    size_t  u;                   
    herr_t  ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(dtable);

    
    dtable->start_bits           = H5VM_log2_of2((uint32_t)dtable->cparam.start_block_size);
    dtable->first_row_bits       = dtable->start_bits + H5VM_log2_of2(dtable->cparam.width);
    dtable->max_root_rows        = (dtable->cparam.max_index - dtable->first_row_bits) + 1;
    dtable->max_direct_bits      = H5VM_log2_of2((uint32_t)dtable->cparam.max_direct_size);
    dtable->max_direct_rows      = (dtable->max_direct_bits - dtable->start_bits) + 2;
    dtable->num_id_first_row     = dtable->cparam.start_block_size * dtable->cparam.width;
    dtable->max_dir_blk_off_size = H5HF_SIZEOF_OFFSET_LEN(dtable->cparam.max_direct_size);

    
    if (NULL == (dtable->row_block_size = (hsize_t *)H5MM_malloc(dtable->max_root_rows * sizeof(hsize_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "can't create doubling table block size table");
    if (NULL == (dtable->row_block_off = (hsize_t *)H5MM_malloc(dtable->max_root_rows * sizeof(hsize_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL, "can't create doubling table block offset table");
    if (NULL ==
        (dtable->row_tot_dblock_free = (hsize_t *)H5MM_malloc(dtable->max_root_rows * sizeof(hsize_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                    "can't create doubling table total direct block free space table");
    if (NULL == (dtable->row_max_dblock_free = (size_t *)H5MM_malloc(dtable->max_root_rows * sizeof(size_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, FAIL,
                    "can't create doubling table max. direct block free space table");
    tmp_block_size            = dtable->cparam.start_block_size;
    acc_block_off             = dtable->cparam.start_block_size * dtable->cparam.width;
    dtable->row_block_size[0] = dtable->cparam.start_block_size;
    dtable->row_block_off[0]  = 0;
    for (u = 1; u < dtable->max_root_rows; u++) {
        dtable->row_block_size[u] = tmp_block_size;
        dtable->row_block_off[u]  = acc_block_off;
        tmp_block_size *= 2;
        acc_block_off *= 2;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5HF__dtable_lookup(const H5HF_dtable_t *dtable, hsize_t off, unsigned *row, unsigned *col)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dtable);
    assert(row);
    assert(col);

    
    if (off < dtable->num_id_first_row) {
        *row = 0;
        H5_CHECKED_ASSIGN(*col, unsigned, (off / dtable->cparam.start_block_size), hsize_t);
    } 
    else {
        unsigned high_bit = H5VM_log2_gen(off);       
        hsize_t  off_mask = ((hsize_t)1) << high_bit; 

        *row = (high_bit - dtable->first_row_bits) + 1;
        H5_CHECKED_ASSIGN(*col, unsigned, ((off - off_mask) / dtable->row_block_size[*row]), hsize_t);
    } 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5HF__dtable_dest(H5HF_dtable_t *dtable)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dtable);

    
    H5MM_xfree(dtable->row_block_size);

    
    H5MM_xfree(dtable->row_block_off);

    
    H5MM_xfree(dtable->row_tot_dblock_free);

    
    H5MM_xfree(dtable->row_max_dblock_free);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

unsigned
H5HF__dtable_size_to_row(const H5HF_dtable_t *dtable, size_t block_size)
{
    unsigned row = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dtable);

    if (block_size == dtable->cparam.start_block_size)
        row = 0;
    else
        row =
            (H5VM_log2_of2((uint32_t)block_size) - H5VM_log2_of2((uint32_t)dtable->cparam.start_block_size)) +
            1;

    FUNC_LEAVE_NOAPI(row)
} 

unsigned
H5HF__dtable_size_to_rows(const H5HF_dtable_t *dtable, hsize_t size)
{
    unsigned rows = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dtable);

    rows = (H5VM_log2_gen(size) - dtable->first_row_bits) + 1;

    FUNC_LEAVE_NOAPI(rows)
} 

hsize_t
H5HF__dtable_span_size(const H5HF_dtable_t *dtable, unsigned start_row, unsigned start_col,
                       unsigned num_entries)
{
    unsigned start_entry;       
    unsigned end_row;           
    unsigned end_col;           
    unsigned end_entry;         
    hsize_t  acc_span_size = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(dtable);
    assert(num_entries > 0);

    
    start_entry = (start_row * dtable->cparam.width) + start_col;

    
    end_entry = (start_entry + num_entries) - 1;
    end_row   = end_entry / dtable->cparam.width;
    end_col   = end_entry % dtable->cparam.width;

    
    acc_span_size = 0;

    

    
    if (start_row != end_row) {
        
        if (start_col > 0) {
            acc_span_size = dtable->row_block_size[start_row] * (dtable->cparam.width - start_col);
            start_row++;
        } 

        
        while (start_row < end_row) {
            acc_span_size += dtable->row_block_size[start_row] * dtable->cparam.width;
            start_row++;
        } 

        
        acc_span_size += dtable->row_block_size[start_row] * (end_col + 1);
    } 
    else {
        
        acc_span_size = dtable->row_block_size[start_row] * ((end_col - start_col) + 1);
    } 

    FUNC_LEAVE_NOAPI(acc_span_size)
} 
