/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5T_FRIEND 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Gprivate.h"  
#include "H5Iprivate.h"  
#include "H5Oprivate.h"  
#include "H5Tpkg.h"      
#include "H5VLprivate.h" 

#include "H5VLnative_private.h" 

void *
H5VL__native_datatype_commit(void *obj, const H5VL_loc_params_t *loc_params, const char *name, hid_t type_id,
                             hid_t lcpl_id, hid_t tcpl_id, hid_t H5_ATTR_UNUSED tapl_id,
                             hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
    H5G_loc_t loc;              
    H5T_t    *dt;               
    H5T_t    *type      = NULL; 
    void     *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    if (H5G_loc_real(obj, loc_params->obj_type, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file or file object");

    if (NULL == (dt = (H5T_t *)H5I_object_verify(type_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a datatype");

    
    if (H5T_STATE_NAMED == dt->shared->state || H5T_STATE_OPEN == dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "datatype is already committed");
    if (H5T_STATE_IMMUTABLE == dt->shared->state)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, NULL, "datatype is immutable");

    
    if (H5T_is_sensible(dt) <= 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "datatype is not sensible");

    
    if (NULL == (type = H5T_copy(dt, H5T_COPY_TRANSIENT)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to copy");

    
    if (NULL != name) {
        
        if (H5T__commit_named(&loc, name, type, lcpl_id, tcpl_id) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to commit datatype");
    } 
    else {
        
        if (H5T__commit_anon(loc.oloc->file, type, tcpl_id) < 0)
            HGOTO_ERROR(H5E_DATATYPE, H5E_CANTINIT, NULL, "unable to commit datatype");
    } 

    ret_value = (void *)type;

done:
    if (NULL == ret_value && type)
        H5T_close(type);

    FUNC_LEAVE_NOAPI(ret_value)
} 

void *
H5VL__native_datatype_open(void *obj, const H5VL_loc_params_t *loc_params, const char *name,
                           hid_t H5_ATTR_UNUSED tapl_id, hid_t H5_ATTR_UNUSED dxpl_id,
                           void H5_ATTR_UNUSED **req)
{
    H5T_t    *type = NULL; 
    H5G_loc_t loc;         
    void     *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    if (H5G_loc_real(obj, loc_params->obj_type, &loc) < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, NULL, "not a file or file object");

    
    if (NULL == (type = H5T__open_name(&loc, name)))
        HGOTO_ERROR(H5E_DATATYPE, H5E_CANTOPENOBJ, NULL, "unable to open named datatype");

    type->vol_obj = NULL;

    ret_value = (void *)type;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_datatype_get(void *obj, H5VL_datatype_get_args_t *args, hid_t H5_ATTR_UNUSED dxpl_id,
                          void H5_ATTR_UNUSED **req)
{
    H5T_t *dt        = (H5T_t *)obj;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    switch (args->op_type) {
        
        case H5VL_DATATYPE_GET_BINARY_SIZE: {
            if (H5T_encode(dt, NULL, args->args.get_binary_size.size) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "can't determine serialized length of datatype");

            break;
        }

        
        case H5VL_DATATYPE_GET_BINARY: {
            if (H5T_encode(dt, args->args.get_binary.buf, &args->args.get_binary.buf_size) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTSERIALIZE, FAIL, "can't serialize datatype");

            break;
        }

        
        case H5VL_DATATYPE_GET_TCPL: {
            if (H5I_INVALID_HID == (args->args.get_tcpl.tcpl_id = H5T__get_create_plist(dt)))
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTGET, FAIL, "can't get object creation info");

            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't get this type of information from datatype");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_datatype_specific(void *obj, H5VL_datatype_specific_args_t *args, hid_t H5_ATTR_UNUSED dxpl_id,
                               void H5_ATTR_UNUSED **req)
{
    H5T_t *dt        = (H5T_t *)obj;
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    switch (args->op_type) {
        
        case H5VL_DATATYPE_FLUSH: {
            
            if (H5F_HAS_FEATURE(dt->oloc.file, H5FD_FEAT_HAS_MPI))
                HGOTO_ERROR(H5E_DATASET, H5E_UNSUPPORTED, FAIL, "H5Oflush isn't supported for parallel");

            if (H5O_flush_common(&dt->oloc, args->args.flush.type_id) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTFLUSH, FAIL, "unable to flush datatype");

            break;
        }

        
        case H5VL_DATATYPE_REFRESH: {
            if ((H5O_refresh_metadata(&dt->oloc, args->args.refresh.type_id)) < 0)
                HGOTO_ERROR(H5E_DATATYPE, H5E_CANTLOAD, FAIL, "unable to refresh datatype");

            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "invalid specific operation");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_datatype_close(void *dt, hid_t H5_ATTR_UNUSED dxpl_id, void H5_ATTR_UNUSED **req)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    if (H5T_close((H5T_t *)dt) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTDEC, FAIL, "can't close datatype");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
