/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Gmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Fprivate.h" 
#include "H5Gpkg.h"     
#include "H5Iprivate.h" 
#include "H5Lprivate.h" 
#include "H5Pprivate.h" 

typedef struct {
    H5F_t       *f;       
    haddr_t      oh_addr; 
    H5O_linfo_t *linfo;   
} H5G_obj_oh_it_ud1_t;

typedef struct {
    H5O_link_t *lnk_table;   
    size_t      nlinks;      
    size_t      alloc_links; 
} H5G_obj_lnk_it_ud1_t;

typedef struct {
    const H5O_loc_t *grp_oloc; 
} H5G_obj_stab_it_ud1_t;

static herr_t H5G__obj_compact_to_dense_cb(const void *_mesg, unsigned idx, void *_udata);
static herr_t H5G__obj_remove_update_linfo(const H5O_loc_t *oloc, H5O_linfo_t *linfo);

herr_t
H5G__obj_create(H5F_t *f, H5G_obj_create_t *gcrt_info, H5O_loc_t *oloc )
{
    H5P_genplist_t *gc_plist;            
    H5O_ginfo_t     ginfo;               
    H5O_linfo_t     linfo;               
    H5O_pline_t     pline;               
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(oloc);

    
    if (NULL == (gc_plist = (H5P_genplist_t *)H5I_object(gcrt_info->gcpl_id)))
        HGOTO_ERROR(H5E_SYM, H5E_BADTYPE, FAIL, "not a property list");

    
    if (H5P_get(gc_plist, H5G_CRT_GROUP_INFO_NAME, &ginfo) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get group info");

    
    if (H5P_get(gc_plist, H5G_CRT_LINK_INFO_NAME, &linfo) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get group info");

    
    if (H5P_peek(gc_plist, H5O_CRT_PIPELINE_NAME, &pline) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't get group info");

    
    if (H5G__obj_create_real(f, &ginfo, &linfo, &pline, gcrt_info, oloc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCREATE, FAIL, "unable to create group");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__obj_create_real(H5F_t *f, const H5O_ginfo_t *ginfo, const H5O_linfo_t *linfo, const H5O_pline_t *pline,
                     H5G_obj_create_t *gcrt_info, H5O_loc_t *oloc )
{
    size_t hdr_size;                       
    bool   use_at_least_v18;               
    hid_t  gcpl_id   = gcrt_info->gcpl_id; 
    herr_t ret_value = SUCCEED;            

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ginfo);
    assert(linfo);
    assert(pline);
    assert(oloc);

    
    if (0 == (H5F_INTENT(f) & H5F_ACC_RDWR))
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "no write intent on file");

    
    
    if ((H5F_LOW_BOUND(f) >= H5F_LIBVER_V18) || linfo->track_corder || (pline && pline->nused))
        use_at_least_v18 = true;
    else
        use_at_least_v18 = false;

    
    if (linfo->index_corder && !linfo->track_corder)
        HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "must track creation order to create index for it");

    
    if (use_at_least_v18) {
        H5O_link_t lnk;              
        char       null_char = '\0'; 
        size_t     ginfo_size;       
        size_t     linfo_size;       
        size_t     pline_size = 0;   
        size_t     link_size;        

        
        linfo_size = H5O_msg_size_f(f, gcpl_id, H5O_LINFO_ID, linfo, (size_t)0);
        assert(linfo_size);

        ginfo_size = H5O_msg_size_f(f, gcpl_id, H5O_GINFO_ID, ginfo, (size_t)0);
        assert(ginfo_size);

        if (pline && pline->nused) {
            pline_size = H5O_msg_size_f(f, gcpl_id, H5O_PLINE_ID, pline, (size_t)0);
            assert(pline_size);
        } 

        lnk.type         = H5L_TYPE_HARD;
        lnk.corder       = 0;
        lnk.corder_valid = linfo->track_corder;
        lnk.cset         = H5T_CSET_ASCII;
        lnk.name         = &null_char;
        link_size        = H5O_msg_size_f(f, gcpl_id, H5O_LINK_ID, &lnk, (size_t)ginfo->est_name_len);
        assert(link_size);

        

        
        hdr_size = linfo_size + ginfo_size + pline_size;

        
        bool compact = ginfo->est_num_entries <= ginfo->max_compact;
        if (compact) {

            size_t size_of_links = ginfo->est_num_entries * link_size;

            if (size_of_links < H5O_MESG_MAX_SIZE)
                hdr_size += size_of_links;
        }
    } 
    else
        hdr_size = (size_t)(4 + 2 * H5F_SIZEOF_ADDR(f));

    
    if (H5O_create(f, hdr_size, (size_t)1, gcpl_id, oloc ) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create header");

    
    if (use_at_least_v18) {
        H5_WARN_CAST_AWAY_CONST_OFF
        
        if (H5O_msg_create(oloc, H5O_LINFO_ID, 0, H5O_UPDATE_TIME, (void *)linfo) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create message");

        
        if (H5O_msg_create(oloc, H5O_GINFO_ID, H5O_MSG_FLAG_CONSTANT, 0, (void *)ginfo) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create message");

        
        if (pline && pline->nused)
            if (H5O_msg_create(oloc, H5O_PLINE_ID, H5O_MSG_FLAG_CONSTANT, 0, (void *)pline) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create message");
        H5_WARN_CAST_AWAY_CONST_ON
    } 
    else {
        H5O_stab_t stab; 

        
        if (H5G__stab_create(oloc, ginfo, &stab) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create symbol table");

        
        gcrt_info->cache_type            = H5G_CACHED_STAB;
        gcrt_info->cache.stab.btree_addr = stab.btree_addr;
        gcrt_info->cache.stab.heap_addr  = stab.heap_addr;
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

htri_t
H5G__obj_get_linfo(const H5O_loc_t *grp_oloc, H5O_linfo_t *linfo)
{
    H5B2_t *bt2_name  = NULL; 
    htri_t  ret_value = FAIL; 

    FUNC_ENTER_PACKAGE_TAG(grp_oloc->addr)

    
    assert(grp_oloc);
    assert(linfo);

    
    if ((ret_value = H5O_msg_exists(grp_oloc, H5O_LINFO_ID)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to read object header");
    if (ret_value) {
        
        if (NULL == H5O_msg_read(grp_oloc, H5O_LINFO_ID, linfo))
            HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "link info message not present");

        
        if (linfo->nlinks == HSIZET_MAX) {
            
            if (H5_addr_defined(linfo->fheap_addr)) {
                
                if (NULL == (bt2_name = H5B2_open(grp_oloc->file, linfo->name_bt2_addr, NULL)))
                    HGOTO_ERROR(H5E_SYM, H5E_CANTOPENOBJ, FAIL, "unable to open v2 B-tree for name index");

                
                
                if (H5B2_get_nrec(bt2_name, &linfo->nlinks) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve # of records in index");
            } 
            else {
                
                if (H5O_get_nlinks(grp_oloc, &linfo->nlinks) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't retrieve # of links for object");
            } 
        }     
    }         

done:
    
    if (bt2_name && H5B2_close(bt2_name) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CLOSEERROR, FAIL, "can't close v2 B-tree for name index");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static herr_t
H5G__obj_compact_to_dense_cb(const void *_mesg, unsigned H5_ATTR_UNUSED idx, void *_udata)
{
    const H5O_link_t    *lnk       = (const H5O_link_t *)_mesg;     
    H5G_obj_oh_it_ud1_t *udata     = (H5G_obj_oh_it_ud1_t *)_udata; 
    herr_t               ret_value = H5_ITER_CONT;                  

    FUNC_ENTER_PACKAGE

    
    assert(lnk);
    assert(udata);

    
    if (H5G__dense_insert(udata->f, udata->linfo, lnk) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert link into dense storage");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5G__obj_stab_to_new_cb(const H5O_link_t *lnk, void *_udata)
{
    H5G_obj_stab_it_ud1_t *udata     = (H5G_obj_stab_it_ud1_t *)_udata; 
    herr_t                 ret_value = H5_ITER_CONT;                    

    FUNC_ENTER_PACKAGE

    
    assert(lnk);
    assert(udata);

    
    H5_WARN_CAST_AWAY_CONST_OFF
    if (H5G_obj_insert(udata->grp_oloc, (H5O_link_t *)lnk, false, H5O_TYPE_UNKNOWN, NULL) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, H5_ITER_ERROR, "can't insert link into group");
    H5_WARN_CAST_AWAY_CONST_ON

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_obj_insert(const H5O_loc_t *grp_oloc, H5O_link_t *obj_lnk, bool adj_link, H5O_type_t obj_type,
               const void *crt_info)
{
    H5O_pline_t  tmp_pline;             
    H5O_pline_t *pline = NULL;          
    H5O_linfo_t  linfo;                 
    htri_t       linfo_exists;          
    bool         use_old_format;        
    bool         use_new_dense = false; 
    herr_t       ret_value     = SUCCEED; 

    FUNC_ENTER_NOAPI_TAG(grp_oloc->addr, FAIL)

    
    assert(grp_oloc && grp_oloc->file);
    assert(obj_lnk);

    
    
    if ((linfo_exists = H5G__obj_get_linfo(grp_oloc, &linfo)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check for link info message");
    if (linfo_exists) {
        H5O_ginfo_t ginfo;         
        size_t      link_msg_size; 

        
        use_old_format = false;

        
        if (linfo.track_corder) {
            
            obj_lnk->corder       = linfo.max_corder;
            obj_lnk->corder_valid = true;

            
            linfo.max_corder++;
        } 

        
        if ((link_msg_size = H5O_msg_raw_size(grp_oloc->file, H5O_LINK_ID, false, obj_lnk)) == 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTGETSIZE, FAIL, "can't get link size");

        
        if (NULL == H5O_msg_read(grp_oloc, H5O_GINFO_ID, &ginfo))
            HGOTO_ERROR(H5E_SYM, H5E_BADMESG, FAIL, "can't get group info");

        
        
        if (H5_addr_defined(linfo.fheap_addr))
            use_new_dense = true;
        else if (linfo.nlinks < ginfo.max_compact && link_msg_size < H5O_MESG_MAX_SIZE)
            use_new_dense = false;
        else {
            htri_t              pline_exists; 
            H5G_obj_oh_it_ud1_t udata;        
            H5O_mesg_operator_t op;           

            
            if ((pline_exists = H5O_msg_exists(grp_oloc, H5O_PLINE_ID)) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "unable to read object header");
            if (pline_exists) {
                if (NULL == H5O_msg_read(grp_oloc, H5O_PLINE_ID, &tmp_pline))
                    HGOTO_ERROR(H5E_SYM, H5E_BADMESG, FAIL, "can't get link pipeline");
                pline = &tmp_pline;
            } 

            
            if (H5G__dense_create(grp_oloc->file, &linfo, pline) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "unable to create 'dense' form of new format group");

            
            udata.f       = grp_oloc->file;
            udata.oh_addr = grp_oloc->addr;
            udata.linfo   = &linfo;

            
            op.op_type  = H5O_MESG_OP_APP;
            op.u.app_op = H5G__obj_compact_to_dense_cb;
            if (H5O_msg_iterate(grp_oloc, H5O_LINK_ID, &op, &udata) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "error iterating over links");

            
            if (H5O_msg_remove(grp_oloc, H5O_LINK_ID, H5O_ALL, false) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete link messages");

            use_new_dense = true;
        } 
    }     
    else {
        
        if (obj_lnk->cset != H5T_CSET_ASCII || obj_lnk->type > H5L_TYPE_BUILTIN_MAX) {
            H5O_linfo_t           new_linfo = H5G_CRT_LINK_INFO_DEF;  
            H5O_ginfo_t           new_ginfo = H5G_CRT_GROUP_INFO_DEF; 
            H5G_obj_stab_it_ud1_t udata;                              

            

            
            if (H5O_msg_create(grp_oloc, H5O_LINFO_ID, 0, 0, &new_linfo) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create message");

            
            if (H5O_msg_create(grp_oloc, H5O_GINFO_ID, H5O_MSG_FLAG_CONSTANT, H5O_UPDATE_TIME, &new_ginfo) <
                0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create message");

            
            udata.grp_oloc = grp_oloc;

            
            if (H5G__stab_iterate(grp_oloc, H5_ITER_NATIVE, (hsize_t)0, NULL, H5G__obj_stab_to_new_cb,
                                  &udata) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTNEXT, FAIL, "error iterating over old format links");

            
            if (H5O_msg_remove(grp_oloc, H5O_STAB_ID, 0, false) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete old format link storage");

            
            if (H5G_obj_insert(grp_oloc, obj_lnk, adj_link, obj_type, crt_info) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert link into group");

            
            HGOTO_DONE(SUCCEED);
        } 
        else
            use_old_format = true;
    } 

    
    if (use_old_format) {
        
        if (H5G__stab_insert(grp_oloc, obj_lnk, obj_type, crt_info) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert entry into symbol table");
    } 
    else {
        if (use_new_dense) {
            
            if (H5G__dense_insert(grp_oloc->file, &linfo, obj_lnk) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert link into dense storage");
        } 
        else {
            
            if (H5G__compact_insert(grp_oloc, obj_lnk) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTINSERT, FAIL, "unable to insert link as link message");
        } 
    }     

    
    if (!use_old_format) {
        linfo.nlinks++;
        if (H5O_msg_write(grp_oloc, H5O_LINFO_ID, 0, H5O_UPDATE_TIME, &linfo) < 0)
            HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINIT, FAIL, "can't update link info message");
    } 

    
    if (adj_link && obj_lnk->type == H5L_TYPE_HARD) {
        H5O_loc_t obj_oloc; 
        H5O_loc_reset(&obj_oloc);

        
        obj_oloc.file = grp_oloc->file;
        obj_oloc.addr = obj_lnk->u.hard.addr;

        
        if (H5O_link(&obj_oloc, 1) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_LINKCOUNT, FAIL, "unable to increment hard link count");
    } 

done:
    
    if (pline && H5O_msg_reset(H5O_PLINE_ID, pline) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "can't release pipeline");

    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5G__obj_iterate(const H5O_loc_t *grp_oloc, H5_index_t idx_type, H5_iter_order_t order, hsize_t skip,
                 hsize_t *last_lnk, H5G_lib_iterate_t op, void *op_data)
{
    H5O_linfo_t linfo;            
    htri_t      linfo_exists;     
    herr_t      ret_value = FAIL; 

    FUNC_ENTER_PACKAGE_TAG(grp_oloc->addr)

    
    assert(grp_oloc);
    assert(op);

    
    if ((linfo_exists = H5G__obj_get_linfo(grp_oloc, &linfo)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check for link info message");
    if (linfo_exists) {
        
        if (skip > 0 && (size_t)skip >= linfo.nlinks)
            HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "index out of bound");

        
        if (idx_type == H5_INDEX_CRT_ORDER) {
            
            if (!linfo.track_corder)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "creation order not tracked for links in group");
        } 

        if (H5_addr_defined(linfo.fheap_addr)) {
            
            if ((ret_value = H5G__dense_iterate(grp_oloc->file, &linfo, idx_type, order, skip, last_lnk, op,
                                                op_data)) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "can't iterate over dense links");
        } 
        else {
            
            if ((ret_value = H5G__compact_iterate(grp_oloc, &linfo, idx_type, order, skip, last_lnk, op,
                                                  op_data)) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "can't iterate over compact links");
        } 
    }     
    else {
        
        if (idx_type != H5_INDEX_NAME)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "no creation order index to query");

        
        if ((ret_value = H5G__stab_iterate(grp_oloc, order, skip, last_lnk, op, op_data)) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_BADITER, FAIL, "can't iterate over symbol table");
    } 

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5G__obj_info(const H5O_loc_t *oloc, H5G_info_t *grp_info)
{
    H5G_t      *grp = NULL;          
    H5G_loc_t   grp_loc;             
    H5G_name_t  grp_path;            
    H5O_loc_t   grp_oloc;            
    H5O_linfo_t linfo;               
    htri_t      linfo_exists;        
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oloc);
    assert(grp_info);

    
    grp_loc.oloc = &grp_oloc;
    grp_loc.path = &grp_path;
    H5G_loc_reset(&grp_loc);

    
    if (H5O_loc_copy_deep(&grp_oloc, oloc) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTCOPY, FAIL, "can't copy object location");

    
    if (NULL == (grp = H5G_open(&grp_loc)))
        HGOTO_ERROR(H5E_FILE, H5E_MOUNT, FAIL, "mount point not found");

    
    grp_info->mounted = H5G_MOUNTED(grp);

    
    if ((linfo_exists = H5G__obj_get_linfo(oloc, &linfo)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check for link info message");
    if (linfo_exists) {
        
        grp_info->nlinks     = linfo.nlinks;
        grp_info->max_corder = linfo.max_corder;

        
        if (H5_addr_defined(linfo.fheap_addr))
            grp_info->storage_type = H5G_STORAGE_TYPE_DENSE;
        else
            grp_info->storage_type = H5G_STORAGE_TYPE_COMPACT;
    } 
    else {
        
        if (H5G__stab_count(oloc, &grp_info->nlinks) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTCOUNT, FAIL, "can't count objects");

        
        grp_info->storage_type = H5G_STORAGE_TYPE_SYMBOL_TABLE;
        grp_info->max_corder   = 0;
    } 

done:
    
    if (grp && H5G_close(grp) < 0)
        HDONE_ERROR(H5E_SYM, H5E_CANTCLOSEOBJ, FAIL, "unable to close queried group");

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_obj_get_name_by_idx(const H5O_loc_t *oloc, H5_index_t idx_type, H5_iter_order_t order, hsize_t n,
                        char *name, size_t name_size, size_t *name_len)
{
    H5O_linfo_t linfo;               
    htri_t      linfo_exists;        
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_TAG(oloc->addr, FAIL)

    
    assert(oloc && oloc->file);

    
    if ((linfo_exists = H5G__obj_get_linfo(oloc, &linfo)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check for link info message");
    if (linfo_exists) {
        
        if (idx_type == H5_INDEX_CRT_ORDER)
            
            if (!linfo.track_corder)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "creation order not tracked for links in group");

        
        if (H5_addr_defined(linfo.fheap_addr)) {
            
            if (H5G__dense_get_name_by_idx(oloc->file, &linfo, idx_type, order, n, name, name_size,
                                           name_len) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't locate name");
        } 
        else {
            
            if (H5G__compact_get_name_by_idx(oloc, &linfo, idx_type, order, n, name, name_size, name_len) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't locate name");
        } 
    }     
    else {
        
        if (idx_type != H5_INDEX_NAME)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "no creation order index to query");

        
        if (H5G__stab_get_name_by_idx(oloc, order, n, name, name_size, name_len) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't locate name");
    } 

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

static herr_t
H5G__obj_remove_update_linfo(const H5O_loc_t *oloc, H5O_linfo_t *linfo)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(oloc);
    assert(linfo);

    
    linfo->nlinks--;

    
    if (linfo->nlinks == 0)
        linfo->max_corder = 0;

    
    if (H5_addr_defined(linfo->fheap_addr)) {
        
        if (linfo->nlinks == 0) {
            
            if (H5G__dense_delete(oloc->file, linfo, false) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete dense link storage");
        } 
        
        else {
            H5O_ginfo_t ginfo; 

            
            if (NULL == H5O_msg_read(oloc, H5O_GINFO_ID, &ginfo))
                HGOTO_ERROR(H5E_SYM, H5E_BADMESG, FAIL, "can't get group info");

            
            if (linfo->nlinks < ginfo.min_dense) {
                struct H5O_t    *oh = NULL;          
                H5G_link_table_t ltable;             
                bool             can_convert = true; 
                size_t           u;                  

                
                if (H5G__dense_build_table(oloc->file, linfo, H5_INDEX_NAME, H5_ITER_NATIVE, &ltable) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTNEXT, FAIL, "error iterating over links");

                
                if (NULL == (oh = H5O_pin(oloc)))
                    HGOTO_ERROR(H5E_SYM, H5E_CANTPIN, FAIL, "unable to pin group object header");

                
                for (u = 0; u < linfo->nlinks; u++)
                    if (H5O_msg_size_oh(oloc->file, oh, H5O_LINK_ID, &(ltable.lnks[u]), (size_t)0) >=
                        H5O_MESG_MAX_SIZE) {
                        can_convert = false;
                        break;
                    } 

                
                if (can_convert) {
                    
                    for (u = 0; u < linfo->nlinks; u++)
                        if (H5O_msg_append_oh(oloc->file, oh, H5O_LINK_ID, 0, H5O_UPDATE_TIME,
                                              &(ltable.lnks[u])) < 0) {
                            
                            if (H5O_unpin(oh) < 0)
                                HDONE_ERROR(H5E_SYM, H5E_CANTUNPIN, FAIL,
                                            "unable to unpin group object header");

                            HGOTO_ERROR(H5E_SYM, H5E_CANTINIT, FAIL, "can't create message");
                        } 

                    
                    if (H5G__dense_delete(oloc->file, linfo, false) < 0)
                        HGOTO_ERROR(H5E_SYM, H5E_CANTDELETE, FAIL, "unable to delete dense link storage");
                } 

                
                if (H5O_unpin(oh) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTUNPIN, FAIL, "unable to unpin group object header");

                
                if (H5G__link_release_table(&ltable) < 0)
                    HGOTO_ERROR(H5E_SYM, H5E_CANTFREE, FAIL, "unable to release link table");
            } 
        }     
    }         

    
    if (H5O_msg_write(oloc, H5O_LINFO_ID, 0, H5O_UPDATE_TIME, linfo) < 0)
        HGOTO_ERROR(H5E_DATASPACE, H5E_CANTINIT, FAIL, "can't update link info message");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G_obj_remove(const H5O_loc_t *oloc, H5RS_str_t *grp_full_path_r, const char *name)
{
    H5O_linfo_t linfo;               
    htri_t      linfo_exists;        
    bool        use_old_format;      
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_TAG(oloc->addr, FAIL)

    
    assert(oloc);
    assert(name && *name);

    
    if ((linfo_exists = H5G__obj_get_linfo(oloc, &linfo)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check for link info message");
    if (linfo_exists) {
        
        use_old_format = false;

        
        if (H5_addr_defined(linfo.fheap_addr)) {
            
            if (H5G__dense_remove(oloc->file, &linfo, grp_full_path_r, name) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't remove object");
        } 
        else
            
            if (H5G__compact_remove(oloc, grp_full_path_r, name) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't remove object");
    } 
    else {
        
        use_old_format = true;

        
        if (H5G__stab_remove(oloc, grp_full_path_r, name) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't remove object");
    } 

    
    if (!use_old_format)
        if (H5G__obj_remove_update_linfo(oloc, &linfo) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTUPDATE, FAIL, "unable to update link info");

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5G_obj_remove_by_idx(const H5O_loc_t *grp_oloc, H5RS_str_t *grp_full_path_r, H5_index_t idx_type,
                      H5_iter_order_t order, hsize_t n)
{
    H5O_linfo_t linfo;               
    htri_t      linfo_exists;        
    bool        use_old_format;      
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(grp_oloc && grp_oloc->file);

    
    if ((linfo_exists = H5G__obj_get_linfo(grp_oloc, &linfo)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check for link info message");
    if (linfo_exists) {
        
        if (idx_type == H5_INDEX_CRT_ORDER) {
            
            if (!linfo.track_corder)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "creation order not tracked for links in group");
        } 

        
        use_old_format = false;

        
        if (H5_addr_defined(linfo.fheap_addr)) {
            
            if (H5G__dense_remove_by_idx(grp_oloc->file, &linfo, grp_full_path_r, idx_type, order, n) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't remove object");
        } 
        else {
            
            if (H5G__compact_remove_by_idx(grp_oloc, &linfo, grp_full_path_r, idx_type, order, n) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't remove object");
        } 
    }     
    else {
        
        if (idx_type != H5_INDEX_NAME)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "no creation order index to query");

        
        use_old_format = true;

        
        if (H5G__stab_remove_by_idx(grp_oloc, grp_full_path_r, order, n) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't remove object");
    } 

    
    if (!use_old_format)
        if (H5G__obj_remove_update_linfo(grp_oloc, &linfo) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_CANTUPDATE, FAIL, "unable to update link info");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5G__obj_lookup(const H5O_loc_t *grp_oloc, const char *name, bool *found, H5O_link_t *lnk)
{
    H5O_linfo_t linfo;               
    htri_t      linfo_exists;        
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE_TAG(grp_oloc->addr)

    
    assert(grp_oloc && grp_oloc->file);
    assert(name && *name);

    
    if ((linfo_exists = H5G__obj_get_linfo(grp_oloc, &linfo)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check for link info message");
    if (linfo_exists) {
        
        if (H5_addr_defined(linfo.fheap_addr)) {
            
            if (H5G__dense_lookup(grp_oloc->file, &linfo, name, found, lnk) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't locate object");
        } 
        else {
            
            if (H5G__compact_lookup(grp_oloc, name, found, lnk) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't locate object");
        } 
    }     
    else
        
        if (H5G__stab_lookup(grp_oloc, name, found, lnk) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't locate object");

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 

herr_t
H5G_obj_lookup_by_idx(const H5O_loc_t *grp_oloc, H5_index_t idx_type, H5_iter_order_t order, hsize_t n,
                      H5O_link_t *lnk)
{
    H5O_linfo_t linfo;               
    htri_t      linfo_exists;        
    herr_t      ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_TAG(grp_oloc->addr, FAIL)

    
    assert(grp_oloc && grp_oloc->file);

    
    if ((linfo_exists = H5G__obj_get_linfo(grp_oloc, &linfo)) < 0)
        HGOTO_ERROR(H5E_SYM, H5E_CANTGET, FAIL, "can't check for link info message");
    if (linfo_exists) {
        
        if (idx_type == H5_INDEX_CRT_ORDER) {
            
            if (!linfo.track_corder)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "creation order not tracked for links in group");
        } 

        
        if (H5_addr_defined(linfo.fheap_addr)) {
            
            if (H5G__dense_lookup_by_idx(grp_oloc->file, &linfo, idx_type, order, n, lnk) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't locate object");
        } 
        else {
            
            if (H5G__compact_lookup_by_idx(grp_oloc, &linfo, idx_type, order, n, lnk) < 0)
                HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't locate object");
        } 
    }     
    else {
        
        if (idx_type != H5_INDEX_NAME)
            HGOTO_ERROR(H5E_SYM, H5E_BADVALUE, FAIL, "no creation order index to query");

        
        if (H5G__stab_lookup_by_idx(grp_oloc, order, n, lnk) < 0)
            HGOTO_ERROR(H5E_SYM, H5E_NOTFOUND, FAIL, "can't locate object");
    } 

done:
    FUNC_LEAVE_NOAPI_TAG(ret_value)
} 
