% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/contour.R
\name{crit_ICU}
\alias{crit_ICU}
\title{Integrated Contour Uncertainty criterion}
\usage{
crit_ICU(x, model, thres = 0, Xref, w = NULL, preds = NULL, kxprime = NULL)
}
\arguments{
\item{x}{matrix of new designs, one point per row (size n x d)}

\item{model}{\code{homGP} or \code{hetGP} model, including inverse matrices}

\item{thres}{for contour finding}

\item{Xref}{matrix of input locations to approximate the integral by a sum}

\item{w}{optional weights vector of weights for \code{Xref} locations}

\item{preds}{optional predictions at \code{Xref} to avoid recomputing if already done}

\item{kxprime}{optional covariance matrix between \code{model$X0} and \code{Xref} to avoid its recomputation}
}
\description{
Computes ICU infill criterion
}
\examples{
## Infill criterion example
set.seed(42)
branin <- function(x){
  m <- 54.8104; s <- 51.9496
  if(is.null(dim(x))) x <- matrix(x, nrow = 1)
  xx <- 15 * x[,1] - 5; y <- 15 * x[,2]
  f <- (y - 5.1 * xx^2/(4 * pi^2) + 5 * xx/pi - 6)^2 + 10 * (1 - 1/(8 * pi)) * cos(xx) + 10
  f <- (f - m)/s
  return(f)
}

ftest <- function(x, sd = 0.1){
  if(is.null(dim(x))) x <- matrix(x, nrow = 1)
  return(apply(x, 1, branin) + rnorm(nrow(x), sd = sd))
}

ngrid <- 51; xgrid <- seq(0, 1, length.out = ngrid)
Xgrid <- as.matrix(expand.grid(xgrid, xgrid))
Zgrid <- ftest(Xgrid)

n <- 20
N <- 500
X <- Xgrid[sample(1:nrow(Xgrid), n),]
X <- X[sample(1:n, N, replace = TRUE),]
Z <- ftest(X)
model <- mleHetGP(X, Z, lower = rep(0.001,2), upper = rep(1,2))

# Precalculations for speedup
preds <- predict(model, x = Xgrid)
kxprime <- cov_gen(X1 = model$X0, X2 = Xgrid, theta = model$theta, type = model$covtype)
 
critgrid <- apply(Xgrid, 1, crit_ICU, model = model, Xref = Xgrid,
                  preds = preds, kxprime = kxprime)

filled.contour(matrix(critgrid, ngrid), color.palette = terrain.colors, main = "ICU criterion")

}
\references{
Lyu, X., Binois, M. & Ludkovski, M. (2021). Evaluating Gaussian Process Metamodels and Sequential Designs for Noisy Level Set Estimation. Statistics and Computing, 31(4), 43. arXiv:1807.06712.
}
