#include <RcppArmadillo.h>
#include "ieTest.h"
using namespace Rcpp;

// Function that accepts cumulative probabilities for the three terms that comprise
// the product of the indirect effect - the effect of the independent variable
// on the first mediator, the effect of the first mediator (and the interaction
// with the independent variable) on the second mediator, and the effect of the 
// second mediator (and the interaction with the independent variable) on the response. 

//' Is the mediation effect significant?
//'
//' This function takes 3 U values for the asq-test for 2 ordered
//' mediators, as well as an alpha level. It returns whether the
//' test would reject at the given alpha level. Additionally,
//' the cutoff, either specified by number of squares or the 
//' percentage towards the center of the transformation region, 
//' can be specified.
//' This function is primarily called by the asq_ord function
//' to determine if the test is significant at one of a chosen
//' set of alpha values. 
//'
//' @param alpha Significance level for the test.
//' @param u1,u2,u3 The U values to be used in the test
//' @param numSquares The number of squares to be used in the asq-test. Always superceded by upLim
//' @param upLim The allowed extension, between 0 and 1, of the squares towards the center of the region
//' @returns A boolean variable indicating if the indirect effect null hypothesis is rejected.
//' @export
//' @examples
//' asq_ord_bool(.05, .02, .015, .995, upLim = .75)
// [[Rcpp::export]]
bool asq_ord_bool(double alpha, double u1, double u2, double u3, 
                  long int numSquares = 4, double upLim = 0.5){
  // If upper limit is too large (greater than 100%), lower to max.
  if(upLim > 1.0){
    Rcout << "Upper Limit too large. Running with Upper Limit set at 1\n";
    upLim = 1.0;
  }
  // If upLim specified, set numSquares off of that;
  if(upLim > 0){
    numSquares = floor(upLim / alpha);
  }
  u1 = std::min(u1, 1 - u1);
  u2 = std::min(u2, 1 - u2);
  u3 = std::min(u3, 1 - u3);
  
  long double u1Square = floor(u1 / (alpha/2.0));
  long double u2Square = floor(u2 / (alpha/2.0));
  long double u3Square = floor(u3 / (alpha/2.0));
  return((u1Square == u2Square) & (u1Square == u3Square) & (u1Square < numSquares));
}
