% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/intcalibrate.R
\name{intcalibrate}
\alias{intcalibrate}
\title{Integer Calibration Function}
\usage{
intcalibrate(
  weights,
  formula,
  targets,
  objective = c("L1", "aL1", "rL1", "LB1", "rB1", "L2", "aL2", "rL2", "LB2", "rB2", "LC",
    "aLC", "rLC", "SE", "aSE", "rSE"),
  tgtBnds = NULL,
  lower = -Inf,
  upper = Inf,
  scale = NULL,
  sparse = FALSE,
  penalty = c("null", "l0norm", "lasso", "ridge", "raking", "minentropy", "quadrat",
    "quadmod", "hellinger", "mcp", "scad", "relasso", "modrelasso", "rehuber",
    "modrehuber"),
  tuning = 0,
  data
)
}
\arguments{
\item{weights}{A numerical vector of real or integer weights to be calibrated. If real values are provided, they will be rounded before applying the calibration algorithm}

\item{formula}{A formula to express a linear system for hitting the \code{targets}}

\item{targets}{A numerical vector of point-targets to hit}

\item{objective}{A character specifying the objective function used for calibration. By default \code{"L1"}. See details for more information}

\item{tgtBnds}{A two-column matrix containing the bounds for the point-targets}

\item{lower}{A numerical vector or value defining the lower bounds of the weights}

\item{upper}{A numerical vector or value defining the upper bounds of the weights}

\item{scale}{A numerical vector of positive values}

\item{sparse}{A logical value denoting if the linear system is sparse or not. By default it is \code{FALSE}}

\item{penalty}{A character specifying the penalty function. By default \code{"null"} (for backward compatibility). See details for more information}

\item{tuning}{A positive value denoting the tuning parameter to control the intensity of the penalty function}

\item{data}{A \code{data.frame} or \code{matrix} object containing the data to be used for calibration}
}
\value{
A numerical vector of calibrated integer weights.
}
\description{
This function performs an integer programming algorithm developed for calibrating integer weights, 
in order to reduce a specific objective function
}
\details{
The integer programming algorithm for calibration can be performed by considering one of the following objective functions:
\describe{
  \item{\code{"L1"}}{for the summation of absolute errors}
  \item{\code{"aL1"}}{for the asymmetric summation of absolute errors}
  \item{\code{"rL1"}}{for the summation of absolute relative errors}
  \item{\code{"LB1"}}{for the summation of absolute errors if outside the boundaries}
  \item{\code{"rB1"}}{for the summation of absolute relative errors if outside the boundaries}
  \item{\code{"L2"}}{for the summation of square errors}
  \item{\code{"aL2"}}{for the asymmetric summation of square errors}
  \item{\code{"rL2"}}{for the summation of square relative errors}
  \item{\code{"LB2"}}{for the summation of square errors if outside the boundaries}
  \item{\code{"rB2"}}{for the summation of square relative errors if outside the boundaries}
  \item{\code{"LC"}}{for the summation of the logcosh errors}
  \item{\code{"aLC"}}{for the asymmetric summation of the logcosh errors}
  \item{\code{"rLC"}}{for the summation of the logcosh relative errors}
  \item{\code{"SE"}}{for the summation of the exponential absolute errors}
  \item{\code{"aSE"}}{for the asymmetric summation of the exponential absolute errors}
  \item{\code{"rSE"}}{for the summation of the exponential absolute relative errors}
}

The calibrated weights can also be restricted further using one of the following penalty functions:
\describe{
  \item{\code{"null"}}{does not penalize, and it is used for backward compatibility}
  \item{\code{"l0norm"}}{counts the number of non-zero adjustments}
  \item{\code{"lasso"}}{sums the absolute values of the adjustments}
  \item{\code{"ridge"}}{sums the adjustments squared}
  \item{\code{"raking"}}{uses raking ratios}
  \item{\code{"minentropy"}}{uses the minimum entropy}
  \item{\code{"quadrat"}}{uses a nomalized euclidean distance}
  \item{\code{"quadmod"}}{uses a modified normalization in the euclidean distance}
  \item{\code{"hellinger"}}{uses the Hellinger's distance}
  \item{\code{"mcp"}}{uses a variation of the minimax concave penalty}
  \item{\code{"scad"}}{uses a variation of the smoothly clipped absolute deviations}
  \item{\code{"relasso"}}{sums the absolute value of the relative adjustments}
  \item{\code{"modrelasso"}}{sums the absolute value of the modified relative adjustments}
  \item{\code{"rehuber"}}{uses the Huber loss on the relative adjustments}
  \item{\code{"modrehuber"}}{uses the Huber loss on the modified relative adjustmnets}
}
In particular, the adjustments are considered from the initial rounded weights rather than the input vector of weights with real numbers.

A two-column matrix must be provided to \code{tgtBnds} when \code{objective = "aL1"}, \code{objective = "LB1"}, \code{objective = "rB1"},
\code{objective = "aL2"}, \code{objective = "LB2"}, \code{objective = "rB2"}, \code{objective = "aLC"}, and \code{objective = "aSE"}..

The argument \code{scale} must be specified with a vector of positive real numbers when \code{objective = "rL1"},
\code{objective = "rL2"}, \code{objective = "rLC"}, or \code{objective = "rSE"}.
}
\examples{
library(inca)
set.seed(0)
w <- rpois(10, 4)
data <- matrix(rbinom(1000, 1, .3) * rpois(1000, 4), 100, 10)
y <- data \%*\% w
w <- runif(10, 0, 7.5)
print(sum(abs(y - data \%*\% w)))
cw <- intcalibrate(w, ~. + 0, y, lower = 1, upper = 7, sparse = TRUE, data = data)
print(sum(abs(y - data \%*\% cw)))
qw <- intcalibrate(w, ~. + 0, y, lower = 1, upper = 7, sparse = TRUE, data = data,
                   penalty = "quadrat", tuning = 0.7)
print(sum(abs(y - data \%*\% qw)))
barplot(table(cw), main = "Calibrated integer weights")
barplot(table(qw), main = "Calibrated integer weights")

}
