#' Generic summary function for bcbnp_0 object in jarbes.
#'
#' @param object The object generated by the bcbnp function.
#'
#' @param digits The number of significant digits printed. The default value is 3.
#' @param show.clustering A logical value indicating whether to show the clustering parameters table. Defaults to `TRUE`.
#' @param show.neff A logical value indicating whether to show the effective sample size column (n.eff) and Rhat. Defaults to `TRUE`.
#'
#' @param ... \dots
#'
#' @export
summary.bcbnp_0 = function(object, digits = 3, show.clustering = FALSE, show.neff = FALSE, ...) {
  bugs.output = object$BUGSoutput
  bugs.summary = bugs.output$summary

  summary.m = list()

  # Model specifications from the fitted object
  model.spec = list()
  model.spec$link = "Normal approximation"

  model.spec$hyper_corrected = list(
    mean.mu.theta = object$prior$mean.mu,
    sd.mu.theta = object$prior$sd.mu,
    scale.sigma.between = object$prior$scale.sigma.between,
    df.scale.between = object$prior$df.scale.between
  )

  model.spec$hyper_bias = list(
    mean.mu.k = object$prior$mean.mu.k,
    sd.mu.k = object$prior$sd.mu.k,
    g.0 = object$prior$g.0,
    g.1 = object$prior$g.1,
    a.0 = object$prior$a.0,
    a.1 = object$prior$a.1,
    alpha.0 = object$prior$alpha.0,
    alpha.1 = object$prior$alpha.1,
    K = object$prior$K
  )

  summary.m$model.specification = model.spec

  # Filter columns based on show.neff before extracting subsets
  if (!show.neff) {
    bugs.summary = bugs.summary[, !(colnames(bugs.summary) %in% c("n.eff", "Rhat")), drop = FALSE]
  }

  # Extract posterior distributions for main model parameters
  theta_new_name = paste0("theta[", object$N, "]")
  param_names = c("mu.theta", theta_new_name, "tau.theta")
  param_names = intersect(param_names, rownames(bugs.summary))
  summary.m$model_corrected = bugs.summary[param_names, , drop = FALSE]

  # The row names should be informative for the user
  row.names(summary.m$model_corrected) = c("Mean", "Predictive Effect", "Tau (between studies sd)")

  # Extract bias model parameters
  bias_param_regex = "pi.B|alpha|K.hat"
  summary.m$model_biased = bugs.summary[grep(bias_param_regex, rownames(bugs.summary)), , drop = FALSE]

  if (show.clustering) {
    # Extract posterior means for clustering parameters
    mu_k_means = bugs.summary[grep("mu.beta\\[.+\\]", rownames(bugs.summary)), "mean"]
    sigma_k_means = bugs.summary[grep("sigma.beta\\[.+\\]", rownames(bugs.summary)), "mean"]
    pi_k_means = bugs.summary[grep("pi\\[.+\\]", rownames(bugs.summary)), "mean"]

    clustering_df = data.frame(
      mu.k = mu_k_means,
      sigma.k = sigma_k_means,
      pi.k = pi_k_means
    )
    num_clusters = length(mu_k_means)
    row.names(clustering_df) = paste0("Cluster ", 1:num_clusters)
    summary.m$clustering_params = clustering_df
  }

  # DIC and MCMC setup information
  summary.m$DIC = bugs.output$DIC
  summary.m$pD = bugs.output$pD
  mcmc.setup = list(
    n.chains = bugs.output$n.chains,
    n.iter = bugs.output$n.iter,
    n.burnin = bugs.output$n.burnin,
    n.thin = object$BUGSoutput$n.thin # Fetch thin rate from BUGS output
  )
  summary.m$mcmc.setup = mcmc.setup

  class(summary.m) = "summary.bcbnp_0"

  # Print the summary object
  print(summary.m, digits, ...)
}

# ----------------------------------------------------------------------

#' @export
print.summary.bcbnp_0 = function(x, digits = 3, ...) {
  cat('Model specifications:\n')
  model.spec = x$model.specification

  cat(paste('  Link function: ', model.spec$link, sep = ''))
  cat('\n')
  cat('\n')

  cat('  Hyper-priors parameters for the Model of Interest (Bias-Corrected):\n')
  hyper_corrected = model.spec$hyper_corrected
  cat(paste('  Prior for mu.theta: Normal', '[', round(hyper_corrected$mean.mu.theta, digits), ', ', round(hyper_corrected$sd.mu.theta^2, digits), ']', sep = ''))
  cat('\n')
  cat(paste('  Prior for 1/tau.theta^2: Scale.Gamma', '[', round(hyper_corrected$scale.sigma.between, digits), ', ', round(hyper_corrected$df.scale.between, digits), ']', sep = ''))
  cat('\n')

  cat('\n')
  cat('  Hyper-priors parameters for the Model of Bias:\n')
  hyper_bias = model.spec$hyper_bias
  cat(paste('  Prior for mu.k: Normal', '[', round(hyper_bias$mean.mu.k, digits), ', ', round(hyper_bias$sd.mu.k^2, digits), ']', sep = ''))
  cat('\n')
  cat(paste('  Prior for tau.k: Gamma', '[', round(hyper_bias$g.0, digits), ', ', round(hyper_bias$g.1, digits), ']', sep = ''))
  cat('\n')
  cat(paste('  Prior for pi.B: Beta', '[', round(hyper_bias$a.0, digits), ', ', round(hyper_bias$a.1, digits), ']', sep = ''))
  cat('\n')
  cat(paste('  Prior for alpha: Uniform', '[', round(hyper_bias$alpha.0, digits), ', ', round(hyper_bias$alpha.1, digits), ']', sep = ''))
  cat('\n')
  cat(paste('  Maximum number of clusters (K): ', hyper_bias$K, sep = ''))
  cat('\n')

  cat('\n')
  cat('Posterior distributions:\n')
  cat('-------------------\n')
  cat('Model of Interest (Bias-Corrected):\n')
  print(round(x$model_corrected, digits))

  # Removed: if ("regression_coefficients" %in% names(x)) {...}

  cat('\n-------------------\n')
  cat('Model of Bias:\n')
  print(round(x$model_biased, digits))

  if ("clustering_params" %in% names(x)) {
    cat('\n-------------------\n')
    cat('Clustering parameters (posterior means):\n')
    print.data.frame(round(x$clustering_params, digits), row.names = TRUE)
  }

  cat('\n-------------------\n')

  mcmc = x$mcmc.setup
  cat(paste('MCMC setup (fit using jags): ', mcmc$n.chains, ' chains, each with ', mcmc$n.iter, ' total iterations (first ', mcmc$n.burnin, ' discarded, thinning rate ', mcmc$n.thin, ')' , sep = ''))
  cat('\n')
  cat(paste('DIC: ', round(x$DIC, digits), sep = ''))
  cat('\n')
  cat(paste('pD: ', round(x$pD, digits), sep = ''))
  cat('\n')
}
