% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gpOptimizationInterface.R
\name{gpCappedL1}
\alias{gpCappedL1}
\title{gpCappedL1}
\usage{
gpCappedL1(
  par,
  fn,
  gr = NULL,
  ...,
  regularized,
  lambdas,
  thetas,
  method = "glmnet",
  control = lessSEM::controlGlmnet()
)
}
\arguments{
\item{par}{labeled vector with starting values}

\item{fn}{R function which takes the parameters AND their labels
as input and returns the fit value (a single value)}

\item{gr}{R function which takes the parameters AND their labels
as input and returns the gradients of the objective function.
If set to NULL, numDeriv will be used to approximate the gradients}

\item{...}{additional arguments passed to fn and gr}

\item{regularized}{vector with names of parameters which are to be regularized.}

\item{lambdas}{numeric vector: values for the tuning parameter lambda}

\item{thetas}{parameters whose absolute value is above this threshold will be penalized with
a constant (theta)}

\item{method}{which optimizer should be used? Currently implemented are ista
and glmnet.}

\item{control}{used to control the optimizer. This element is generated with
the controlIsta and controlGlmnet functions. See ?controlIsta and ?controlGlmnet
for more details.}
}
\value{
Object of class gpRegularized
}
\description{
Implements cappedL1 regularization for general purpose optimization problems.
The penalty function is given by:
\deqn{p( x_j) = \lambda \min(| x_j|, \theta)}
where \eqn{\theta > 0}. The cappedL1 penalty is identical to the lasso for
parameters which are below \eqn{\theta} and identical to a constant for parameters
above \eqn{\theta}. As adding a constant to the fitting function will not change its
minimum, larger parameters can stay unregularized while smaller ones are set to zero.
}
\details{
The interface is similar to that of optim. Users have to supply a vector
with starting values (important: This vector \emph{must} have labels) and a fitting
function. This fitting functions \emph{must} take a labeled vector with parameter
values as first argument. The remaining arguments are passed with the ... argument.
This is similar to optim.

The gradient function gr is optional. If set to NULL, the \pkg{numDeriv} package
will be used to approximate the gradients. Supplying a gradient function
can result in considerable speed improvements.

CappedL1 regularization:
\itemize{
\item Zhang, T. (2010). Analysis of Multi-stage Convex Relaxation for Sparse Regularization.
Journal of Machine Learning Research, 11, 1081–1107.
}

For more details on GLMNET, see:
\itemize{
\item Friedman, J., Hastie, T., & Tibshirani, R. (2010).
Regularization Paths for Generalized Linear Models via Coordinate Descent.
Journal of Statistical Software, 33(1), 1–20. https://doi.org/10.18637/jss.v033.i01
\item Yuan, G.-X., Chang, K.-W., Hsieh, C.-J., & Lin, C.-J. (2010).
A Comparison of Optimization Methods and Software for Large-scale
L1-regularized Linear Classification. Journal of Machine Learning Research, 11, 3183–3234.
\item Yuan, G.-X., Ho, C.-H., & Lin, C.-J. (2012).
An improved GLMNET for l1-regularized logistic regression.
The Journal of Machine Learning Research, 13, 1999–2030. https://doi.org/10.1145/2020408.2020421
}

For more details on ISTA, see:
\itemize{
\item Beck, A., & Teboulle, M. (2009). A Fast Iterative Shrinkage-Thresholding
Algorithm for Linear Inverse Problems. SIAM Journal on Imaging Sciences, 2(1),
183–202. https://doi.org/10.1137/080716542
\item Gong, P., Zhang, C., Lu, Z., Huang, J., & Ye, J. (2013).
A General Iterative Shrinkage and Thresholding Algorithm for Non-convex
Regularized Optimization Problems. Proceedings of the 30th International
Conference on Machine Learning, 28(2)(2), 37–45.
\item Parikh, N., & Boyd, S. (2013). Proximal Algorithms. Foundations and
Trends in Optimization, 1(3), 123–231.
}
}
\examples{
# This example shows how to use the optimizers
# for other objective functions. We will use
# a linear regression as an example. Note that
# this is not a useful application of the optimizers
# as there are specialized packages for linear regression
# (e.g., glmnet)

# This example shows how to use the optimizers
# for other objective functions. We will use
# a linear regression as an example. Note that
# this is not a useful application of the optimizers
# as there are specialized packages for linear regression
# (e.g., glmnet)

library(lessSEM)
set.seed(123)

# first, we simulate data for our
# linear regression.
N <- 100 # number of persons
p <- 10 # number of predictors
X <- matrix(rnorm(N*p),	nrow = N, ncol = p) # design matrix
b <- c(rep(1,4),
       rep(0,6)) # true regression weights
y <- X\%*\%matrix(b,ncol = 1) + rnorm(N,0,.2)

# First, we must construct a fiting function
# which returns a single value. We will use
# the residual sum squared as fitting function.

# Let's start setting up the fitting function:
fittingFunction <- function(par, y, X, N){
  # par is the parameter vector
  # y is the observed dependent variable
  # X is the design matrix
  # N is the sample size
  pred <- X \%*\% matrix(par, ncol = 1) #be explicit here:
  # we need par to be a column vector
  sse <- sum((y - pred)^2)
  # we scale with .5/N to get the same results as glmnet
  return((.5/N)*sse)
}

# let's define the starting values:
b <- c(solve(t(X)\%*\%X)\%*\%t(X)\%*\%y) # we will use the lm estimates
names(b) <- paste0("b", 1:length(b))
# names of regularized parameters
regularized <- paste0("b",1:p)

# optimize
cL1 <- gpCappedL1(
  par = b,
  regularized = regularized,
  fn = fittingFunction,
  lambdas = seq(0,1,.1),
  thetas = c(0.001, .5, 1),
  X = X,
  y = y,
  N = N
)

# optional: plot requires plotly package
# plot(cL1)

# for comparison

fittingFunction <- function(par, y, X, N, lambda, theta){
  pred <- X \%*\% matrix(par, ncol = 1)
  sse <- sum((y - pred)^2)
  smoothAbs <- sqrt(par^2 + 1e-8)
  pen <- lambda * ifelse(smoothAbs < theta, smoothAbs, theta)
  return((.5/N)*sse + sum(pen))
}

round(
  optim(par = b,
      fn = fittingFunction,
      y = y,
      X = X,
      N = N,
      lambda =  cL1@fits$lambda[15],
      theta =  cL1@fits$theta[15],
      method = "BFGS")$par,
  4)
cL1@parameters[15,]
}
