% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/msl.R
\name{msf_sc}
\alias{msf_sc}
\title{Compute MCTQ sleep-corrected local time of mid-sleep on work-free days}
\usage{
msf_sc(msf, sd_w, sd_f, sd_week, alarm_f)
}
\arguments{
\item{msf}{An \code{\link[hms:hms]{hms}} object corresponding to the \strong{local time of
mid-sleep on work-free days} from a standard, micro, or shift version of
the MCTQ questionnaire. You can use \code{\link[=msl]{msl()}} to compute it.}

\item{sd_w}{A \code{\link[lubridate:duration]{Duration}} object corresponding to the
\strong{sleep duration on work days} from a standard, micro, or shift version of
the MCTQ questionnaire. You can use \code{\link[=sdu]{sdu()}} to compute it.}

\item{sd_f}{A \code{\link[lubridate:duration]{Duration}} object corresponding to the
\strong{sleep duration on work-free days} from a standard, micro, or shift
version of the MCTQ questionnaire. You can use \code{\link[=sdu]{sdu()}} to
compute it.}

\item{sd_week}{A \code{\link[lubridate:duration]{Duration}} object corresponding to
the \strong{average weekly sleep duration} from a standard or micro version of
the MCTQ questionnaire (you can use \code{\link[=sd_week]{sd_week()}} to
compute it) \strong{or} the \strong{overall sleep duration of a particular shift}
from a shift version of the MCTQ questionnaire (you can use
\code{\link[=sd_overall]{sd_overall()}} to compute it).}

\item{alarm_f}{A \code{\link[base:logical]{logical}} object corresponding to the
\strong{alarm clock use on work-free days} from a standard, micro, or shift
version of the MCTQ questionnaire. Note that, if \code{alarm_f == TRUE},
\code{msf_sc} cannot be computed, \code{msf_sc()} will return \code{NA} for these cases.
For the \eqn{\mu}MCTQ, this value must be set as \code{FALSE} all times, since
the questionnaire considers only the work-free days when the respondent
does not use an alarm (e.g., \code{alarm_f = rep(FALSE, length(msf))}).}
}
\value{
An \code{\link[hms:hms]{hms}} object corresponding to the MCTQ chronotype or
sleep-corrected local time of mid-sleep on work-free days.
}
\description{
\ifelse{html}{\href{https://lifecycle.r-lib.org/articles/stages.html#maturing}{\figure{lifecycle-maturing.svg}{options: alt='[Maturing]'}}}{\strong{[Maturing]}}

\code{msf_sc()} computes the \strong{sleep-corrected local time of mid-sleep on
work-free days} for standard, micro, and shift versions of the Munich
ChronoType Questionnaire (MCTQ).

When using the shift version of the MCTQ, replace the value of \code{sd_week} to
\code{sd_overall}, as instructed in the Arguments section.
}
\details{
\strong{Standard MCTQ} functions were created following the guidelines in
Roenneberg, Wirz-Justice, & Merrow (2003), Roenneberg, Allebrandt, Merrow, &
Vetter (2012), and from The Worldwide Experimental Platform (theWeP, n.d.).

\strong{\eqn{\mu}MCTQ} functions were created following the guidelines in Ghotbi
et al. (2020), in addition to the guidelines used for the standard MCTQ.

\strong{MCTQ\eqn{^{Shift}}{ Shift}} functions were created following the
guidelines in Juda, Vetter, & Roenneberg (2013), in addition to the
guidelines used for the standard MCTQ.

See the References section to learn more.
\subsection{Class requirements}{

The \code{mctq} package works with a set of object classes specially created to
hold time values. These classes can be found in the
\link[lubridate:lubridate-package]{lubridate} and \link[hms:hms-package]{hms}
packages. Please refer to those package documentations to learn more about
them.
}

\subsection{Rounding and fractional time}{

Some operations may produce an output with fractional time (e.g.,
\code{"19538.3828571429s (~5.43 hours)"}, \code{01:15:44.505}). If you want, you
can round it with \code{\link[=round_time]{round_time()}}.

Our recommendation is to avoid rounding, but, if you do, make sure that you
only round your values after all computations are done. That way you avoid
\href{https://en.wikipedia.org/wiki/Round-off_error}{round-off errors}.
}
}
\section{Guidelines}{


Roenneberg, Allebrandt, Merrow, & Vetter (2012), Ghotbi et al. (2020), Juda,
Vetter, & Roenneberg (2013), and The Worldwide Experimental Platform (n.d.)
guidelines for \code{msf_sc()} (\eqn{MSF_{sc}}{MSF_sc}) computation are as
follows.
\subsection{Notes}{
\itemize{
\item For all cases, \eqn{MSF_{sc}}{MSF_sc} cannot be computed if the participant
wakes up with an alarm clock on work-free days (\eqn{Alarm_F}{Alarm_F}).
\item For MCTQ\eqn{^{Shift}}{ Shift}, the computation below must be applied to
each shift section of the questionnaire.
\item \eqn{MSF_{sc}}{MSF_sc} is a proxy for the subject chronotype in
standard and micro versions of the MCTQ.
\item The basis for estimating chronotype in shift-workers is the mid-sleep on
work-free days after evening shifts (\eqn{MSF^E}{MSF_E}). In case work
schedules do not comprise evening shifts, Juda, Vetter, & Roenneberg (2013)
propose to derive it from the \eqn{MSF_{sc}}{MSF_sc} of other shifts (e.g.,
by using a linear model). Unfortunately, the \code{mctq} package can't help you
with that, as it requires a closer look at your data.
\item \eqn{MSF_{sc}}{MSF_sc} depends on developmental and environmental
conditions (e.g., age, light exposure). For epidemiological and genetic
studies, \eqn{MSF_{sc}}{MSF_sc} must be normalized for age and sex to make
populations of different age and sex compositions comparable (Roenneberg,
Allebrandt, Merrow, & Vetter, 2012).
\item If you are visualizing this documentation in plain text, you may have some
trouble understanding the equations. You can see this documentation on the
package \href{https://docs.ropensci.org/mctq/reference/}{website}.
}
}

\subsection{For standard and micro versions of the MCTQ}{

\deqn{\textrm{If } Alarm_{F} = True \; , \; MSF_{sc} =
\textrm{Not Available (NA)}}
\deqn{\textrm{Else if } SD_F \leq SD_W \; , \; MSF_{sc} = MSF}
\deqn{\textrm{Else } \; , \; MSF_{sc} = MSF - \frac{SD_F - SD_{week}}{2}}

Where:
\itemize{
\item \eqn{MSF_{sc}} = Sleep-corrected local time of mid-sleep on work-free days.
\item \eqn{Alarm_{F}} = A \code{\link[base:logical]{logical}} value indicating if the
respondent uses an alarm clock to wake up on work-free days.
\item \eqn{MSF} = Local time of mid-sleep on work-free days.
\item \eqn{SD_W} = Sleep duration on workdays.
\item \eqn{SD_F} = Sleep duration on work-free days.
\item \eqn{SD_{week}} = Average weekly sleep duration.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days.

Note that, since:

\deqn{MSF = SO_{F} + \frac{SD_{F}}{2}}

Where:
\itemize{
\item \eqn{MSF} = Local time of mid-sleep on work-free days.
\item \eqn{SO_{F}} = Local time of sleep onset on work-free days.
\item \eqn{SD_{F}} = Sleep duration on work-free days.
}

The last condition of the \eqn{MSF_{sc}}{MSF_sc} computation can be
simplified to:

\deqn{MSF_{sc} = SO_{F} + \frac{SD_{F}}{2} -
\frac{SD_{F} - SD_{week}}{2}}
\deqn{MSF_{sc} = SO_{F} + \frac{SD_{F}}{2} - \frac{SD_{F}}{2} +
\frac{SD_{week}}{2}}
\deqn{MSF_{sc} = SO_{F} + \frac{SD_{week}}{2}}
}

\subsection{For the shift version of the MCTQ}{

\deqn{\textrm{If } Alarm_{F}^{M/E/N} = True \; , \; MSF_{sc}^{M/E/N} =
\textrm{Not Available (NA)}}
\deqn{\textrm{Else if } SD_{F}^{M/E/N} \leq SD_{W}^{M/E/N} \; , \;
MSF_{sc}^{M/E/N} = MSF^{M/E/N}}
\deqn{\textrm{Else } \; , \; MSF_{sc}^{M/E/N} = MSF^{M/E/N} -
\frac{SD_{F}^{M/E/N} - \emptyset SD^{M/E/N}}{2}}

Where:
\itemize{
\item \eqn{MSF_{sc}^{M/E/N}} = Sleep-corrected local time of mid-sleep between
two free days after a particular shift.
\item \eqn{Alarm_{F}^{M/E/N}} = A \code{\link[base:logical]{logical}} value indicating
if the respondent uses an alarm clock to wake up between two free days after
a particular shift.
\item \eqn{MSF^{M/E/N}}{MSF_M/E/N} = Local time of mid-sleep between two free
days after a particular shift.
\item \eqn{SD_{W}^{M/E/N}}{SD_W_M/E/N} = Sleep duration between two days in a
particular shift.
\item \eqn{SD_{F}^{M/E/N}}{SD_F_M/E/N} = Sleep duration between two free days
after a particular shift.
\item \eqn{\emptyset SD^{M/E/N}}{OSD_M/E/N} = Overall sleep duration of a
particular shift.
}

\strong{*} \eqn{W} = Workdays; \eqn{F} = Work-free days, \eqn{M} =
Morning shift; \eqn{E} = Evening shift; \eqn{N} = Night shift.

Note that, since:

\deqn{MSF^{M/E/N} = SO_{F}^{M/E/N} + \frac{SD_{F}^{M/E/N}}{2}}

Where:
\itemize{
\item \eqn{MSF^{M/E/N}}{MSF_M/E/N} = Local time of mid-sleep between two free
days after a particular shift.
\item \eqn{SO_{F}^{M/E/N}} = Local time of sleep onset between two free days
after a particular shift.
\item \eqn{SD_{F}^{M/E/N}} = Sleep duration between two free days after a
particular shift.
}

The last condition of the \eqn{MSF_{sc}^{M/E/N}}{MSF_sc_M/E/N} computation
can be simplified to:

\deqn{MSF_{sc}^{M/E/N} = SO_{F}^{M/E/N} + \frac{SD_{F}^{M/E/N}}{2} -
\frac{SD_{F}^{M/E/N} - \emptyset SD^{M/E/N}}{2}}
\deqn{MSF_{sc}^{M/E/N} = SO_{F}^{M/E/N} + \frac{SD_{F}^{M/E/N}}{2} -
\frac{SD_{F}^{M/E/N}}{2} + \frac{\emptyset SD^{M/E/N}}{2}}
\deqn{MSF_{sc}^{M/E/N} = SO_{F}^{M/E/N} + \frac{\emptyset SD^{M/E/N}}{2}}
}
}

\examples{
## Scalar example

msf <- hms::parse_hms("04:00:00")
sd_w <- lubridate::dhours(6)
sd_f <- lubridate::dhours(7)
sd_week <- lubridate::dhours(6.29)
alarm_f <- FALSE
msf_sc(msf, sd_w, sd_f, sd_week, alarm_f)
#> 03:38:42 # Expected

msf <- hms::parse_hm("01:00:00")
sd_w <- lubridate::dhours(5.5)
sd_f <- lubridate::dhours(9)
sd_week <- lubridate::dhours(6.75)
alarm_f <- FALSE
msf_sc(msf, sd_w, sd_f, sd_week, alarm_f)
#> 23:52:30 # Expected

msf <- hms::parse_hms("05:40:00")
sd_w <- lubridate::dhours(7.5)
sd_f <- lubridate::dhours(10)
sd_week <- lubridate::dhours(8.5)
alarm_f <- TRUE
msf_sc(msf, sd_w, sd_f, sd_week, alarm_f)
#> NA # Expected (`msf_sc` cannot be computed if `alarm_f == TRUE`)

## Vector example

msf <- c(hms::parse_hms("03:45:00"), hms::parse_hm("04:45:00"))
sd_w <- c(lubridate::dhours(9), lubridate::dhours(6.45))
sd_f <- c(lubridate::dhours(5), lubridate::dhours(10))
sd_week <- c(lubridate::dhours(8.5), lubridate::dhours(9.2))
alarm_f <- c(FALSE, FALSE)
msf_sc(msf, sd_w, sd_f, sd_week, alarm_f)
#> 03:45:00 # Expected
#> 04:21:00 # Expected

## Rounding the output at the seconds level

msf <- hms::parse_hms("05:40:00")
sd_w <- lubridate::dhours(5.43678)
sd_f <- lubridate::dhours(9.345111)
sd_week <- lubridate::dhours(7.5453)
alarm_f <- FALSE
msf_sc(msf, sd_w, sd_f, sd_week, alarm_f)
#> 04:46:00.3402 # Expected

round_time(msf_sc(msf, sd_w, sd_f, sd_week, alarm_f))
#> 04:46:00 # Expected
}
\references{
Ghotbi, N., Pilz, L. K., Winnebeck, E. C., Vetter, C., Zerbini, G., Lenssen,
D., Frighetto, G., Salamanca, M., Costa, R., Montagnese, S., & Roenneberg, T.
(2020). The \eqn{\mu}MCTQ: an ultra-short version of the Munich ChronoType
Questionnaire. \emph{Journal of Biological Rhythms}, \emph{35}(1), 98-110.
\doi{10.1177/0748730419886986}

Juda, M., Vetter, C., & Roenneberg, T. (2013). The Munich ChronoType
Questionnaire for shift-workers (MCTQ\eqn{^{Shift}}{ Shift}). \emph{Journal of
Biological Rhythms}, \emph{28}(2), 130-140. \doi{10.1177/0748730412475041}

Roenneberg T., Allebrandt K. V., Merrow M., & Vetter C. (2012). Social jetlag
and obesity. \emph{Current Biology}, \emph{22}(10), 939-43.
\doi{10.1016/j.cub.2012.03.038}

Roenneberg, T., Wirz-Justice, A., & Merrow, M. (2003). Life between clocks:
daily temporal patterns of human chronotypes. \emph{Journal of Biological
Rhythms}, \emph{18}(1), 80-90. \doi{10.1177/0748730402239679}

The Worldwide Experimental Platform (n.d.). MCTQ.
\url{https://www.thewep.org/documentations/mctq/}
}
\seealso{
Other MCTQ functions: 
\code{\link{fd}()},
\code{\link{gu}()},
\code{\link{le_week}()},
\code{\link{msl}()},
\code{\link{napd}()},
\code{\link{sd24}()},
\code{\link{sd_overall}()},
\code{\link{sd_week}()},
\code{\link{sdu}()},
\code{\link{sjl_sc}()},
\code{\link{sjl_weighted}()},
\code{\link{sjl}()},
\code{\link{so}()},
\code{\link{tbt}()}
}
\concept{MCTQ functions}
