% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/CLONALG.Algorithm.R
\name{CLONALG}
\alias{CLONALG}
\title{Optimization using Clonal Selection Algorithm}
\usage{
CLONALG(FUN, optimType = "MIN", numVar, numPopulation = 40,
  maxIter = 500, rangeVar, selectionSize = as.integer(numPopulation/4),
  multipicationFactor = 0.5, hypermutationRate = 0.1)
}
\arguments{
\item{FUN}{an objective function or cost function,}

\item{optimType}{a string value that represent the type of optimization.
There are two option for this arguments: \code{"MIN"} and \code{"MAX"}.
The default value is \code{"MIN"}, which the function will do minimization.
Otherwise, you can use \code{"MAX"} for maximization problem.
The default value is \code{"MIN"}.}

\item{numVar}{a positive integer to determine the number variables.}

\item{numPopulation}{a positive integer to determine the number populations. The default value is 40.}

\item{maxIter}{a positive integer to determine the maximum number of iterations. The default value is 500.}

\item{rangeVar}{a matrix (\eqn{2 \times n}) containing the range of variables,
where \eqn{n} is the number of variables, and first and second rows
are the lower bound (minimum) and upper bound (maximum) values, respectively.
If all variable have equal upper bound, you can define \code{rangeVar} as
matrix (\eqn{2 \times 1}).}

\item{selectionSize}{a positive integer between 0 and numVar
to determine selection size (see details). The default value is \code{as.integer(numPopulation/4)}.}

\item{multipicationFactor}{a positive numeric between 0 and 1 to determine number of clones. The default value is 0.5.}

\item{hypermutationRate}{a positive numeric between 0 and 1 to determine probabilty of variable in
clone candidate solutions to be mutated, close to 1 probability is high and vice versa.
The default value is 0.1.}
}
\value{
\code{Vector [v1, v2, ..., vn]} where \code{n} is number variable
        and \code{vn} is value of \code{n-th} variable.
}
\description{
This is the internal function that implements Clonal Selection
Algorithm. It is used to solve continuous optimization tasks.
Users do not need to call it directly,
but just use \code{\link{metaOpt}}.
}
\details{
This algorithm was proposed by (Castro & Zuben, 2002). The Clonal Selection Algorithm (CLONALG)
mimics maturation proses of imumune system. CLONALG consist 5 step initialize, selection, clonal,
hypermutation and maturation.

In order to find the optimal solution, the algorithm follow the following steps.
\itemize{
\item initialize population randomly.
\item select top selectionSize candidate solutions from population with best fitness.
\item clone each selected candidate solutions.
\item hypermutation each variable in cloned candidate solutions.
\item maturation combine each hypermutated candidate solution with population.
      Select top n candidate solution from population as new population.
\item If a termination criterion (a maximum number of iterations or a sufficiently good fitness) is met,
      exit the loop.
}
}
\examples{
##################################
## Optimizing the quartic with noise function

# define Quartic with noise function as objective function
quartic <- function(x){
    dim <- length(x)
    result <- sum(c(1:dim)*(x^4))+runif(1)
    return(result)
}

## Define parameter
numVar <- 5
rangeVar <- matrix(c(-1.28, 1.28), nrow=2)

## calculate the optimum solution clonal selection algorithm
resultCLONALG <- CLONALG(quartic, optimType="MIN", numVar, numPopulation=20,
                 maxIter=100, rangeVar)

## calculate the optimum value using quartic with noise function
optimum.value <- quartic(resultCLONALG)

}
\references{
Castro, L. & Zuben, F. J. V. (2002).
Learning and optimization using the clonal selection principle.
IEEE Transactions on Evolutionary Computation, Special Issue on
Artificial. Immune Systems, 6(3), 239–251. https://doi.org/10.1109/TEVC.2002.1011539
}
\seealso{
\code{\link{metaOpt}}
}
