% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pool.R
\name{pool}
\alias{pool}
\title{Pool results from models fitted on multiply imputed datasets}
\usage{
pool(fits, dfcom = NULL)
}
\arguments{
\item{fits}{a list of fitted model objects of length \emph{m} >= 2.
Each model must support \code{coef()} and \code{vcov()} methods.
All models must have the same number of coefficients.}

\item{dfcom}{a positive integer or \code{Inf}. The complete-data degrees of freedom.
If \code{NULL} (default), it is extracted from the fitted models via
\code{\link[stats]{df.residual}}. Set to \code{Inf} to skip the Barnard-Rubin
small-sample correction.}
}
\value{
A data.frame with one row per parameter and columns:
  \describe{
    \item{term}{Coefficient name.}
    \item{m}{Number of imputations.}
    \item{estimate}{Pooled estimate (average across \emph{m} models).}
    \item{std.error}{Pooled standard error (\code{sqrt(t)}).}
    \item{statistic}{t-statistic (\code{estimate / std.error}).}
    \item{p.value}{Two-sided p-value from a t-distribution with \code{df} degrees of freedom.}
    \item{df}{Degrees of freedom (Barnard-Rubin adjusted).}
    \item{riv}{Relative increase in variance due to nonresponse: \code{(1 + 1/m) * b / ubar}.}
    \item{lambda}{Proportion of total variance attributable to missingness: \code{(1 + 1/m) * b / t}.}
    \item{fmi}{Fraction of missing information.}
    \item{ubar}{Within-imputation variance (average of the \emph{m} variance estimates).}
    \item{b}{Between-imputation variance (variance of the \emph{m} point estimates).}
    \item{t}{Total variance: \code{ubar + (1 + 1/m) * b}.}
    \item{dfcom}{Complete-data degrees of freedom used.}
    \item{conf.low}{Lower bound of the 95\% confidence interval.}
    \item{conf.high}{Upper bound of the 95\% confidence interval.}
  }
}
\description{
Combines parameter estimates and standard errors from models fitted on
\emph{m} multiply imputed datasets using Rubin's rules (Rubin, 1987).
Degrees of freedom are adjusted using the Barnard-Rubin (1999) small-sample correction.

This function works with any fitted model that supports \code{\link[stats]{coef}} and
\code{\link[stats]{vcov}} methods (e.g., \code{\link[stats]{lm}}, \code{\link[stats]{glm}},
\code{survival::coxph}, etc.).

Results are validated against \code{\link[mice]{pool}} from the \pkg{mice} package
for \code{lm}, \code{glm} (logistic and Poisson), weighted regression, interactions,
and varying numbers of imputations.
}
\examples{
library(miceFast)
set.seed(123)
data(air_miss)

# Step 1: Generate m = 5 completed datasets using fill_NA with a stochastic model
completed <- lapply(1:5, function(i) {
  dat <- air_miss
  dat$Ozone <- fill_NA(
    x = dat,
    model = "lm_bayes",
    posit_y = "Ozone",
    posit_x = c("Solar.R", "Wind", "Temp")
  )
  dat
})

# Step 2: Fit a model on each completed dataset
fits <- lapply(completed, function(d) {
  lm(Ozone ~ Solar.R + Wind + Temp, data = d)
})

# Step 3: Pool using Rubin's rules
pool(fits)

}
\references{
Rubin, D.B. (1987). \emph{Multiple Imputation for Nonresponse in Surveys}.
John Wiley & Sons.

Barnard, J. and Rubin, D.B. (1999). Small-sample degrees of freedom with
multiple imputation. \emph{Biometrika}, 86(4), 948-955.
}
\seealso{
\code{\link{fill_NA}} \code{\link{fill_NA_N}}
}
