#' Simulate coefficient matrices with specified density
#'
#' Simulates coefficient matrices used to generate data from a vector autoregressive process.
#'
#' Coefficient values are drawn from a Uniform(coefmin, coefmax) or a Uniform(-coefmax, -coefmin) each with 50\% probability.
#'
#' @param k Integer. Dimension of process.
#' @param coefmin Numeric. Minimum value of coefficient. See Details.
#' @param coefmax Numeric. Maximum value of coefficient. See Details.
#' @param dens Numeric. Must be between 0 and 1. Specifies the proportion of non-zero entries in the coefficient matrix. The number of non-zero entries is computed as \code{floor(k^2*dens)}.
#'
#' @returns \code{k x k} matrix.
#' @examples
#' # bivariate coefficient matrix
#' coefmat <- gen_coef_mat(k = 2, coefmin = 0.1, coefmax = 0.3, dens = 0.8)
#' print(coefmat)
#' @export
gen_coef_mat <- function(k, coefmin, coefmax, dens){
  A <- matrix(0, nrow = k, ncol = k)
  nsamp <- floor(k^2*dens)
  A[sample(1:k^2, size = nsamp)] <- ifelse(stats::runif(nsamp) <= 0.5, stats::runif(nsamp, coefmin, coefmax), stats::runif(nsamp, -coefmax, -coefmin))
  return(A)
}

#' Verify stability of a vector autoregressive model
#'
#' Stability is verified using the method the method on pages 14-17 of \insertCite{lutkepohl2005new}{micvar}. Specifically we generate the coefficient matrix for the VAR(1) representation of the process and check that all eigenvalues have modulus less than 1.
#'
#' @param A List of coefficient matrices.
#'
#' @returns None. Throws error if not stable process.

#' @references
#' \insertRef{lutkepohl2005new}{micvar}
#'
#' @examples
#' VAR3_2_A <- list(gen_coef_mat(3, 0.1, 0.3, 0.8), # lag 1
#'                  gen_coef_mat(3, 0.1, 0.4 , 0.5)) # lag 2
#' verify_stability(VAR3_2_A)
#' @export
verify_stability <- function(A){
  p <- length(A)
  k <- nrow(A[[1]])
  if(p > 1){
    boldA_top <- do.call(cbind, A)
    Ipm1 <- diag(1,nrow = (p-1)*(k))
    extraZeros <- matrix(0,nrow = nrow(Ipm1), ncol=k)
    boldA_bottom <- cbind(Ipm1, extraZeros)
    boldA <- rbind(boldA_top, boldA_bottom)
  } else{
    boldA <- A[[1]]
  }
  evals <- eigen(boldA)$values
  stable <- all(abs(evals)<1)
  if(!stable){
    stop("VAR process is not stable, some roots in or on the unit circle")
  }
  return(NULL)
}

#' Simulate data from a vector autoregressive model with specified coefficient matrices
#'
#' Simulates data from a stable vector autoregressive model with Gaussian innovations and specified coefficient matrices. Stability of the process is verified using \link{verify_stability}.
#'
#' @param A List of coefficient matrices. Each element in A must be a square matrix. Dimension of matrix determines the number of variables. Length of A determines the order of the process. In the case of univariate time series each entry of A should be a \code{1 x 1} matrix.
#' @param n Integer. Number of data points to simulate.
#' @param mu Vector (default 0s). Means of Gaussian innovations.
#' @param Sigma Square matrix (default Identity). Variance of Gaussian innovations.
#' @param burn_in Integer (default 500). Number of observations used to start up simulated process. In total \code{n + burn_in} observations are simulated but the first \code{burn_in} are discarded.
#'
#' @returns \code{n x k} data matrix.
#'
#' @examples
#' # multivariate
#' VAR3_2_A <- list(gen_coef_mat(3, 0.1, 0.3, 0.8), # lag 1
#'                  gen_coef_mat(3, 0.1, 0.4 , 0.5)) # lag 2
#' x <- sim_var(VAR3_2_A, n = 1000)
#'
#' # univariate
#' AR2 <- list(matrix(0.5), matrix(0.2))
#' x <- sim_var(AR2, n = 1000)
#'
#' # non-identity covariance of Gaussian innovations
#' Sigma <- matrix(c(1,0.5,0.9,0.5,1.5,0.7,0.9,0.7,1.25), nrow = 3)
#' x <- sim_var(VAR3_2_A, n = 1000, Sigma = Sigma)
#'
#' @importFrom Rdpack reprompt
#'
#' @export
sim_var <- function(A, n, mu = NULL, Sigma = NULL, burn_in = 500){
  verify_stability(A)
  order <- length(A)
  if(is.matrix(A[[1]])){ # VAR process
    k <- nrow(A[[1]])
    stopifnot(k == nrow(Sigma))
  } else { # AR process
    k <- 1
  }
  if(is.null(mu)){
    mu = rep(0,k)
  }
  if(!is.null(Sigma)){errors <- MASS::mvrnorm(n = n + burn_in + order, mu = rep(0,k), Sigma = Sigma)}
  else{errors <- MASS::mvrnorm(n = n + burn_in + order, mu = rep(0,k), Sigma = diag(1,k))}

  x <- matrix(0, nrow = n + burn_in + order, ncol = k)

  Amat <- do.call(cbind, A)
  for(i in (order+1):(n + burn_in + order)){
    # Initial vals are set to 0 as in lutkepohl 2005, pg 708
    x[i,] <- Amat %*% matrixcalc::vec(t(x[(i-1):(i-order),])) + errors[i,]
  }
  xres <- x[(burn_in + order + 1):(n + burn_in + order),]

  return(xres)
}

#' Estimate order by mean square information criteria (MIC)
#'
#' Fits an autoregressive model to the data where the order is selected by minimizing the mean square information criteria. Model fitting is performed using \link[stats]{ar}.
#' Any of the methods available in the \code{method} argument of \link[stats]{ar} can be used.
#'
#' This function uses the \link[stats]{ar} functions for fitting. For relevant details of those methods see the Details section of \link[stats]{ar}.
#'
#' @param x \code{n x p} time series data matrix. Can be univariate or multivariate time series. If x is not a matrix it will be coerced using \code{as.matrix(x)}.
#' @param pmax Integer. Maxmium number of lags to consider. Considered lags will to be \code{0},\code{1},...,\code{pmax}.
#' @param pmaxst Integer (default is \code{2pmax}). Maximum lag used for computing self-tuned lambda. Must be larger than \code{pmax}.
#' @param method Character string (default is "ols"). Specifies method to fit the model. Options are: \code{c("ols", "burg", "mle", "yule-walker", "yw")}. Note this function uses \link[stats]{ar} to perform model fitting.
#' @param na.action Function for missing values (default is \code{na.fail}). See the \code{na.action} argument in \link[stats]{ar}.
#' @param series Character string. Name of series. See the \code{series} argument in \link[stats]{ar}.
#' @param demean Boolean (default is TRUE). Whether or not to demean the series. See the \code{demean} argument in \link[stats]{ar}.
#' @param ... Additional arguments for specific method. See \link[stats]{ar} and its various methods such as \link[stats]{ar.yw} and \link[stats]{ar.ols} and their corresponding arguments.
#'
#'
#' @returns List with elements. Many of these elements are similar to \link[stats]{ar}.
#' \item{order}{Order of fitted model selected by MIC}
#' \item{penalized_losses}{Numeric vector of penalized losses for orders \code{0},\code{1},...,\code{pmax}.}
#' \item{ar}{Estimated autoregression coefficients. See the \code{ar} return value from \link[stats]{ar}.}
#' \item{var.pred}{Prediction variance. See the \code{var.pred} return value from \link[stats]{ar}.}
#' \item{x.mean}{Estimated mean. See the \code{x.mean} return value from \link[stats]{ar}.}
#' \item{x.intercept}{Intercept. See the \code{x.intercept} return value from \link[stats]{ar}.}
#' \item{n.used}{Number of observations in the time series including missing. See the \code{n.used} return value from \link[stats]{ar}.}
#' \item{n.obs}{Number of non-missing observations. See the \code{n.obs} return value from \link[stats]{ar}.}
#' \item{pmax}{The value of \code{pmax} argument.}
#' \item{partialacf}{Estimate of partial autocorrelation. See the \code{partialacf} return value from \link[stats]{ar}.}
#' \item{resid}{Residuals from fitted model. See the \code{resid} return value from \link[stats]{ar}.}
#' \item{method}{Value of \code{method} argument.}
#' \item{series}{Name of the series. See the \code{series} return value from \link[stats]{ar}.}
#' \item{call}{Function call.}
#' \item{asy.var.coef}{Asymptotic-theory variance matrix of coefficient estimates. See the \code{asy.var.coef} return value from \link[stats]{ar}.}
#'
#' @examples
#' # multivariate example - default is OLS
#' VAR3_2_A <- list(gen_coef_mat(3, 0.1, 0.3, 0.8), # lag 1
#'                  gen_coef_mat(3, 0.1, 0.4 , 0.5)) # lag 2
#' x <- sim_var(VAR3_2_A, n = 5000)
#' mic_model <- micvar(x, pmax = 10)
#'
#' # burg and yule-walker examples
#' mic_model_burg <- micvar(x, pmax = 10, method = "burg")
#' mic_model_yw <- micvar(x, pmax = 10, method = "yw")
#'
#' # univariate example
#' ar_coefs <- list(matrix(0.3,nrow=1), matrix(0.1,nrow=1))
#' x <- sim_var(ar_coefs, n = 5000)
#' mic_model <- micvar(x, pmax = 10)
#' @export
micvar <- function (x, pmax, pmaxst=2*pmax, method = "ols",
                    na.action = stats::na.fail, series = deparse1(substitute(x)),
                    demean = TRUE,
                    ...)
{
  force(pmaxst)

  if (length(pmax) > 1){stop("pmax must be an integer")}
  if (pmax < 1) {stop("pmax must be a positive integer >= 1")}
  if (length(pmaxst) > 1){stop("pmaxst must be an integer")}
  if (pmaxst < pmax){stop("pmaxst must be larger than pmax")}

  # MLE expects a vector, errors if matrix with one column
  if (method != "mle"){
    x <- as.matrix(x)
  }
  if(is.matrix(x)){
    n <- nrow(x)
    k <- ncol(x)
  } else if(is.numeric(x)) {
    n <- length(x)
    k <- 1
  } else{
    stop("")
  }

  ar_method <- switch(match.arg(method, choices = c("ols", "burg", "mle", "yule-walker", "yw")),
                   yw = stats::ar.yw, `yule-walker` = stats::ar.yw, burg = stats::ar.burg, ols = stats::ar.ols, mle = stats::ar.mle)


  # get error matrices and losses (trace of error mats) for each fitted order
  err_mats <- lapply(c(1:pmax, pmaxst), function(p){
    as.matrix(ar_method(x, aic = FALSE, order.max = p, na.action = na.action, series = series, demean = demean, ...)[["var.pred"]])
  })

  # do order 0 separately.
  if (demean){
    xcenter <- scale(x, center = TRUE, scale = FALSE)
    err_mat_order0 <- (crossprod(x) / nrow(x))
  } else{
    err_mat_order0 <- (crossprod(x) / nrow(x))
  }

  # error handling to ensure err_mats are all matrices
  is_mat <- sapply(err_mats, is.matrix)
  if(!all(is_mat)){stop("Detected non-matrix entry for error mat. All entries should be matrices")}

  dimnames(err_mat_order0) <- dimnames(err_mats[[1]])
  err_mats <- stats::setNames(c(list(err_mat_order0), err_mats), c(0,1:pmax, pmaxst))

  # compute losses
  losses <- sapply(err_mats, function(mat) sum(diag(mat)))

  # compute MD and self-tuning lambda
  md <- (losses[length(losses)-1] - losses[length(losses)])/(pmaxst - pmax)
  md_scale <- sqrt(n/(k^2*(log(n))))
  selftune_lam <- abs(md)*sqrt(n/(k^2*(log(n))))

  # select optimal order based on penalized loss
  orders <- c(0,1:pmax)
  pen_losses <- losses[seq_along(orders)] + orders*selftune_lam

  mic_order <- orders[which.min(pen_losses)] # in case of ties which.min selects first (in this case the smaller order)

  # return model
  if(mic_order != 0){
    mic_model <- ar_method(x, aic = FALSE, order.max = mic_order, na.action = na.action, series = series, ...)
    mic_model$aic <- NULL
    mic_model$order.max <- NULL
    mic_model$pmax <- pmax
    mic_model$call <- match.call()
    mic_model$penalized_losses <- pen_losses
  } else{
    mic_model <- list(order = mic_order, var.pred = err_mats[[1]])
    mic_model$penalized_losses <- pen_losses
  }

  return(mic_model)
}


