#' Default S3 plot method for process objects (derived from `process()`)
#'
#' This function sets up the default plotting method
#' for outputs from process function
#'
#' @param x process object as generated by process
#' @param plot_type defaults to both plots. Can specify 'mass' or
#' 'rate' curves by themselves.
#' @param cex size of plots features
#' @param ... other options passed to plot
#' @return plot
#' @method plot process
#' @importFrom graphics legend lines par plot axis
#'
#' @export

plot.process <- function (x, plot_type = NULL, cex = 1, ...) {

  df <- x$data

  p_massloss <- function (data) {
    plot(data$temp_C, data$mass_T, xlab = 'Temperature (C)',
         ylim = c(0, max(data$mass_T) + 0.5),
         yaxt = 'n',
         xaxt = 'n',
         ylab = 'Mass remaining (mg)', pch = 20, cex = 0.7*cex,
         cex.lab = 1.2*cex)
    axis(side = 1, at = c(100, 300, 500, 700), cex.axis = cex,
         labels = c(100, 300, 500, 700))
    top_y <- round(max(data$mass_T), 0)
    mid_y <- top_y/2
    axis(side = 2, at = c(0, mid_y, top_y), cex.axis = cex,
         labels = c(0, mid_y, top_y))
  }

  p_dtg <- function (data) {
    plot(data$temp_C, data$deriv,
         xlab = 'Temperature (C)',
         ylim = c(0, max(data$deriv) + .0003),
         ylab = expression(paste('Rate of mass loss (-dm/dT) (C'^'-1', ')')),
         pch = 20, cex = 0.7*cex,
         cex.lab = 1.2*cex,
         yaxt = 'n', xaxt = 'n')
    axis(side = 1, at = c(100, 300, 500, 700),
         cex.axis = cex,
         labels = c(100, 300, 500, 700))
    top_y <- round(max(data$deriv), 3)
    mid_y <- top_y/2
    axis(side = 2, at = c(0, mid_y, top_y), cex.axis = cex,
         labels = c(0, mid_y, top_y))
  }

  if (!is.null(plot_type) & isTRUE(plot_type == 'mass')) {
    par(mar = c(5, 5, 1, 1))
    p_massloss(df)
  }
  if (!is.null(plot_type) & isTRUE(plot_type == 'rate')) {
    par(mar = c(5, 5, 1, 1))
    p_dtg(df)
  }
  if (is.null(plot_type)) {
    par(mfrow = c(1,2), mar = c(5, 5, 1, 1))
    p_massloss(df)
    p_dtg(df)
  }

  par(mfrow = c(1,1))
}
