% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/makeTinvDF.R
\name{makeTinv}
\alias{makeTinv}
\alias{makeT}
\alias{makeT.default}
\alias{makeT.numPed}
\alias{makeTinv.default}
\alias{makeTinv.numPed}
\alias{makeDiiF}
\alias{makeDiiF.default}
\alias{makeDiiF.numPed}
\title{Creates components of the additive genetic relationship matrix and its inverse}
\usage{
makeTinv(pedigree, ...)

\method{makeTinv}{default}(pedigree, ...)

\method{makeTinv}{numPed}(pedigree, ...)

\method{makeT}{default}(pedigree, genCol = NULL, ...)

\method{makeDiiF}{default}(pedigree, f = NULL, ...)

\method{makeDiiF}{numPed}(pedigree, f = NULL, ...)
}
\arguments{
\item{pedigree}{A pedigree where the columns are ordered ID, Dam, Sire}

\item{\dots}{Arguments to be passed to methods}

\item{genCol}{An integer value indicating the generation up to which the
\code{T} matrix is to be created (corresponding to columns of the lower
triangle \code{T} matrix). The first generation is numbered 0, default is
all generations.}

\item{f}{A numeric vector indicating the level of inbreeding. See Details}
}
\value{
a \code{list}:
  \describe{
    \item{Tinv }{the inverse of the Cholesky decomposition of the additive
      genetic relationship matrix (Ainv=Tinv' Dinv Tinv) in sparse matrix form}
    \item{D }{the diagonal D matrix of the A=TDT' Cholesky decomposition.
      Contains the variance of Mendelian sampling. Matches
      the order of the first/ID column of the pedigree.} 
    \item{f }{the individual coefficients of inbreeding for each individual 
      in the pedigree (matches the order of the first/ID column of the
      pedigree).}
  }
}
\description{
This returns the Cholesky decomposition of the numerator relationship matrix
and its inverse. It can also be used to obtain coefficients of inbreeding for
the pedigreed population.
}
\details{
Missing parents (e.g., base population) should be denoted by either 'NA',
'0', or '*'.

The function implements an adaptation of the Meuwissen and Luo (1992)
algorithm (particularly, following the description of the algorithm in
Mrode 2005) with some code borrowed from the \code{inverseA} function by
Jarrod Hadfield in the \code{MCMCglmm} package. 

The inbreeding level of individuals can be provided instead of calculated.
\code{f} must be a vector that is the same length as individuals in the
pedigree. Supplied coefficients of inbreeding are used instead of being 
calculated until a \code{NA} is encountered in the vector. From this position
on, then coefficients of inbreeding are calculated and replace entries in 
\code{f}. This can be used, for example, to calculate coefficients of
inbreeding for later generations when coefficients of inbreeding in the
previous generations have already been calculated. To specify an average
coefficient of inbreeding for the base population, modify the pedigree to
include a single phantom parent and specify this individual's non-zero
coefficient of inbreeding in \code{f} with the rest of the terms as NA.
}
\examples{

 Tinv <- makeTinv(Mrode2)
 # Method for a numeric pedigree (of `nadiv` class "numPed")
 nPed <- numPed(Mrode2)
 Tinv2 <- makeTinv(nPed)

 ########
 DF <- makeDiiF(Mrode2)
 # manually construct the inverse of the relatedness matrix `Ainv`
 Dinv <- DF$D  #<-- not the inverse yet, just copying the object
 Dinv@x <- 1 / DF$D@x  #<-- inverse of a diagonal matrix
 handAinv <- crossprod(Tinv, Dinv) \%*\% Tinv
   # make the A-inverse directly
   Ainv <- makeAinv(Mrode2)$Ainv
   # Compare
   handAinv
   Ainv
   stopifnot(all(abs((Ainv - handAinv)@x) < 1e-6))

 # supply previous generation coefficients of inbreeding (f)
 ## to keep from re-calculating their f when analyzing subsequent generations
 DF <- makeDiiF(Mrode2[, 1:3])
 Mrode2$gen <- genAssign(Mrode2)
 Mrode2$f_full <- DF$f
 Mrode2$f_in <- with(Mrode2, c(f_full[gen <= 1], rep(NA, sum(gen > 1))))
 DF2 <- makeDiiF(Mrode2[, 1:3], f = Mrode2$f_in) 
 stopifnot(identical(DF, DF2))

}
\references{
Meuwissen, T.H.E & Luo, Z. 1992. Computing inbreeding 
coefficients in large populations. Genetics, Selection, Evolution. 24:305-313.

Mrode, R.A. 2005. Linear Models for the Prediction of Animal Breeding
Values, 2nd ed.  Cambridge, MA: CABI Publishing.
}
\seealso{
\code{\link{makeAinv}}, \code{\link{makeA}}
}
\author{
\email{matthewwolak@gmail.com}
}
